/*	***************************************************************************

	PROJECT:	BasicValueStack
	
	FILE:		BasicValueStack.h
	
	PURPOSE:	The stack on which parameters and variables are kept.
				You have to subclass this for every platform if you need to
				pass along platform-specific data with system messages you
				send.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		2000-11-07	UK		Renamed from ValueStack.
		1999-12-01	UK		Created.
				
	************************************************************************ */

#ifndef	BASICVALUESTACK_H
#define BASICVALUESTACK_H

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkVarValue.h"
#include	<vector>
#include	<iostream>


#pragma mark [Class Declaration]

/* --------------------------------------------------------------------------------
	Class declaration:
   ----------------------------------------------------------------------------- */

class	HyperTalk;	// Forward.

class	BasicValueStack : public vector<TalkVarValue*>
{
protected:
	size_t			mBasePointer;		// This stack's basepointer. The stackpointer is this stack's size.
	long			mErrorLine;			// Current line. Used when error occurs to report where it was.
	HyperTalk*		mErrorScript;		// Current script. Used when error occurs to report where it was.
	TalkVarValue*	mExceptionValue;	// If an exception was thrown, this is the value it threw.

public:
			BasicValueStack() : vector<TalkVarValue*>(), mBasePointer(0)	{};
	virtual ~BasicValueStack()												{};
	
	void			MoveBasePointer( size_t byVal )	{ mBasePointer += byVal; };
	size_t			GetBasePointer()				{ return mBasePointer; };
	void			SetBasePointer( size_t val )	{ mBasePointer = val; };
	
	void			SetExceptionValue( TalkVarValue& inVal )		{ mExceptionValue = new TalkVarValue(0L); inVal.CopyValueTo( *mExceptionValue, true, false ); };
	TalkVarValue*	GetExceptionValue()								{ return mExceptionValue; };
	void			ClearExceptionValue( bool doDispose = true )	{ if( doDispose && mExceptionValue ) delete mExceptionValue; mExceptionValue = NULL; };
	
	void	DumpOneValue( TalkVarValue* theValue )
	{
		static TextMunger		spaces("\t");
		TalkVarValue*			currVal;
		
		spaces.SetOffset(1);
		spaces.Insert<char>('\t');
		
		if( theValue->mList.size() != 0 && theValue->GetNativeType() != VALUE_TYPE_LIST )
			std::cout << "-array not cleared before changing value type-" << endl;
		
		if( theValue->GetNativeType() == VALUE_TYPE_LIST )	// Array.
		{
			TalkValueList::iterator		currElem;
			
			std::cout << spaces.String() << "array (" << (long) theValue << ")" << endl;
			std::cout << spaces.String() << "{" << endl;
			for( currElem = theValue->mList.begin(); currElem != theValue->mList.end(); currElem++ )
			{
				std::cout << spaces.String() << "\t";
				std::cout << currElem->first.String() << ": ";	// output "key:"
				currVal = dynamic_cast<TalkVarValue*>(currElem->second);
				if( currVal )
					DumpOneValue( currVal );	// Output actual value.
				else
					std::cout << "-unknown value type-";
				std::cout << endl;
			}
			std::cout << spaces.String() << "}";
		}
		else
		{
			switch( theValue->GetNativeType() )
			{
				case VALUE_TYPE_LONG:
					std::cout << spaces.String() << "long (" << (long) theValue << ") ";
					std::cout << theValue->mValue.longType;
					break;
				
				case VALUE_TYPE_DOUBLE:
					std::cout << spaces.String() << "double (" << (long) theValue << ") ";
					std::cout << theValue->mValue.doubleType;
					break;
				
				case VALUE_TYPE_BOOL:
					std::cout << spaces.String() << "bool (" << (long) theValue << ") ";
					if( theValue->mValue.boolType )
						std::cout << "true";
					else
						std::cout << "false";
					break;
				
				case VALUE_TYPE_TEXT:
					std::cout << spaces.String() << "text (" << (long) theValue << ") \"";
					std::cout << theValue->mValue.textType->String();
					std::cout << "\"";
					break;
				
				case VALUE_TYPE_VALUE:
					std::cout << spaces.String() << "reference (" << (long) theValue << ") " << (long) theValue->mValue.valueType;
					break;
				
				default:
					std::cout << spaces.String() << "unknown (" << theValue->GetNativeType() << ")";
			}
		}
		
		spaces.SetOffset(1);
		spaces.DeleteData(1);
	};
	void	Dump()
	{
		BasicValueStack::iterator	itty;
		size_t						x = 0;
		
		std::cout << endl;
		std::cout << "Size: " << size() << " Base pointer: " << mBasePointer << endl;
		
		for( itty = begin(); itty != end(); itty++ )
		{
			++x;
			
			if( x == mBasePointer )
				std::cout << "> ";
			else
				std::cout << "  ";
			
			DumpOneValue( *itty );
			
			std::cout << endl;
		}
	};
	
	void	SetErrorSource( long errL, HyperTalk* errS )	{ mErrorLine = errL; mErrorScript = errS; };
	void	GetErrorSource( long& errL, HyperTalk* &errS )	{ errL = mErrorLine; errS = mErrorScript; };
};

typedef BasicValueStack::reverse_iterator	ValueStackRIterator;
typedef size_t ValStackLocation;


#pragma mark [Prototypes]

/* --------------------------------------------------------------------------------
	Prototypes:
   ----------------------------------------------------------------------------- */










#endif /*BASICVALUESTACK_H*/


