/*	***************************************************************************

	PROJECT:	HyperTalk
	
	FILE:		HyperTalk.cpp
	
	PURPOSE:	HyperTalk interpreter.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-02-14	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"HyperTalk.h"
#include	"TalkTokenize.h"
#include	"TalkTempVarList.h"
#include	"TalkInternalVarList.h"
#include	"TalkVarNameList.h"
#include	<iostream>
#include	<stack>
#include	"strcase.h"		// Our own strcasecmp().
#include	"parse_error.h"
#include	"missinghandler_error.h"
#include	"TalkCallRecord.h"


#pragma mark [Globals]

/* --------------------------------------------------------------------------------
	Globals/Static Variables:
   ----------------------------------------------------------------------------- */

// Static variables:
bool						HyperTalk::sInited = false;
TalkGlobalsList				HyperTalk::sGlobals;
long						HyperTalk::sOptions[OPTIONS_LAST];
TalkFunctionList			HyperTalk::sOneArgFunctions;
TalkFunctionList			HyperTalk::sBuiltInFunctions;
TalkFunctionList			HyperTalk::sBuiltInMessages;
TalkBuiltInCmdList			HyperTalk::sBuiltInCommands;
TalkGlobalPropertyList		HyperTalk::sGlobalProperties;
TalkObjectDescriptorList	HyperTalk::sObjectDescriptors;
TalkConstantList			HyperTalk::sConstants;
TalkOperatorList			HyperTalk::sUnaryOperators;
TalkScriptBlockList			HyperTalk::sScriptBlocks;

// Script block objects provided by Joker as defaults:
TalkScriptBlock				gDefaultScriptBlock;
TalkFcnHandlerBlock			gFcnHandlerBlock;
TalkCmdHandlerBlock			gCmdHandlerBlock;

#pragma mark -
#pragma mark [Init/Kill]

/* --------------------------------------------------------------------------------
	* CONSTRUCTOR:
		Do some initialization.
	
	REVISIONS:
		1999-02-14	UK		Created.
   ----------------------------------------------------------------------------- */

HyperTalk::HyperTalk()
{
	mCurrIterator = NULL;
	mBossInHierarchy = NULL;
	mRefCount = 0;
	mCurrHandler = NULL;
	mAssociatedObject = NULL;
	
	InitializeVocabulary();
}

void	HyperTalk::ForceInitialize()
{
	sInited = false;
	InitializeVocabulary();
}


/* --------------------------------------------------------------------------------
	InitializeVocabulary:
		Register all language keywords with Joker. This only does something the
		first time it is called. The HyperTalk constructor calls this
		automatically when a HyperTalk instance is created. If you want to
		register your own keywords and syntax, you can do so after calling this
		yourself at startup.
	
	REVISIONS:
		2000-11-11	UK		Moved here from constructor.
   ----------------------------------------------------------------------------- */

void	HyperTalk::InitializeVocabulary()
{
	if( sInited )
		return;
	
	// One argument functions:
	HyperTalk::RegisterOneArgFunction( TextMunger("param"), HyperGetParamInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("keys"), HyperGetKeysInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("chartonum"), HyperCharToNumInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("numtochar"), HyperNumToCharInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("length"), HyperCountChunkInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("checksum"), HyperCheckSumInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("numToShort"), HyperNumToShortInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("numToLong"), HyperNumToLongInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("shortToNum"), HyperShortToNumInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("longToNum"), HyperLongToNumInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("bitnot"), HyperBitNotInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("hextonum"), HyperHexToNumInstruction );
	HyperTalk::RegisterOneArgFunction( TextMunger("numtohex"), HyperNumToHexInstruction );
	
	// Other built-in functions:
	HyperTalk::RegisterBuiltInFunction( TextMunger("offset"), HyperFindOffsetInstruction );
	HyperTalk::RegisterBuiltInFunction( TextMunger("bitand"), HyperBitAndInstruction );
	HyperTalk::RegisterBuiltInFunction( TextMunger("bitor"), HyperBitOrInstruction );
	
	// Message handlers:
	HyperTalk::RegisterBuiltInMessage( "appleEvent", HyperAppleEventInstruction );
	HyperTalk::RegisterBuiltInMessage( ".makePackage", HyperMakePackageInstruction );
	HyperTalk::RegisterBuiltInMessage( "beep", HyperBeepInstruction );
	
	HyperTalk::RegisterBuiltInCommands();		// Built-in commands.
	HyperTalk::RegisterObjectDescriptors();		// Object descriptors.
	
	// Special variables:
	HyperTalk::RegisterGlobPropOrVar( TextMunger("result"), TALK_PROP_TYPE_LOCAL_VARIABLE,
										TextMunger("the result") );
	HyperTalk::RegisterGlobPropOrVar( TextMunger("itemDelimiter"), TALK_PROP_TYPE_LOCAL_VARIABLE,
										TextMunger("the itemDelimiter") );
	
	// Constants:
	HyperTalk::RegisterConstant( "true", new TalkVarValue( (bool) true ) );
	HyperTalk::RegisterConstant( "false", new TalkVarValue( (bool) false ) );
	HyperTalk::RegisterConstant( "return", new TalkVarValue( new TextMunger("\r") ) );
	HyperTalk::RegisterConstant( "cr", new TalkVarValue( new TextMunger("\r") ) );
	HyperTalk::RegisterConstant( "lineFeed", new TalkVarValue( new TextMunger("\n") ) );
	HyperTalk::RegisterConstant( "lf", new TalkVarValue( new TextMunger("\n") ) );
	HyperTalk::RegisterConstant( "tab", new TalkVarValue( new TextMunger("\t") ) );
	HyperTalk::RegisterConstant( "colon", new TalkVarValue( new TextMunger(":") ) );
	HyperTalk::RegisterConstant( "quote", new TalkVarValue( new TextMunger("\"") ) );
	HyperTalk::RegisterConstant( "space", new TalkVarValue( new TextMunger(" ") ) );
	HyperTalk::RegisterConstant( "comma", new TalkVarValue( new TextMunger(",") ) );
	HyperTalk::RegisterConstant( "empty", new TalkVarValue( new TextMunger("") ) );
	
	char		nullChar = 0;	// Can't pass a NULL character as a C string, so use a variable as an unterminated text buffer.
	HyperTalk::RegisterConstant( "null", new TalkVarValue( TextMunger( &nullChar, 1 ) ) );
	nullChar = TalkTokenizer::GetNewline();
	HyperTalk::RegisterConstant( "newline", new TalkVarValue( new TextMunger(&nullChar, 1) ) );
	
	// Unary operators:
	HyperTalk::RegisterUnaryOperator( OPERATOR_CHAR_MINUS, HyperUnaryMinusInstruction );
	HyperTalk::RegisterUnaryOperator( OPERATOR_CHAR_NOT, HyperUnaryNotInstruction );
	
    // Script blocks:
    RegisterScriptBlock( TOKEN_TYPE_INVALID, &gDefaultScriptBlock );	// TOKEN_TYPE_INVALID is used for default block as it can't occur in a script.
    RegisterScriptBlock( TOKEN_TYPE_START_FUNCTION, &gFcnHandlerBlock );
    RegisterScriptBlock( TOKEN_TYPE_START_HANDLER, &gCmdHandlerBlock );
    
	TryToInitOptions();	// Sets sInited to TRUE.
	
	// Let Tokenizer register its tokens and calculate string hashes for more speed:
	TalkTokenizer::Init();
}


/* --------------------------------------------------------------------------------
	* DESTRUCTOR:
		Do some cleanup.
	
	REVISIONS:
		1999-02-14	UK		Created.
   ----------------------------------------------------------------------------- */

HyperTalk::~HyperTalk()
{
	// Do your stuff here!
}


#pragma mark -
#pragma mark [Running]


/* --------------------------------------------------------------------------------
	TryToInitOptions:
		Initialize our list of options. This function only does this the first
		time it is called. This sets sInited to true.
	
	REVISIONS:
		2000-11-10	UK		Added function/command distinction.
		2000-01-18	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::TryToInitOptions()
{
	if( sInited )
		return;
	
	// First time this is called? Init options to default.
	sOptions[OPT_ALLOW_UNQUOTED_LITERALS] = true;
	sOptions[OPT_DISTINGUISH_FCN_CMD] = true;
	sOptions[OPT_VARS_AS_UNQUOTED_LITERALS] = true;
	sOptions[OPT_ESCAPE_CHARS_IN_STRINGS] = false;
	
	sInited = true;
}


/* --------------------------------------------------------------------------------
	SetOption:
		Set one of our compiler options. Compiler options control the compiler's
		behaviour in certain situations like whether unquoted literals are
		supported or will cause a compile-time error.
		
	TAKES:
		inOption	-	The array index of the option to change.
		inValue		-	The value to set the option to. This depends on inOption
						and is usually either a number or true/false.
	
	REVISIONS:
		2000-01-18	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::SetOption( long inOption, long inValue )
{
	TryToInitOptions();	// Make sure all have been inited.
	
	// First time this is called? Init options to default.
	sOptions[inOption] = inValue;
}


/* --------------------------------------------------------------------------------
	SendMessage:
		Run a handler with the specified name.
		
		If you pass a stack to this function call, you are responsible for pushing
		the parameters followed by the parameter count onto the stack. However
		this call will take care of cleaning up the stack behind you.
	
	TAKES:
		inName	-	Name of handler to call.
		vStack	-	Call stack with parameters on it. This is where this call
					will allocate its variables. Specify NULL to have this call
					automatically create a stack with no parameters on it.
		vResult	-	A memory location where the handler's result will
					be stored. Pass a location of type MEM_LOCATION_TYPE_INVALID
					here to have nothing returned.
		isFcn	-	Specifies whether you are trying to call a function message
					handler or a command. If Joker's options are set to
					distinguish between commands and functions, this will
					automatically perform the proper name mangling.
	
	REVISIONS:
		2000-12-01	UK	Added return value & eatMessage parameter.
		2000-11-10	UK	Created.
   ----------------------------------------------------------------------------- */

TalkCallResult	HyperTalk::SendMessage( const TextMunger& inName, ValueStack *vStack,
								TalkMemLocation vResult, bool isFcn, bool eatMessage )
{
	TextMunger		vRealHandlerName( inName );
	bool			vTrashStack = false;
	TalkCallResult	vCallResult;
	
	if( sOptions[OPT_DISTINGUISH_FCN_CMD] == true )	// Perform name mangling to distinguish functions & commands.
	{
		vRealHandlerName.SetOffset(0);
		if( isFcn )
			vRealHandlerName.Insert<char>( '\t' );	// Tab is impossible to enter as part of a function name.
		else
			vRealHandlerName.Insert<char>( ' ' );	// Space is impossible to enter as part of a command name.
	}
	
	if( vStack == NULL )
	{
		vStack = new ValueStack();
		vTrashStack = true;
	}
	
	try
	{
		if( vTrashStack )	// We created the stack ourselves? Push 0 param count on it!
		{
			TalkVarValue*		vValue;
			
			vValue = new TalkVarValue( (long) 0 );	// Will be disposed by Run(name,stack) below.
			if( !vValue )
				throw bad_alloc();
			vStack->push_back( vValue );	// Push!
		}
		
		vCallResult = Run( vRealHandlerName, *vStack, vResult, eatMessage );
		
		if( vTrashStack )
			delete vStack;
	}
	catch( ... )
	{
		if( vTrashStack )
			delete vStack;
		throw;
	}
	
	return vCallResult;
}

/* --------------------------------------------------------------------------------
	SendMessage:
		Run the specified command Handler. This doesn't allow passing any
		parameters.
	
	TAKES:
		inName	-	Name of handler to call.
	
	
	REVISIONS:
		2001-01-18	UK	Renamed to SendMessage.
		2000-12-01	UK	Added return value & eatMessage parameter.
		1999-12-11	UK	Created.
   ----------------------------------------------------------------------------- */

TalkCallResult	HyperTalk::SendMessage( const TextMunger& inName, bool eatMessage )
{
	ValueStack		vStack;
	TalkVarValue*	vValue;
	TalkMemLocation	vResultMemLoc;
	TextMunger		vRealHandlerName( inName );
	
	if( sOptions[OPT_DISTINGUISH_FCN_CMD] == true )	// Perform name mangling to distinguish functions & commands.
	{
		vRealHandlerName.SetOffset(0);
		vRealHandlerName.Insert<char>( ' ' );	// Space is impossible to enter as part of a command name.
	}
	
	// Push param count (none) onto stack:
	vValue = new TalkVarValue( (long) 0 );	// Will be disposed by Run(name,stack) below.
	if( !vValue )
		throw bad_alloc();
	vStack.push_back( vValue );	// Push!
	
	vResultMemLoc.mType = MEM_LOCATION_TYPE_INVALID;
	return Run( vRealHandlerName, vStack, vResultMemLoc, eatMessage );	// When this call exits it will automatically dispose of param count.
}





/* --------------------------------------------------------------------------------
	Run:
		Run the specified Handler. This is called by the JSR instruction.
	
	WARNING:
		This method expects you to push the parameters followed by the parameter
		count onto the stack before you call it, but will dispose of the params
		automatically. DO NOT TRY TO CLEAN UP THE STACK AFTER THIS CALL!
	
	TAKES:
		inName	-	Name of handler to call (already munged if we distinguish
					between commands and functions).
		vStack	-	Call stack with parameters on it. This is where this call
					will allocate its variables.
		vResult	-	A memory location where the handler's result will
					be stored. Pass a location of type MEM_LOCATION_TYPE_INVALID
					here to have nothing returned.
	
	REVISIONS:
		2000-12-01	UK	Added return value & eatMessage parameter.
		2000-10-26	UK	Added support for messaging hierarchy, built-in functions
						& passing.
		2000-10-23	UK	Added some debugging support.
		2000-10-20	UK	Added support for handler/function result.
		1999-12-12	UK	Added if/else (goto) support.
		1999-12-11	UK	Changed param to TextMunger. Made clean up after return.
		1999-12-05	UK	Implemented, changed param to take string.
		1999-02-14	UK	Created.
   ----------------------------------------------------------------------------- */

TalkCallResult	HyperTalk::Run( const TextMunger& inName, ValueStack &vStack,
								TalkMemLocation vResult, bool eatMessage )
{
	TalkHandler::iterator	dbiterator;
	TalkScript::iterator	iterator;
	size_t					x, finalSize, vOldBP;
	TalkHandler::iterator*	vOldIter;
	TalkHandler*			vOldHand;
	TalkCallRecord			vCallRec;
	HyperTalk*				vBoss;
	TalkCallResult			vWasHandled;	// Only used if eatMessage == true.
	
	Retain();	// Make sure we aren't deleted until we have finished running.
	
	try {
		// Look for handler of that name:
		iterator = mScript.find( inName );
		if( iterator != mScript.end() )		// Found one?
		{
			TalkHandler&	vCode = (*iterator).second;
			
		  #if JOKER_STACK_TRACES
			std::cout << "\nStack before function call:\n";
			vStack.Dump();
		  #endif
				
			// Move base pointer to where our variables start and our params end:
			vOldBP = vStack.GetBasePointer();	// But remember old one so we can restore later.
			vStack.SetBasePointer( vStack.size() );
			
			// Grow stack:
			ValueStackRIterator itty;
			size_t		vStkSz = vStack.size();
			finalSize = vStkSz +vCode.GetStackNeed();
			vStack.resize( finalSize +1 );	// Make stack larger. +1 so there's space for result.
			
			// Now iterate over the var slots just added at the back (from the back) and create the variables:
			x = vCode.GetStackNeed() +1;	// +1 so result is initialized, too.
			for( itty = vStack.rbegin(); itty != vStack.rend(); ++itty )
			{
				(*itty) = new TalkVarValue( (long) 0 );
				
				if( --x == 0 )
					break;
			}
			
			// Make an iterator and remember it in mCurrIterator so goto can access it:
			dbiterator = vCode.begin();
			vOldIter = mCurrIterator;	// Remember previous iterator in case it's a recursive call.
			mCurrIterator = &dbiterator;
			
			// Also remember current handler for debugger/goto:
			vOldHand = mCurrHandler;
			mCurrHandler = &vCode;
			
		  #if JOKER_STACK_TRACES
			std::cout << "\nStack during function call:\n";
			vStack.Dump();
		  #endif
		  #if JOKER_MINI_DEBUGGER
			bool		runDontStep = false;
		  #endif
			
			/* Run handler:
				Either we keep executing until we run out of commands, or we execute until
				some command sets mExitHandler in the callRec to true. Used e.g. by "return". */
			for( ; (dbiterator != vCode.end()) && vCallRec.mExitHandler == false; ++dbiterator )
			{
				if( (*dbiterator).mInstructionProc != NULL )
				{
				  #if JOKER_MINI_DEBUGGER
					if( !runDontStep || ((*dbiterator).mInstructionProc == HyperBreakpointInstruction) )
					{
						char		vChar;
						bool		dontGo = true;
						runDontStep = false;	// If this was a breakpoint, allow to continues stepwise now.
						
						(*dbiterator).Dump();	// Print current instruction.
						
						while( dontGo )
						{
							std::cout << "\njmd# ";
							std::cin >> vChar;
							std::cout << "\n\n";
							
							switch( vChar )
							{
								case 's':
								case 'S':
									vStack.Dump();
									break;
								
								case 'x':
								case 'X':
									vCallRec.mExitHandler = true;
									dontGo = false;
									break;
								
								case 'r':
								case 'R':
									runDontStep = true;
									dontGo = false;
									break;
								
								case 'g':
								case 'G':
									dontGo = false;
									break;
								
								case '.':
									throw runtime_error( "Script aborted." );
									break;
								
								case '?':
								case 'h':
									std::cout << "Type:\n's' for a stack crawl\n'x' to abort this handler\n"
												<< "'.' to exit to top\n'g' to execute the next instruction.\n"
												<< "'r' to run this handler.";
									break;
							}
						}
					}
					
					if( vCallRec.mExitHandler == false )
				  #endif /*JOKER_MINI_DEBUGGER*/
						(*(*dbiterator).mInstructionProc)( (*dbiterator), vStack, vCallRec );
					
				  #if JOKER_PRINT_INSTR_LINEWISE
					(*dbiterator).Dump();	// Print current instruction.
				  #endif
				  
					// FIX ME! Check vCallRec.mException here and jump to next exception handler. Add parsing code for try/catch/throw.
				}
			}
			
			mCurrIterator = vOldIter;	// Restore previous iterator in case it's a recursive call.
			mCurrHandler = vOldHand;
			
			/* Shrink stack:
				We de-allocate all local variables here and get rid of parameters. We do the latter
				by looking at the stack entry before the base pointer, which is the param count, and
				then killing that many params off the end of the stack. */
			ValueStorage		vValue;
			TalkVarValue*		vResultValue;
			
			size_t	endSize = vStack.GetBasePointer();	// Calc final size for stack.
			x = finalSize -vStack.GetBasePointer();		// Calc sum of vars and params but don't add result as that is deleted separately below. +1 for param count.
			
			// If we're not passing the call, also remove params from stack.
			if( !vCallRec.mPassFlag )
			{
				vStack[ vStack.GetBasePointer() -1 ]->GetValue( vValue, VALUE_TYPE_LONG );	// Get param count.
				endSize -= vValue.longType +1;	// +1 is so we delete param count itself, too.
				x += vValue.longType +1;		// +1 again for param count.
			}
			
			// Get result from stack and remove it:
			vResultValue = vStack.back(); vStack.pop_back();
			
			// Reverse-Iterate over the stack and kill variables (& param count):
			for( itty = vStack.rbegin(); itty != vStack.rend(); ++itty )
			{
				delete (*itty);
				
				if( --x == 0 )
					break;
			}
			
			vStack.resize( endSize );	// Make stack smaller.
			
			// Restore base pointer so caller can find variables:
			vStack.SetBasePointer( vOldBP );
			
		  #if JOKER_STACK_TRACES
			std::cout << "\nStack after function call:\n";
			vStack.Dump();
		  #endif
			
			// Now that we're in callER's context again, store the result:
			TalkValue*		vDest;
			if( vResult.mType != MEM_LOCATION_TYPE_INVALID )
			{
				vResult.GetValue( vDest, vStack );
				vResultValue->CopyValueTo( (*vDest), true, true );
			}
			
			delete vResultValue;	// Get rid of value.
		}
		else
			vCallRec.mPassFlag = true;	// None found = same as pass.
		
		if( vCallRec.mPassFlag )	// Message wasn't intercepted? Pass it up the hierarchy.
		{
			vBoss = GetBossInHierarchy();	// Don't directly access this, might be overridden.
			if( !vBoss )	// No boss? Guess we're top of hierarchy:
				vWasHandled = CallBuiltInFunction( inName, vStack, vResult, eatMessage );	// Try built-in functions as a last resort.
			else	// Have a boss? Just let boss handle this or dispatch to built-in functions:
				vWasHandled = vBoss->Run( inName, vStack, vResult, eatMessage );
		}
		else
		{
			if( vCallRec.mExitHandler && vCallRec.mExitCompletely )	// Wasn't just end of handler, was "exit to top" ?
				vWasHandled = CALL_RESULT_EXITED;	// Tell caller so it can tear down calling chain.
			else
				vWasHandled = CALL_RESULT_INTERCEPTED;	// Just tell caller it was intercepted and not passed.
		}
		
		if( vCallRec.mException )
			vWasHandled = CALL_RESULT_EXCEPTION;
		
		Release();
	}
	catch( exception& err )
	{
		Release();
		throw;
	}
	
	return vWasHandled;
}

/* --------------------------------------------------------------------------------
	EmitHandlerNotFoundErr:
		React on an uncaught handler call. This is a VIRTUAL function, you can
		override it to check for XCMDs/XFCNs or something like that.
	
	TAKES:
		isFcn				-	TRUE if this is a function message handler,
								FALSE for a command message handler.
		vRealHandlerName	-	Name of the handler to call.
		vStack				-	The stack with the parameters.
		vResult				-	The location to store the result.
	
	REVISIONS:
		2001-07-10	UK	Moved here from CallBuiltInFunction().
   ----------------------------------------------------------------------------- */

void	HyperTalk::EmitHandlerNotFoundErr( bool isFcn, TextMunger& vRealHandlerName,
											ValueStack& vStack, TalkMemLocation vResult )
{
	throw missinghandler_error( isFcn ? "Couldn't find this function handler." : "Couldn't find this command handler.", vRealHandlerName );
}


/* --------------------------------------------------------------------------------
	CallBuiltInFunction:
		Run the specified built-in Handler. This is called whenever a function or
		handler call has reached the end of the message hierarchy without being
		swallowed. It checks the one-argument functions, built-in functions and
		built-in messages lists for a handler of specified name and runs that. If
		the parameter count is 4 or less, the parameters will be passed to the
		instruction in the HyperInstruction struct, else they will be passed on
		the stack.
	
	WARNING:
		This method expects you to push the parameters followed by the parameter
		count onto the stack before you call it, but will dispose of the params
		automatically. DO NOT TRY TO CLEAN UP THE STACK AFTER THIS CALL!
	
	TAKES:
		inName	-	Name of handler to call.
		vStack	-	Call stack with parameters on it.
		vResult	-	A memory location where the handler's result will
					be stored. Pass a location of type MEM_LOCATION_TYPE_INVALID
					here to have nothing returned.
	
	REVISIONS:
		2001-07-10	UK	Extracted throw missinghandler_error to
						EmitHandlerNotFoundErr().
		2000-12-01	UK	Added return value & eatMessage parameter.
		2000-11-11	UK	Made this also check the built-in message list.
		2000-10-26	UK	Created.
   ----------------------------------------------------------------------------- */

TalkCallResult	HyperTalk::CallBuiltInFunction( const TextMunger& inName, ValueStack &vStack,
												TalkMemLocation vResult, bool eatMessage )
{
	TalkFunctionList::iterator		fli;
	HyperInstruction				vThisInstr;
	TextMunger						vRealHandlerName( inName );
	char							vPrefix = 0;
	
	// FIX ME! This is _waaaay_ too slow.
	
	if( sOptions[OPT_DISTINGUISH_FCN_CMD] == true )	// Undo name mangling done to distinguish functions & commands.
	{
		vRealHandlerName.SetOffset(0);
		vPrefix = vRealHandlerName.Peek<char>();
		vRealHandlerName.Delete<char>();
	}
	
	if( vPrefix != '\t' )	// If we don't distinguish or this is a handler call:
		fli = sBuiltInMessages.find( vRealHandlerName );	// Look for a registered message handler.
	else
		fli = sBuiltInMessages.end();
	
	if( vPrefix != ' ' && fli == sBuiltInMessages.end() )	// If we don't distinguish and/or no cmd handler was found:
	{
		// Look for a one-argument function of this name:
		fli = sOneArgFunctions.find( vRealHandlerName );
		if( fli == sOneArgFunctions.end() )	// Nothing? Try those with several args:
		{
			fli = sBuiltInFunctions.find( vRealHandlerName );
			if( fli == sBuiltInFunctions.end() )
				EmitHandlerNotFoundErr( true, vRealHandlerName, vStack, vResult );
		}
	}
	else if( fli == sBuiltInMessages.end() )
	{
		if( eatMessage )
			return CALL_RESULT_PASSED;	// Just return silently.
		else
			EmitHandlerNotFoundErr( false, vRealHandlerName, vStack, vResult );
	}
	
	// Now generate fake instruction for this function call:
	vThisInstr.mCaller = this;
	vThisInstr.mInstructionProc = (*fli).second;	// Proc comes from one of the three function/command handler lists.
	vThisInstr.mResult = vResult;
	
	ValueStackRIterator itty;
	ValueStorage		vValue;
	
	itty = vStack.rbegin();
	if( itty == vStack.rend() )
		throw logic_error( "Call stack FUBAR! Help!" );
	
	(*itty)->GetValue( vValue, VALUE_TYPE_LONG );	// Get param count.
	if( vValue.longType > TALK_INSTR_MAX_PARAMS )	// too many params?
		vThisInstr.mParamCount = INSTRUCTION_PARAM_COUNT_ON_STACK;	// Just leave them on the stack.
	else	// If TALK_INSTR_MAX_PARAMS or less, pass them to instruction in the HyperInstruction struct:
	{
		vThisInstr.mParamCount = vValue.longType;
		
		// Since we're walking the stack backwards, we also need to have x counting down:
		short	x = vValue.longType;
		while( --x >= 0 )
		{
			++itty;
			
			vThisInstr.mParam[x].mType = MEM_LOCATION_TYPE_IMMEDIATE;
			vThisInstr.mParam[x].mValue = (*itty);
		}
	}
	
	TalkCallRecord			vCallRec;
	
	// Now actually call the instruction:
	(*vThisInstr.mInstructionProc)( vThisInstr, vStack, vCallRec );
	
	// Finally, get rid of params & param count on stack:
	short		x = vValue.longType +1;
	
	for( itty = vStack.rbegin(); itty != vStack.rend(); ++itty )
	{
		delete (*itty);
		
		if( --x == 0 )
			break;
	}
	
	vStack.resize( vStack.size() -vThisInstr.mParamCount -1 );
	
	vThisInstr.ClearAllFields( false );	// Make sure values we already disposed aren't disposed by destructor.
	
	return CALL_RESULT_INTERCEPTED;
}


/* --------------------------------------------------------------------------------
	GetGlobal:
		Get a global from our list of globals, possibly creating a new one if
		requested and there isn't one yet.
	
	REVISIONS:
		1999-12-29	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::GetGlobal( TalkValue* &vValue, TextMunger& inGlobName, bool doCreate )
{
	vValue = sGlobals.GetVarByName( inGlobName, doCreate );	// These globals are shared by all scripts.
}


/* --------------------------------------------------------------------------------
	Goto:
		Used internally to implement if/then/else branching. Please don't try to
		use this to implement a gwbasic-style goto command, as this is something
		that will cause trouble with OOP stuff and is error-prone.
		
		This is where the offset needed when initialization code is inserted
		at the top of the code is applied.
	
	NOTE:	This only works on the current handler and relative to the current
			handler (like C gotos).
	
	TAKES:
		vAddress	-	The instruction to jump to. This counts the first
						instruction after the initialization block as number
						zero.
	
	REVISIONS:
		1999-12-12	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::Goto( long vAddress )
{
	(*mCurrIterator) = mCurrHandler->begin();	// Address is absolute, we need to go to start before we can add it.
	(*mCurrIterator) += vAddress;	// Add address.
	(*mCurrIterator) += mCurrHandler->GetInitCodeSize();
}


#pragma mark -
#pragma mark [Parser Initialization]



/* --------------------------------------------------------------------------------
	RegisterOneArgFunction:
		Register a one-argument built-in function with the parser. These functions
		can be called just like any function, but can also be called bypassing the
		message hierarchy using "[the] <fcnName> of <param1>". Since all these
		functions have the same syntax, the ProcPtrs for their instructions are
		kept in a list with the function names as the keys. The parser simply
		looks whether an identifier is in this list when it encounters the "of"
		keyword or an unresolved function call, and if it is generates the
		appropriate instruction.
	
	REVISIONS:
		2000-10-23	UK	Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::RegisterOneArgFunction( const TextMunger& vFcnName,
											HyperInstrProcPtr vInstructionProc )
{
	vFcnName.RecalcHash();	// Will be taken over by function map --> faster.
	
	sOneArgFunctions[vFcnName] = vInstructionProc;
}


/* --------------------------------------------------------------------------------
	RegisterUnaryOperator:
		Register a unary operator, i.e. an operator that takes only one argument.
	
	REVISIONS:
		2001-04-07	UK	Changed to use only operator type instead of entire token.
		2001-03-10	UK	Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::RegisterUnaryOperator( OperatorTypeEnum vOpType,
											HyperInstrProcPtr vInstructionProc )
{
	sUnaryOperators[vOpType] = vInstructionProc;
}


/* --------------------------------------------------------------------------------
	RegisterBuiltInFunction:
		Register a built-in function with the parser.
		
	Note:
		Do not register one-argument functions with this. If a one-argument
		function is called with the traditional function-calling syntax, it will
		receive all parameters, not just one.
	
	REVISIONS:
		2000-10-26	UK	Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::RegisterBuiltInFunction( const TextMunger& vFcnName,
											HyperInstrProcPtr vInstructionProc )
{
	vFcnName.RecalcHash();	// Will be taken over by function map --> faster.
	
	sBuiltInFunctions[vFcnName] = vInstructionProc;
}


/* --------------------------------------------------------------------------------
	RegisterBuiltInMessage:
		Register a built-in command message handler with the parser.
		
		If Joker is set not to distinguish between commands and functions, this
		is exactly the same as RegisterBuiltInFunction, with the exception that
		if you register a handler name with this it will override any existing
		functions of same name in the list of built-in or one-argument functions.
		
		If Joker is set to distinguish between commands and functions, commands
		are looked up in this list, while functions are looked up in the lists
		of one-arg functions and functions.
	
	REVISIONS:
		2000-11-10	UK	Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::RegisterBuiltInMessage( const TextMunger& vFcnName,
											HyperInstrProcPtr vInstructionProc )
{
	vFcnName.RecalcHash();	// Will be taken over by function map --> faster.
	
	sBuiltInMessages[vFcnName] = vInstructionProc;
}


/* --------------------------------------------------------------------------------
	RegisterBuiltInCommand:
		Register a built-in command with the parser. Built-in commands may have
		english-like syntax and can not be called using function syntax. However,
		they are a little faster as they do not need to go through the message
		hierarchy but instead are compiled directly to instructions.
		
	Note:
		You can only register commands using this function that begin with a
		token that is not used by any other command. E.g. "put", which compiles
		to two different instructions depending on whether it has an indirect
		argument or not, can not be registered using this function, as registering
		the second variant would replace the first one (they're looked up by their
		first token). However, commands that take optional parameters can easily
		be registered using this function.
			
	REVISIONS:
		2000-10-26	UK	Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::RegisterBuiltInCommand( TalkCommandEntry* vCommand )
{
	sBuiltInCommands[vCommand->mParam[0].mTokenType] = vCommand;
}


/* --------------------------------------------------------------------------------
	RegisterScriptBlock:
        Registers a script block parser object with the parser.
        
    TAKES:
        vStartIdentifier	-	The token that starts off this kind of block in
                                the script. If you register a token twice, the
                                second one will replace the first one.
        block				-	A TalkScriptBlock (or subclass) object to handle
                                parsing this particular kind of script block.
        
    DIRECTIONS:
		A script block is an object that parses part of a script. To Joker,
        everything in a script it can make something of is a script block. The
        only thing the main parser loop knows is how to find a token, check it
        against the registered script block tokens, and have the script block
        handle it. The default script block simply skips the current line.
        
        So, how does Joker parse handlers? There are two script block objects
        in Joker: TalkFcnHandlerBlock and TalkCmdHandlerBlock. The first one is
        registered under TOKEN_TYPE_START_FUNCTION ("function") and the second
        one is registered under TOKEN_TYPE_START_HANDLER ("on"). Whenever the
        parser encounters any of these tokens, it finds these script block objects
        in its registry and calls upon them to actually do the parsing, (For
        historical reasons these two just call the parser's ParseHandler() method,
        but that's a detail that needn't concern us here) and that's what they'll
        do until they hit an "end <handlerName>" statement, which is the point
        when they transfer control back to the parser's main loop.
        
        If you wanted to add your own kind of script block to Joker (e.g. another
        kind of handler or a way to declare script-global variables or a way to
        declare some sort of object class), simply subclass TalkScriptBlock and
        override its Parse() method. Then register an object of your new type with
        this call. The "start identifier" is the token that signals to Joker that
        now begins a block of your kind and not a handler or whatever. E.g. if
        the syntax you want is
        
        class userObject
          name: "Peter"
          password: "secret"
          userID: 500
        end class
        
        then you'd register your TalkClassBlock class with TOKEN_TYPE_CLASS
        (given you have registered this token with the parser).
        
        Note that the default block is registered as TOKEN_TYPE_INVALID, in case
        you want to replace that.
			
	REVISIONS:
		2001-09-29	UK	Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::RegisterScriptBlock( TokenTypeEnum vStartIdentifier,
                                        TalkScriptBlock* block )
{
	sScriptBlocks[vStartIdentifier] = block;
}

/* --------------------------------------------------------------------------------
	RegisterConstant:
		Register a constant with the parser.
	
	REVISIONS:
		2001-01-05	UK	Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::RegisterConstant( const TextMunger& vName, TalkValue* vValue )
{
	vName.RecalcHash();	// Will be taken over by function map --> faster.
	
	sConstants[vName] = vValue;
}


/* --------------------------------------------------------------------------------
	RegisterObjectDescriptor:
		Register a built-in object descriptor with the parser. Built-in object
		descriptors may have english-like syntax and describe an object in the
		current context.
		
	Note:
		You can only register object descriptors using this function that begin
		with a token that is not used by any other descriptor, as registering
		the second variant would replace the first one (they're looked up by their
		first token). However, you can use optional parameters that change the
		instruction used to work around this.
			
	REVISIONS:
		2000-12-22	UK	Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::RegisterObjectDescriptor( TalkCommandEntry* vDescriptor )
{
	sObjectDescriptors[vDescriptor->mParam[0].mTokenType] = vDescriptor;
}


#pragma mark -
#pragma mark [Parsing]


/* --------------------------------------------------------------------------------
	Parse:
		Parse the script that was tokenized before.
	
	REVISIONS:
		1999-06-22	UK		Removed parameter.
		1999-02-14	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::Parse()
{
	HTLIterator						iterator;
    TalkScriptBlockList::iterator	vSBlock;
	long							vCurrLineNum = 1;
	
	for( iterator = mTokens.begin(); iterator != mTokens.end(); ++iterator )
	{
		HyperToken*		vToken;
		
		vToken = (*iterator);
        
        // Look up proper script block for this token:
        vSBlock = sScriptBlocks.find( vToken->GetType() );
        if( vSBlock == sScriptBlocks.end() )	// None found? Use default.
            vSBlock = sScriptBlocks.find( TOKEN_TYPE_INVALID );	// Default one is registered as TOKEN_TYPE_INVALID. It just skips this line.
        
        // Now call upon script block to do the parsing:
        (*vSBlock).second->Parse( *this, iterator, vCurrLineNum );
        
		/*	// Superceded by script block registry.
		switch( vToken->GetType() )
		{
			// Handler start? Parse a handler:
			case TOKEN_TYPE_START_HANDLER:
				ParseHandler( iterator, false, vCurrLineNum );
				break;
			
			// Function start? Parse a function:
			case TOKEN_TYPE_START_FUNCTION:
				ParseHandler( iterator, true, vCurrLineNum );
				break;
			
			// Anything else? Just skip this line.
			default:
				while( vToken->GetType() != TOKEN_TYPE_NEWLINE )
				{
					++iterator;
					if( iterator == mTokens.end() )
						return;
					
					vToken = (*iterator);
				}
				vCurrLineNum++;
		}
		*/
		if( iterator == mTokens.end() )	// If handler parsing uses up all tokens.
			return;		// Exit.
	}
}


/* --------------------------------------------------------------------------------
	ParseHandlerBody:
		Parse the script that was tokenized before for a couple of commands,
        until it hits either the end of the tokenized script, or an "end
        <handlername>" statement.
	
	TAKES:
		iterator -			An iterator to the next token to parse.
		vCode -				A reference to the handler object we are to add to.
		vNameOfHandler -	Name of the handler we are parsing.
		vTemporaries -		A list of indices to all temporary variables.
		vUserVarNames -		A list of all names of user variables plus their
							indices.
		vInternals -		A list of indices to all internal variables.
		vCurrLineNum -		A reference to a variable in which we keep track of
							what line we are currently at.
	
	REVISIONS:
        2001-09-27	UK		Modified to support "then" on next line for "if".
		2001-01-18	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::ParseHandlerBody( TalkHandler& vCode, TextMunger& vNameOfHandler,
										HTLIterator &iterator,
										TalkTempVarList& vTemporaries,
										TalkVarNameList& vUserVarNames,
										TalkInternalVarList& vInternals,
										long& vCurrLineNum )
{
	HyperToken*				vToken;
	TalkMemLocation			vDestination;
	HyperInstruction		vThisInstr;
	stack<IfStackEntry>		vIfStack;		// Stack used for parsing ifs.
	stack<RepeatStackEntry>	vRepeatStack;	// Stack used for parsing repeats.
	RepeatStackEntry		vRSEntry;
	IfStackEntry			vISEntry;
	long					vParamCount;
	HyperToken				vDummyNewline( mTokens.back()->GetStartChar(), mTokens.back()->GetEndChar(), TOKEN_TYPE_NEWLINE );	// Used in case script doesn't end on a newline to fake one.
	
	for( ; iterator != mTokens.end(); ++iterator )
	{
		vToken = (*iterator);
		
		vTemporaries.ResetUnusedPointer();	// New command: We can re-use our temporaries.
		
		switch( vToken->GetType() )
		{
			// Built-in command "put":
			case TOKEN_TYPE_PUT_COMMAND:
				ParsePutCommand( iterator, vCode, vTemporaries, vUserVarNames, vInternals );
				break;
			
			// Built-in qualifier for declaring local variables:
			case TOKEN_TYPE_LOCAL_COMMAND:
				vDestination = ParseVariableToken( ++iterator, vCode, vTemporaries, vUserVarNames, vInternals, true );
				if( vDestination.mLocation == MEM_INDEX_ACCU )
					throw runtime_error( "Expected valid variable name after \"local\" qualifier." );
				break;
			
			// Line starts with string? Just take this as being "put":
			case TOKEN_TYPE_STRING:
				ParsePutCommand( --iterator, vCode, vTemporaries, vUserVarNames, vInternals );
				break;
				
			case TOKEN_TYPE_GLOBAL_IDENTIFIER:
				++iterator;
				vToken = (*iterator);
				
				switch( vToken->GetType() )
				{
					case TOKEN_TYPE_STRING:		// Could work around this, but we don't want people to wonder why "strings" work while variables don't.
					case TOKEN_TYPE_INTEGER:
					case TOKEN_TYPE_OPERATOR:
					case TOKEN_TYPE_NEWLINE:
						throw parse_error( "Expected identifier after \"global\".", vToken->GetStartChar(), vToken->GetEndChar() );
						break;
				}
				
				vThisInstr.mCaller = this;
				vThisInstr.mInstructionProc = HyperDeclareGlobalInstruction;
				vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_IMMEDIATE;
				vThisInstr.mParam[0].mValue = new TalkVarValue( new TextMunger( (char*) vToken->GetValue() ) );
				vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
				vThisInstr.mResult.mLocation = vUserVarNames.GetVarByName( (char*) vToken->GetValue(), true );	// Get variable that'll become ref to this.
				
				vCode.push_back( vThisInstr );	// Never forget this, or the above is almost useless :-/
				vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
				break;
			
			case TOKEN_TYPE_IF_IDENTIFIER:
				ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals );
				vThisInstr.mCaller = this;
				vThisInstr.mInstructionProc = HyperIfInstruction;
				vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;
				vThisInstr.mParam[0].mLocation = MEM_INDEX_ACCU;	// Argument is in Accu.
				//vThisInstr.mResult = 0;	// Unused.
				
				vCode.push_back( vThisInstr );	// Add if instruction.
				vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
				
				++iterator;
				vToken = (*iterator);
				
                // Support "then" on next line:
                while( iterator != mTokens.end()
                    && vToken->GetType() == TOKEN_TYPE_NEWLINE )	// Line break after condition?
                {
                    ++iterator;		// Just skip the newline. "then" is allowed to be on next line. If it isn't, the code below will complain.
                    if( iterator != mTokens.end() )
                        vToken = (*iterator);
                }
                // Now make sure user wrote a "then":
				if( iterator == mTokens.end()
                    || vToken->GetType() != TOKEN_TYPE_THEN_IDENTIFIER )
					throw parse_error( "Expected \"then\" after \"if\".", vToken->GetStartChar(), vToken->GetEndChar() );
				
				// Prepare address of if's dest field for storing on "if" stack:
				vISEntry.mIfDestField = &vCode.back().mParam[1];
				vISEntry.mIfNotElse = true;
				
				// Now check whether this is a one-line "if":
				++iterator;
				if( (*iterator)->GetType() != TOKEN_TYPE_NEWLINE )	// No return! Command name of one-line "if"!
				{
					vISEntry.mOneLineIf = true;
					vIfStack.push( vISEntry );	// Remember RHS so we can fill in jump address later.
					--iterator;	// Backtrack to make sure command name can be parsed.
					continue;	// Skip line break processing below as there is no break.
				}
				else
				{
					vISEntry.mOneLineIf = false;
					vIfStack.push( vISEntry );	// Remember RHS so we can fill in jump address later.
					--iterator;	// Was a return? Backtrack and let the code below handle line break.
				}
				break;
			
			case TOKEN_TYPE_ELSE_IDENTIFIER:
				/* Before the instructions of the "else" start we add a GOTO instruction that
					will be executed at the end of the "if" to jump over the instructions of our
					"else":  */
				vThisInstr.mCaller = this;
				vThisInstr.mInstructionProc = HyperIfInstruction;
				vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_LONG;
				vThisInstr.mParam[0].mLong = true;
				//vThisInstr.mResult = 0;	// Unused.
				
				vCode.push_back( vThisInstr );	// Add goto instruction.
				vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
				
				// Have "if" know where to jump if it evaluates false:
				(*vIfStack.top().mIfDestField).mType = MEM_LOCATION_TYPE_LONG;
				(*vIfStack.top().mIfDestField).mLong = vCode.size() -1;	// Right behind our goto.
				if( vIfStack.top().mIfNotElse == false )	// Last one was an "else"? Huh?!
					throw runtime_error( "Missing \"end if\" after previous \"else\" or missing \"if\" before this \"else\"." );
				vIfStack.pop();	// Remove if's RHS from stack.
				
				vISEntry.mIfDestField = &vCode.back().mParam[1];	// remember address where "end if" is to insert jump address later.
				vISEntry.mIfNotElse = false;
				
				// Check whether this is a one-line else?
				++iterator;
				if( (*iterator)->GetType() != TOKEN_TYPE_NEWLINE )	// Yes! Not a line break! Name of command after then!
				{
					--iterator;	// Make sure command gets an opportunity to be parsed.
					vISEntry.mOneLineIf = true;
					vIfStack.push( vISEntry );	// Now push pointer to our goto's jump destination on the "if" stack:
					continue;	// Skip line break processing after this "else" as there is no break between this and the command.
				}
				else	// No, multi-line:
				{
					--iterator;	// Make sure line break is accounted for below.
					vISEntry.mOneLineIf = false;
					vIfStack.push( vISEntry );	// Now push pointer to our goto's jump destination on the "if" stack:
				}
				break;
			
			case TOKEN_TYPE_REPEAT_IDENTIFIER:
				++iterator;
				vToken = (*iterator);
				switch( vToken->GetType() )
				{
					case TOKEN_TYPE_WHILE_IDENTIFIER:	// Repeat while <condition>
						vRSEntry.mJumpBackAddress = vCode.size() -1;	// Remember place repeat starts so we can jump there at end of commands.
						vRSEntry.mStepWidth = 0;	// Identify this as counter-less repeat.
						
						ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals );
						
						vThisInstr.mCaller = this;
						vThisInstr.mInstructionProc = HyperIfInstruction;
						vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;
						vThisInstr.mParam[0].mLocation = MEM_INDEX_ACCU;	// Argument is in Accu.
						//vThisInstr.mResult = 0;	// Unused.
						
						vCode.push_back( vThisInstr );	// Add if instruction.
						vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
						
						vRSEntry.mIfDestField = &vCode.back().mParam[1];	// Remember where we are to stuff end of repeat.
						
						vRepeatStack.push( vRSEntry );	// Remember RHS so we can fill in jump address later.
						break;
					
					case TOKEN_TYPE_WITH_IDENTIFIER:	// Repeat with x = <start> [down] to <end>
					{
                    	char*				vLoopCounterVarName;
						ValStackLocation	vStartNum,
											vStopNum,
											vCounter,
											vCondition;
						unsigned long		vLoopSteps = 1;	// Positive for up, negative for down, value is step width.
						HyperInstrProcPtr	vIncrementProc = HyperAddInstruction;	// Instr. used for changing loop counter.
						
						++iterator;
						vToken = (*iterator);
						switch( vToken->GetType() )
						{
							case TOKEN_TYPE_INTEGER:
							case TOKEN_TYPE_OPERATOR:
							case TOKEN_TYPE_STRING:
							case TOKEN_TYPE_NEWLINE:
								throw parse_error( "Expected variable name after \"repeat with\".", vToken->GetStartChar(), vToken->GetEndChar() );
								break;
						}
						
						vLoopCounterVarName = (char*) vToken->GetValue();	// Remember name of variable to use as counter.
						
						// ... is / = ...
						++iterator;
						vToken = (*iterator);
						if( TalkTokenizer::GetOperatorTokenKind( vToken ) != OPERATOR_CHAR_EQUAL )
							throw parse_error( "Expected \"is\" or \"=\" after \"repeat with <varName>\".", vToken->GetStartChar(), vToken->GetEndChar() );
						
						// ... <startNum> ...
						vStartNum = vInternals.GetNewVariable();	// FIX ME! "end repeat" needs to release this!
						ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals, vStartNum );
						
						// ... [down] ...
						++iterator;
						vToken = (*iterator);
						if( vToken->GetType() == TOKEN_TYPE_DOWN_IDENTIFIER )
						{
							vIncrementProc = HyperSubInstruction;	// Decrement instead of incrementing, we're to go backwards.
							vLoopSteps *= -1;
							
							++iterator;
							vToken = (*iterator);
						}
						// ... to ...
						if( vToken->GetType() != TOKEN_TYPE_TO_IDENTIFIER )
							throw parse_error( "Expected \"to\" after \"repeat with <varName> = <start> [down]\".", vToken->GetStartChar(), vToken->GetEndChar() );
						
						// ... <endNum>
						vStopNum = vInternals.GetNewVariable();	// FIX ME! "end repeat" needs to release this!
						ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals, vStopNum );
						
						// Now init loop counter:
						vCounter = vUserVarNames.GetVarByName( vLoopCounterVarName, true );
						
						vThisInstr.mCaller = this;
						vThisInstr.mInstructionProc = HyperPutInstruction;
						vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;
						vThisInstr.mParam[0].mLocation = vStartNum;
						vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
						vThisInstr.mResult.mLocation = vCounter;
						
						vCode.push_back( vThisInstr );	// Add instruction.
						vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
						
						// Remember place repeat starts so we can jump there at end of commands:
						vRSEntry.mJumpBackAddress = vCode.size() -1;
						
						/* Add compare instruction for loop condition:
							If we're looping up (stepWidth > 0) we do counter <= stopNum, if we
							are looping down (stepWidth < 0) we do counter >= startNum. */
						vThisInstr.mCaller = this;
						vThisInstr.mInstructionProc = (vLoopSteps > 0) ? HyperLessThanEqualInstruction : HyperGreaterThanEqualInstruction;
						vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;
						vThisInstr.mParam[0].mLocation = vCounter;
						vThisInstr.mParam[1].mType = MEM_LOCATION_TYPE_STACK;
						vThisInstr.mParam[1].mLocation = (vLoopSteps > 0) ? vStopNum : vStartNum;
						vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
						vThisInstr.mResult.mLocation = vCondition = vTemporaries.GetNewTemp();
						
						vCode.push_back( vThisInstr );	// Add compare instruction.
						vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
						
						/* Add "if" instruction:
							If the condition is TRUE, this will simply execute the commands once more, otherwise
							it will jump to the location in mParam[1], which will be filled out by the "end repeat"
							statement. */
						vThisInstr.mCaller = this;
						vThisInstr.mInstructionProc = HyperIfInstruction;
						vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;
						vThisInstr.mParam[0].mLocation = vCondition;	// Condition's value.
						//vThisInstr.mResult = 0;	// Unused.
						
						vCode.push_back( vThisInstr );	// Add if instruction.
						vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
						
						vRSEntry.mIfDestField = &vCode.back().mParam[1];	// Remember where we are to stuff end of repeat.
						vRSEntry.mStepWidth = vLoopSteps;	// Remember what steps to use when incrementing.
						vRSEntry.mCounter = vCounter;		// Remember our counter variable's location so we can increment it.
						// Note: the jump back address of this struct has been set up above the compare instruction.
						vRepeatStack.push( vRSEntry );	// Remember RHS so we can fill in jump address later.
						
						// FIX ME! Maybe add "in steps of <number>" parameter that defaults to 1 ?
						// FIX ME! Add another variable to be used as an internal counter.
						break;
                    }
					
					default:
						throw parse_error( "Expected \"while\" or \"with\" after \"repeat\".", vToken->GetStartChar(), vToken->GetEndChar() );
				}
				break;
			
			// Handler call:
			case TOKEN_TYPE_IDENTIFIER:
            {
				vParamCount = 0;
				
				TextMunger*	vHandlerName;
				HyperToken	vEndToken( 0, 0, TOKEN_TYPE_OPERATOR, OPERATOR_CHAR_COMMA );
				char*		vHandlerToCall = (char*) vToken->GetValue();
				
				++iterator;
				vToken = (*iterator);
				
				// Push params onto the stack:
				if( vToken->GetType() != TOKEN_TYPE_NEWLINE )	// We have params.
				{
					iterator--;	// Make sure first token of 1st param gets parsed.
					
					do
					{
						ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals,
										MEM_INDEX_ACCU, &vEndToken );
						
						vThisInstr.mCaller = this;
						vThisInstr.mInstructionProc = HyperPushParamInstruction;
						vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;
						vThisInstr.mParam[0].mLocation = MEM_INDEX_ACCU;	// Argument is in Accu.
						//vThisInstr.mResult = 0;	// Unused.
						
						vParamCount++;	// Increase param count.
					
						vCode.push_back( vThisInstr );	// Append instruction to our code.
						vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
						
						iterator++;
						vToken = (*iterator);
					}
					while( TalkTokenizer::GetOperatorTokenKind( vToken ) == OPERATOR_CHAR_COMMA );
				}
				
				iterator--;	// Make sure newline gets a chance to be parsed.
				
				// Now push param count onto stack:
				vThisInstr.mCaller = this;
				vThisInstr.mInstructionProc = HyperPushParamInstruction;
				vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_LONG;
				vThisInstr.mParam[0].mLong = vParamCount;	// Param count as LHS.
				//vThisInstr.mResult = 0;	// Unused.
			
				vCode.push_back( vThisInstr );	// Append instruction to our code.
				vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
				
				vHandlerName = new TextMunger( vHandlerToCall );
				if( sOptions[OPT_DISTINGUISH_FCN_CMD] == true )	// Perform name mangling to distinguish functions & commands.
				{
					vHandlerName->SetOffset(0);
					vHandlerName->Insert<char>( ' ' );	// Space is impossible to enter as part of a function name.
				}
				
				vThisInstr.mInstructionProc = HyperJSRInstruction;
				vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_IMMEDIATE;
				vThisInstr.mParam[0].mValue = new TalkVarValue( vHandlerName );	// Value takes over the text munger.
				vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
				vThisInstr.mResult.mLocation = vUserVarNames.GetVarByName( "the result", true );
				
				vCode.push_back( vThisInstr );	// Add JSR instruction.
				vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
				break;
            }
			
			case TOKEN_TYPE_NEWLINE:	// Empty line, just skip.
				iterator--;	// Let code below process this.
				break;
			
			// End of this handler?
			case TOKEN_TYPE_END:
				++iterator;
				
				if( iterator == mTokens.end() )
					throw parse_error( "Don't know what to end here.", vToken->GetStartChar(), vToken->GetEndChar() );
				
				vToken = (*iterator);
				if( vToken->GetType() == TOKEN_TYPE_REPEAT_IDENTIFIER )		// End Repeat:
				{
					if( vRepeatStack.size() == 0 )	// Don't have any more repeats that need an "end" ?
						throw parse_error( "Too many \"end\"s for too few \"repeat\"s", vToken->GetStartChar(), vToken->GetEndChar() );
					
					// Add counter-increment instruction if this loop has a counter:
					if( vRepeatStack.top().mStepWidth != 0 )	// Have a counter?
					{
						vThisInstr.mCaller = this;
						vThisInstr.mInstructionProc = HyperAddInstruction;				// Add.
						vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_LONG;
						vThisInstr.mParam[0].mLong = vRepeatStack.top().mStepWidth;		// Step width (positive or negative).
						vThisInstr.mParam[1].mType = MEM_LOCATION_TYPE_STACK;
						vThisInstr.mParam[1].mLocation = vRepeatStack.top().mCounter;	// To counter's value.
						vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
						vThisInstr.mResult.mLocation = vRepeatStack.top().mCounter;		// And store result in counter.
						
						vCode.push_back( vThisInstr );	// Add the instruction.
						vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
					}
					
					// Add jump-back ("goto") instruction at end of loop:
					vThisInstr.mCaller = this;
					vThisInstr.mInstructionProc = HyperIfInstruction;	// Goto.
					vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_LONG;
					vThisInstr.mParam[0].mLong = 1;	// Condition is hard-wired to TRUE, i.e. "always jump"
					vThisInstr.mParam[1].mType = MEM_LOCATION_TYPE_LONG;
					vThisInstr.mParam[1].mLong = vRepeatStack.top().mJumpBackAddress -vCode.GetInitCodeSize();	// We jump back to our "if", address is relative to end of init code block.
					//vThisInstr.mResult = 0;	// Unused.
					
					vCode.push_back( vThisInstr );	// Add if instruction.
					vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
					
					// Have "repeat" know to jump after this on exit:
					(*vRepeatStack.top().mIfDestField).mType = MEM_LOCATION_TYPE_LONG;
					(*vRepeatStack.top().mIfDestField).mLong = vCode.size() -1;
					vRepeatStack.pop();	// Remove repeat's entry from stack.
				}
				else if( vToken->GetType() == TOKEN_TYPE_IF_IDENTIFIER )		// End If:
				{
					ParserHandleEndIf( iterator, vCode, vIfStack );
					
					/*if( vIfStack.size() == 0 )	// Don't have any ifs that still need an end?
						throw parse_error( "Too many \"end\"s for too few \"if\"s", vToken->GetStartChar(), vToken->GetEndChar() );
					
					(*vIfStack.top().mIfDestField).mType = MEM_LOCATION_TYPE_LONG;
					(*vIfStack.top().mIfDestField).mLong = vCode.size() -1 -vCode.GetInitCodeSize();	// Tell "if" to jump down here, address is relative to end of init-code block.
					vIfStack.pop();	// Remove if's or else's RHS from stack.*/
				}
				else	// End handler:
				{
					switch( vToken->GetType() )
					{
						case TOKEN_TYPE_NEWLINE:
						case TOKEN_TYPE_STRING:
						case TOKEN_TYPE_INTEGER:
						case TOKEN_TYPE_OPERATOR:
							throw parse_error( "Don't know what to end here.", vToken->GetStartChar(), vToken->GetEndChar() );
							break;
						
						default:
							break;// Is an identifier? Great! Use it as handler name.
					}
					
					char*		vThisHdName;
					
					vThisHdName = (char*) vToken->GetValue();
					
					if( !vNameOfHandler.EqualsString( vThisHdName, strlen(vThisHdName) ) )
						throw parse_error( "Expected handler name after this \"end\".", vToken->GetStartChar(), vToken->GetEndChar() );
					
					if( vRepeatStack.size() > 0 )	// Still have repeat waiting to get a location to jump back to?
						throw parse_error( "Missing \"end repeat\".", vToken->GetStartChar(), vToken->GetEndChar() );
					
					if( vIfStack.size() > 0 )	// Still have if/else waiting to get a jump destination?
						throw parse_error( "Missing \"end if\".", vToken->GetStartChar(), vToken->GetEndChar() );
					
					return;		// We're finished with this handler.
				}
				break;
			
			default:
				ParseBuiltInCommand( iterator, vCode, vTemporaries, vUserVarNames, vInternals );
		}
		
		/* WARNING:
			The stuff here is skipped by e.g one-line "if"s and "else"s using "continue". Keep that
			in mind when adding code here. */
		
		// Exit if parsing exceeded number of commands to prevent iterator from being increased beyond end:
		if( iterator == mTokens.end() )
			return;
		
		// Get next token: Should be end of line.
		++iterator;
		
		// No tokens left?
		if( iterator == mTokens.end() )
		{
			if( vIfStack.size() == 0 )	// No open "if"s anymore?
				return;	// Just stop parsing.
			else	// Otherwise, give the code another chance at handling any open one-line "if"/"else"s.
				vToken = &vDummyNewline;	// Make sure theres still a token to work with.
		}
		else
			vToken = (*iterator);
		
		// Check whether we're working on a one-line "if"/"else", in which case the "if"/"else" block ends here:
		if( vIfStack.size() != 0 )
		{
			if( vIfStack.top().mOneLineIf == true )	// We had a one-line if/else?
			{
				if( vIfStack.top().mIfNotElse == true )	// It is an "if"? Check for following "else" before doing "end if".
				{
					if( vToken->GetType() == TOKEN_TYPE_ELSE_IDENTIFIER )	// Oh, it's an "else" immediately after an "if"'s command!
						iterator--;	// Step back so else can be parsed and gracefully allow missing line break.
					else if( vToken->GetType() != TOKEN_TYPE_NEWLINE )	// Must have either line break or else here.
						throw parse_error( "Expected line break or \"else\" here.", vToken->GetStartChar(), vToken->GetEndChar() );
					
					++iterator;
					if( (*iterator)->GetType() == TOKEN_TYPE_ELSE_IDENTIFIER )	// It's an "else"? Then let \"else\" care of ending this block.
						--iterator;	// Make sure "else" can be parsed.
					else	// No "else" after this? Then we have to end the block:
					{
						--iterator;	// Make sure this token can be parsed.
						ParserHandleEndIf( iterator, vCode, vIfStack );
					}
					
					if( vToken->GetType() == TOKEN_TYPE_NEWLINE )	// vToken is still the line break or "else" from the start of this block.
						ParserMakeLineEndInstruction( vCurrLineNum, vCode, iterator, vIfStack, vRepeatStack );	// Add line break code now that we had the chance to fake our "end if".
				}
				else	// It is a one-line "else"?
					ParserHandleEndIf( iterator, vCode, vIfStack );	// Then we must end its block here.
			}
			else	// Not a one-line "if"/"else"? Then just handle it like any other line break:
			{
				if( vToken->GetType() != TOKEN_TYPE_NEWLINE )
					throw parse_error( "Identifiers found after end of line.", vToken->GetStartChar(), vToken->GetEndChar() );
				ParserMakeLineEndInstruction( vCurrLineNum, vCode,iterator, vIfStack, vRepeatStack );	// Make sure it's handled.
			}
		}
		else	// No "if"s pending? Then this line _must_ end with a line break:
		{
			if( vToken->GetType() != TOKEN_TYPE_NEWLINE )
				throw parse_error( "Identifiers found after end of line.", vToken->GetStartChar(), vToken->GetEndChar() );
			
			ParserMakeLineEndInstruction( vCurrLineNum, vCode,iterator, vIfStack, vRepeatStack );
		}
	}
}


/* --------------------------------------------------------------------------------
	ParserMakeLineEndInstruction:
		Add a "new line" instruction to the code. This is called whenever a line
		has ended to provide a hook for a debugger as well as to increase the
		line number counter used in error reporting.
	
	TAKES:
		vCurrLineNum	-	Line number counter used for error reporting. This
							call takes care of adding 1 to the counter.
		vCode			-	The current handler to which to append the "new line"
							instruction.
		iterator		-	Iterator into the token stream that is being parsed.
		ifStack			-	The "if" stack keeping track of nested "if"s.
		repStack		-	The "repeat" stack keeping track of nested repeats.
		
	REVISIONS:
		2001-09-28	UK		Extracted from ParseHandlerBody().
   ----------------------------------------------------------------------------- */

void	HyperTalk::ParserMakeLineEndInstruction( long& vCurrLineNum, TalkHandler& vCode,
												HTLIterator& iterator,
												stack<IfStackEntry>& ifStack,
												stack<RepeatStackEntry>& repStack )
{
	HyperInstruction		vThisInstr;
	
	// Now generate debugging aid instruction that denotes ends of user's lines:
	vThisInstr.mCaller = this;
	vThisInstr.mInstructionProc = HyperDebuggerLineInstruction;
	vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_LONG;
	vThisInstr.mParam[0].mLong = vCurrLineNum++;	// Increase line counter!
	vCode.push_back( vThisInstr );	// Add line instruction.
	
	vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
	
	// End of script?
	if( iterator == mTokens.end() )
	{
		HyperToken*		vToken = (*(--iterator));
		
		if( ifStack.size() != 0 )
			throw parse_error( "Error: Missing \"end if\" at end of script.", vToken->GetStartChar(), vToken->GetEndChar() );
		
		if( repStack.size() != 0 )
			throw parse_error( "Error: Missing \"end repeat\" at end of script.", vToken->GetStartChar(), vToken->GetEndChar() );
	
		++iterator;	// balance "--" above so loop won't endlessly parse the last token.
	}
}


/* --------------------------------------------------------------------------------
	ParserHandleEndIf:
		Finish an "if" or "else" block. Called by "if" as well as at the end
		of a line if we're processing a one-line-if.
	
	TAKES:
		currToken	-	Iterator pointing at current token (used only for
						reporting error offset.
		vIfStack	-	The "if"-stack used to keep track of nested "if"s and
						"else"s.
		
	REVISIONS:
		2001-09-28	UK		Extracted from ParseHandlerBody().
   ----------------------------------------------------------------------------- */

void	HyperTalk::ParserHandleEndIf( HTLIterator& currToken, TalkHandler& vCode,
										stack<IfStackEntry> &vIfStack )
{
	if( vIfStack.size() == 0 )	// Don't have any ifs that still need an end?
		throw parse_error( "Too many \"end\"s for too few \"if\"s", (*currToken)->GetStartChar(), (*currToken)->GetEndChar() );
	
	(*vIfStack.top().mIfDestField).mType = MEM_LOCATION_TYPE_LONG;
	(*vIfStack.top().mIfDestField).mLong = vCode.size() -1 -vCode.GetInitCodeSize();	// Tell "if" to jump down here, address is relative to end of init-code block.
	vIfStack.pop();	// Remove if's or else's RHS from stack.
}


/* --------------------------------------------------------------------------------
	ParseHandler:
		Parse the script that was tokenized before for one handler or function.
	
	TAKES:
		iterator	-	An iterator into the current script's token stream which
						we use for parsing. It will be moved past the tokens we
						parsed.
		isFcn		-	Boolean whether this is a function being parsed or a
						message. Implementation-wise they are the same, but
						functions are used in expressions while messages are used
						as commands.
		vCurrLineNum	-	Line number this handler starts at.
		
	REVISIONS:
		2001-02-13	UK		Added script-wide line counting.
		2000-10-22	UK		Added "return" command.
		1999-11-27	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::ParseHandler( HTLIterator &iterator, bool isFcn, long& vCurrLineNum )
{
	HyperToken*				vToken;
	unsigned long			vVariables = 1;	// # of variables. 1 is accu.
	TalkTempVarList			vTemporaries( vVariables );		// List of temporary vars.
	TalkInternalVarList		vInternals( vVariables );		// List of internal vars.
	TalkVarNameList			vUserVarNames( vVariables );	// Var name -> stack index mappings.
	TalkMemLocation			vDestination;
	HyperInstruction		vThisInstr;
	unsigned long			vParamCount;
	
	++iterator;
	vToken = (*iterator);
	
	switch( vToken->GetType() )
	{
		case TOKEN_TYPE_NEWLINE:
		case TOKEN_TYPE_STRING:
		case TOKEN_TYPE_INTEGER:
		case TOKEN_TYPE_OPERATOR:
			throw parse_error( "Expected identifier specifying handler name.", vToken->GetStartChar(), vToken->GetEndChar() );
			break;
		
		default:
			break;// Is an identifier? Great! Use it as handler name.
	}
	
	TextMunger		vNameOfHandler( (char*) vToken->GetValue() );	// Get token value to use as handler name. Belongs to token!
	++iterator;
	
	if( sOptions[OPT_DISTINGUISH_FCN_CMD] == true )	// Perform name mangling to distinguish functions & commands.
	{
		vNameOfHandler.SetOffset(0);
		if( isFcn )
			vNameOfHandler.Insert<char>( '\t' );	// Tab is impossible to enter as part of a function name.
		else
			vNameOfHandler.Insert<char>( ' ' );		// Space is impossible to enter as part of a command name.
	}
	
	if( mScript.find( vNameOfHandler ) != mScript.end() )
		throw parse_error( "Handler declared twice.", vToken->GetStartChar(), vToken->GetEndChar() );
	
	TalkHandler&	vCode = mScript[ vNameOfHandler ];	// Create handler and get its instruction list.
	
	vCode.SetIsFunction( isFcn );	// Make handler remember it's fcn or message.
	
	vToken = (*iterator);

	// Look for parameters:
	if( vToken->GetType() != TOKEN_TYPE_NEWLINE )	// We have params.
	{
		iterator--;	// Make sure first token of 1st param gets parsed.
		
		vParamCount = 0;
		
		do
		{
			vDestination = ParseValue( iterator, vCode, vTemporaries, vUserVarNames, vInternals, true );
			
			if( vDestination.mType != MEM_LOCATION_TYPE_STACK )
				throw parse_error( "Expected variable name here.", vToken->GetStartChar(), vToken->GetEndChar() );
			
			vThisInstr.mCaller = this;
			vThisInstr.mInstructionProc = HyperGetParamInstruction;
			vThisInstr.mParamCount = 1;
			vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_LONG;
			vThisInstr.mParam[0].mLong = ++vParamCount;
			vThisInstr.mResult = vDestination;
		
			vCode.push_back( vThisInstr );	// Append instruction to our code.
			
			vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
			
			iterator++;
			vToken = (*iterator);
		}
		while( TalkTokenizer::GetOperatorTokenKind( vToken ) == OPERATOR_CHAR_COMMA );
		
		if( vToken->GetType() != TOKEN_TYPE_NEWLINE )
			throw parse_error( "Expected comma or return here.", vToken->GetStartChar(), vToken->GetEndChar() );
	}
	iterator++;	// Move past this newline.
	vCurrLineNum++;	// Make sure line counter reflects this.
	
	ParseHandlerBody( vCode, vNameOfHandler, iterator, vTemporaries, vUserVarNames,
						vInternals, vCurrLineNum );
	
	vCode.SetStackNeed( vVariables );
}


/* --------------------------------------------------------------------------------
	ParseAsHandler:
		Parse the tokenized script that this instance contains as if it were the
		contents of a command handler of specified name.
		
		This can be used to implement places where code occurs outside a handler,
		like the "do" command or the message box.
	
	TAKES:
		inName	-	Name of handler to generate.
		
	REVISIONS:
		2001-01-18	UK		Created based on ParseHandler.
   ----------------------------------------------------------------------------- */

void	HyperTalk::ParseAsHandler( const TextMunger& inName )
{
	HyperToken*				vToken;
	unsigned long			vVariables = 1;	// # of variables. 1 is accu.
	TalkTempVarList			vTemporaries( vVariables );		// List of temporary vars.
	TalkInternalVarList		vInternals( vVariables );		// List of internal vars.
	TalkVarNameList			vUserVarNames( vVariables );	// Var name -> stack index mappings.
	HyperInstruction		vThisInstr;
	long					vCurrLineNum = 1;
	HTLIterator				iterator;
	TextMunger				vNameOfHandler( inName );
	
	if( sOptions[OPT_DISTINGUISH_FCN_CMD] == true )	// Perform name mangling to distinguish functions & commands.
	{
		vNameOfHandler.SetOffset(0);
		vNameOfHandler.Insert<char>( ' ' );		// Space is impossible to enter as part of a command name.
	}
	
	if( mScript.find( vNameOfHandler ) != mScript.end() )
		throw parse_error( "Handler declared twice.", 0, 0 );
	
	TalkHandler&	vCode = mScript[ vNameOfHandler ];	// Create handler and get its instruction list.
	
	vCode.SetIsFunction( false );	// Make handler remember it's fcn or message.
	
	iterator = mTokens.begin();
	vToken = (*iterator);

	ParseHandlerBody( vCode, vNameOfHandler, iterator, vTemporaries, vUserVarNames,
						vInternals, vCurrLineNum );
	
	vCode.SetStackNeed( vVariables );
}


/* --------------------------------------------------------------------------------
	ParsePutCommand:
		Parse the script that was tokenized before for a put command plus
		parameters.
	
	TAKES:
		iterator -		An iterator to the next token to parse.
		vTemporaries -	A list of indices to all temporary variables.
		vUserVarNames -	A list of all names of user variables plus their indices.
		vInternals -	A list of indices to all internal variables.
		doCreate -		True if we need this value to be a destination. This will
						create variables instead of turning them into literals,
						however it may still return immediates if the code
						explicitly specified them.
	
	REVISIONS:
		2000-11-07	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::ParsePutCommand( HTLIterator &iterator, TalkHandler& vCode,
									TalkTempVarList& vTemporaries,
									TalkVarNameList& vUserVarNames,
									TalkInternalVarList& vInternals )
{
	HyperToken*				vToken;
	HyperInstruction		vThisInstr;
	TalkMemLocation			vDestination;
	
	// put *what* ???
	if( iterator == mTokens.end() )
	{
		--iterator;
		throw parse_error( "Expected value, found EOF.", (*iterator)->GetEndChar(), (*iterator)->GetEndChar() );
	}
	
	ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals );
	
	// put <value> *into* ???
	bool haveInto = false;
	
	if( iterator != mTokens.end() )
	{
		++iterator;
		vToken = (*iterator);
		if( vToken->GetType() != TOKEN_TYPE_INTO_IDENTIFIER
			&& vToken->GetType() != TOKEN_TYPE_NEWLINE )
			throw parse_error( "Expected \"into\" or end of line here.", vToken->GetStartChar(), vToken->GetEndChar() );
		haveInto = vToken->GetType() == TOKEN_TYPE_INTO_IDENTIFIER;
	}
	
	if( haveInto )
	{
		// put <value> into *what*???
		if( iterator == mTokens.end() )
			throw parse_error( "Expected container, found EOF.", vToken->GetStartChar(), vToken->GetEndChar() );
		
		vDestination = ParseValue( iterator, vCode, vTemporaries, vUserVarNames, vInternals, true );
		if( vDestination.mType == MEM_LOCATION_TYPE_IMMEDIATE )
			throw parse_error( "Expected container here.", vToken->GetStartChar(), vToken->GetEndChar() );
		
		// Create "put" instruction:
		vThisInstr.mCaller = this;
		vThisInstr.mInstructionProc = HyperPutInstruction;
		vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;
		vThisInstr.mParam[0].mLocation = MEM_INDEX_ACCU;	// Accu contains our argument now.
		vThisInstr.mResult = vDestination;	// Destination as result address.
		
		vCode.push_back( vThisInstr );	// Append instruction to our code.
		vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
	}
	else	// "put" w/o destination means "output to console":
	{
		if( (iterator != mTokens.end())
			&& (vToken->GetType() != TOKEN_TYPE_NEWLINE) )
			throw parse_error( "Expected \"into\" here.", vToken->GetStartChar(), vToken->GetEndChar() );
		
		if( iterator != mTokens.end() )
			iterator--;	// Make sure newline is parsed again below.
		
		// Generate "output" instruction:
		vThisInstr.mCaller = this;
		vThisInstr.mInstructionProc = HyperOutputInstruction;
		vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;
		vThisInstr.mParam[0].mLocation = MEM_INDEX_ACCU;	// Accu contains our argument now.
		//vThisInstr.mResult = 0;	// Unused.
		
		vCode.push_back( vThisInstr );	// Append instruction to our code.
		vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
	}
}


/* --------------------------------------------------------------------------------
	ParseBuiltInCommand:
		Parse the script that was tokenized before for one of the built-in commands
		that have been registered with the parser and generate an appropriate
		instruction.
	
	TAKES:
		iterator -		An iterator to the next token to parse.
		vTemporaries -	A list of indices to all temporary variables.
		vUserVarNames -	A list of all names of user variables plus their indices.
		vInternals -	A list of indices to all internal variables.
		doCreate -		True if we need this value to be a destination. This will
						create variables instead of turning them into literals,
						however it may still return immediates if the code
						explicitly specified them.
	
	REVISIONS:
		2000-11-07	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::ParseBuiltInCommand( HTLIterator &iterator, TalkHandler& vCode,
										TalkTempVarList& vTemporaries,
										TalkVarNameList& vUserVarNames,
										TalkInternalVarList& vInternals )
{
	HyperToken*						vToken;
	HyperInstruction				vThisInstr;
	TalkMemLocation*				vDestination;
	TalkCommandEntry*				vFoundCommand;
	short							x,
									vDestParam;
	TalkBuiltInCmdList::iterator	tbici;
	bool							vParamsSet[TALK_INSTR_MAX_PARAMS+1];
	
	/* Clear array. 0 is result, 1 to TALK_INSTR_MAX_PARAMS are params.
		The array is used to keep track of which params we did already set. */
	for( x = 0; x <= TALK_INSTR_MAX_PARAMS; x++ )
		vParamsSet[x] = false;
	
	vToken = *iterator;
	
	tbici = sBuiltInCommands.find( vToken->GetType() );
	if( tbici == sBuiltInCommands.end() )	// none found?
		throw parse_error( "Unknown command or handler name.", vToken->GetStartChar(), vToken->GetEndChar() );
	
	vFoundCommand = tbici->second;
	vThisInstr.mInstructionProc = vFoundCommand->mParam[0].mCommandInstruction;
	vThisInstr.SetRefCon( vFoundCommand->mRefCon );	// Copy over refCon from command entry.
	
	--iterator;	// Step back one so we can get the command name & direct param with the same code used for params.
	
	// Parse the command entry for our command and its params:
	for( x = 0; x < vFoundCommand->mParamCount; x++ )
	{
		vDestParam = vFoundCommand->mParam[x].mDestParam;
		if( vFoundCommand->mParam[x].mParamType != TALK_PARAM_TYPE_INVALID )	// We're not supposed to ignore this param?
		{
			switch( vDestParam )
			{
				case TALK_PARAM_RESULT:
					vDestination = &vThisInstr.mResult;
					break;
				
				case TALK_PARAM_IT:
					throw logic_error( "Invalid internal command definition." );
					break;
				
				default:
					if( vParamsSet[vDestParam] == false )	// Param not already specified?
					{
						vParamsSet[vDestParam] = true;
						vDestination = &vThisInstr.mParam[vDestParam-1];
						vThisInstr.mParamCount = max<short>(vDestParam, vThisInstr.mParamCount);
					}
					else	// It is? Just skip this one.
						continue;
			}
		}
		
		++iterator;
		vToken = *iterator;
		
		// Check whether "label" of this parameter matches:
		if( iterator == mTokens.end()								// Is end of token stream?
			|| (vFoundCommand->mParam[x].mTokenType != TOKEN_TYPE_INVALID	// Or not an unlabeled param?
			&& vFoundCommand->mParam[x].mTokenType != vToken->GetType()) )	// And it's also not a match?
		{
			if( vFoundCommand->mParam[x].mOptional != TALK_PARAM_IS_OPTIONAL )	// Required param? Error!
				throw parse_error( "This is not the right identifier.", vToken->GetStartChar(), vToken->GetEndChar() );
			else	// Optional param?
			{
				switch( vFoundCommand->mParam[x].mParamType )
				{
					case TALK_PARAM_TYPE_QUALIFIER:	// Qualifier? Set to FALSE since it isn't present:
						vDestination->mType = MEM_LOCATION_TYPE_IMMEDIATE;
						vDestination->mValue = new TalkVarValue( (bool) false );
						break;
					
					case TALK_PARAM_TYPE_INVALID:	// Syntactic sugar or no direct argument? Just ignore.
						break;
					
					default:	// Any other value? Set to empty to indicate it isn't present:
						vDestination->mType = MEM_LOCATION_TYPE_IMMEDIATE;
						vDestination->mValue = new TalkVarValue( TextMunger("") );
				}
				// Make sure the next param can try whether this token fits.
				--iterator;
			}
		}
		else	// Label match!
		{
			if( vFoundCommand->mParam[x].mTokenType == TOKEN_TYPE_INVALID )	// Unlabeled param?
				--iterator;	// Make sure token we took for label gets parsed.
			
			// This labeled param changes the kind of command generated?
			if( vFoundCommand->mParam[x].mCommandInstruction != NULL )	// Change it:
				vThisInstr.mInstructionProc = vFoundCommand->mParam[x].mCommandInstruction;
			
			switch( vFoundCommand->mParam[x].mParamType )	// Parse whatever param goes after the label:
			{
				case TALK_PARAM_TYPE_INVALID:	// Invalid? Syntactic sugar or no direct arg. Just ignore.
					break;
				
				case TALK_PARAM_TYPE_QUALIFIER:	// Qualifier is also syntactic sugar but sets the param to TRUE if present and to FALSE if not:
					vDestination->mType = MEM_LOCATION_TYPE_IMMEDIATE;
					vDestination->mValue = new TalkVarValue( (bool) true );
					break;
				
				case TALK_PARAM_TYPE_EXPRESSION:
					vDestination->mType = MEM_LOCATION_TYPE_STACK;
					vDestination->mLocation = vTemporaries.GetNewTemp();
					ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals,
										vDestination->mLocation );
					break;
				
				case TALK_PARAM_TYPE_VALUE:
					*vDestination = ParseValue( iterator, vCode, vTemporaries, vUserVarNames,
												vInternals, false );
					break;
				
				case TALK_PARAM_TYPE_CONTAINER:
					*vDestination = ParseValue( iterator, vCode, vTemporaries, vUserVarNames,
												vInternals, true );
					if( vDestination->mType == MEM_LOCATION_TYPE_IMMEDIATE )
						throw parse_error( "Expected container here.", vToken->GetStartChar(), vToken->GetEndChar() );
					break;
				
				case TALK_PARAM_TYPE_IDENTIFIER:
					++iterator;
					vToken = *iterator;
					switch( vToken->GetType() )
					{
						case TOKEN_TYPE_INTEGER:
						case TOKEN_TYPE_STRING:
						case TOKEN_TYPE_OPERATOR:
						case TOKEN_TYPE_NEWLINE:
							throw parse_error( "Expected identifier here.", vToken->GetStartChar(), vToken->GetEndChar() );
							break;
						
						default:
							vDestination->mType = MEM_LOCATION_TYPE_IMMEDIATE;
							vDestination->mValue = new TalkVarValue( new TextMunger( (char*) vToken->GetValue() ) );
					}
					break;
			}
		}
	}
	
	// Create appropriate instruction (params have been added above):
	vThisInstr.mCaller = this;
	vThisInstr.mParamCount = x;	// FIX ME! Not accurate if several optional params map to the same param.
	// If we are interested in the instruction's result, remember what param to store it in:
	switch( vFoundCommand->mResultParam )
	{
		case TALK_PARAM_IGNORE:	// Don't care about result.
			break;
		
		case TALK_PARAM_IT:
			vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mResult.mLocation = vUserVarNames.GetVarByName( "it", true );
			break;
		
		case TALK_PARAM_RESULT:
			vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mResult.mLocation = vUserVarNames.GetVarByName( "the result", true );
			break;
		
		default:
			vThisInstr.mResult = vThisInstr.mParam[vFoundCommand->mResultParam-1];
			break;
	}
	
	vCode.push_back( vThisInstr );	// Append instruction to our code.
	vThisInstr.ClearAllFields( false );	// Make sure destructor doesn't dispose of arguments.
}


/* --------------------------------------------------------------------------------
	ParseIntegerValue:
		Parse an integer, point, color or rect from the token stream and return
		it as an immediate.
	
	TAKES:
		iterator -		An iterator to the next token to parse.
		vTemporaries -	A list of indices to all temporary variables.
		vUserVarNames -	A list of all names of user variables plus their indices.
		vInternals -	A list of indices to all internal variables.
		doCreate -		True if we need this value to be a destination. This will
						create variables instead of turning them into literals,
						however it may still return immediates if the code
						explicitly specified them.
	
	REVISIONS:
		2000-11-07	UK		Created.
   ----------------------------------------------------------------------------- */

TalkMemLocation	HyperTalk::ParseIntegerValue( HTLIterator &iterator, TalkHandler& vCode,
											TalkTempVarList& vTemporaries,
											TalkVarNameList& vUserVarNames,
											TalkInternalVarList& vInternals, bool doCreate,
											HyperToken* vEndToken )
{
#pragma unused(vCode)
#pragma unused(vTemporaries)
#pragma unused(vUserVarNames)
#pragma unused(vInternals)
#pragma unused(doCreate)
	
	bool			vDoInteger = true;
	HyperToken		*vTop,
					*vRight,
					*vBottom,
					*vToken;
	bool			vDoPoint = false;
	bool			vDoRect = false;
	bool			vDoColor = false;
	TalkMemLocation	vResult;
	
	vToken = (*iterator);
	
	if( (vEndToken != NULL)
		&& (TalkTokenizer::GetOperatorTokenKind( vEndToken ) == OPERATOR_CHAR_COMMA) )
		;
	else
	{
		// Try to get comma and 2nd coordinate:
		iterator++;
		if( (iterator != mTokens.end())
			&& (TalkTokenizer::GetOperatorTokenKind( *iterator ) == OPERATOR_CHAR_COMMA) )
		{
			iterator++;
			if( (iterator != mTokens.end()) && ((*iterator)->GetType() == TOKEN_TYPE_INTEGER) )
			{
				vTop = (*iterator);
				vDoPoint = true;
				vDoInteger = false;
			}
			else
			{
				iterator--; iterator--;
			}
		}
		else
			iterator--;
			
		// Try to get comma and 3rd coordinate:
		if( vDoPoint )
		{
			iterator++;
			if( (iterator != mTokens.end()) && (TalkTokenizer::GetOperatorTokenKind( *iterator ) == OPERATOR_CHAR_COMMA) )
			{
				iterator++;
				if( (iterator != mTokens.end()) && ((*iterator)->GetType() == TOKEN_TYPE_INTEGER) )
				{
					vRight = (*iterator);
					vDoColor = true;
				}
				else
				{
					iterator--; iterator--;
				}
			}
			else
				iterator--;
		}
		
		// Try to get comma and 4th coordinate:
		if( vDoColor )
		{
			iterator++;
			if( (iterator != mTokens.end()) && (TalkTokenizer::GetOperatorTokenKind( *iterator ) == OPERATOR_CHAR_COMMA) )
			{
				iterator++;
				if( (iterator != mTokens.end()) && ((*iterator)->GetType() == TOKEN_TYPE_INTEGER) )
				{
					vBottom = (*iterator);
					vDoRect = true;
					vDoPoint = false;
				}
				else
				{
					iterator--; iterator--;
				}
			}
			else
			{
				iterator--;
			}
		}
	}
	
	if( vDoInteger )
	{
		vResult.mType = MEM_LOCATION_TYPE_IMMEDIATE;
		vResult.mValue = new TalkVarValue( (long) vToken->GetValue() );
	}
	else if( vDoRect )
	{
		LongRect		vBox;
		
		vBox.left = vToken->GetValue();
		vBox.top = vTop->GetValue();
		vBox.right = vRight->GetValue();
		vBox.bottom = vBottom->GetValue();
		
		vResult.mType = MEM_LOCATION_TYPE_IMMEDIATE;
		vResult.mValue = new TalkVarValue( vBox );
	}
	else if( vDoColor )
	{
		LongColor		vColor;
		
		vColor.red = vToken->GetValue();
		vColor.green = vTop->GetValue();
		vColor.blue = vRight->GetValue();
		
		vResult.mType = MEM_LOCATION_TYPE_IMMEDIATE;
		vResult.mValue = new TalkVarValue( vColor );
	}
	else if( vDoPoint )
	{
		LongPoint		vPos;
		
		vPos.x = vToken->GetValue();
		vPos.y = vTop->GetValue();
		
		vResult.mType = MEM_LOCATION_TYPE_IMMEDIATE;
		vResult.mValue = new TalkVarValue( vPos );
	}
	
	return vResult;
}

/* --------------------------------------------------------------------------------
	ParseValue:
		Parse the script that was tokenized before for one value.
	
	TAKES:
		iterator -		An iterator to the next token to parse.
		vTemporaries -	A list of indices to all temporary variables.
		vUserVarNames -	A list of all names of user variables plus their indices.
		vInternals -	A list of indices to all internal variables.
		doCreate -		True if we need this value to be a destination. This will
						create variables instead of turning them into literals,
						however it may still return immediates if the code
						explicitly specified them.
	
	REVISIONS:
		1999-11-29	UK		Created.
   ----------------------------------------------------------------------------- */

TalkMemLocation	HyperTalk::ParseValue( HTLIterator &iterator, TalkHandler& vCode,
										TalkTempVarList& vTemporaries,
										TalkVarNameList& vUserVarNames,
										TalkInternalVarList& vInternals, bool doCreate,
										HyperToken* vEndToken )
{
	HyperToken*			vToken;
	TalkMemLocation		vResult;
	HyperInstruction	vThisInstr,
						vPrevInstr;
	ValStackLocation	vTheTemp;
	long				vChunkType;
	bool				vAreSubChunk = false;
	
	++iterator;
	vToken = (*iterator);
	
	if( iterator == mTokens.end() )
		throw parse_error( "Expected value, found EOF.", vToken->GetStartChar(), vToken->GetEndChar() );
	
	switch( vToken->GetType() )
	{
		case TOKEN_TYPE_INTEGER:	// Integer, point, color or rect.
			vResult = ParseIntegerValue( iterator, vCode, vTemporaries, vUserVarNames, vInternals,
											doCreate, vEndToken );
			break;
		
		case TOKEN_TYPE_IDENTIFIER:
			vResult = ParseAnyIdentifier( iterator, vCode, vTemporaries, vUserVarNames,
											vInternals, doCreate, vEndToken );
			break;
		
		case TOKEN_TYPE_STRING:
			vResult = ParseStringToken( iterator, vCode, vTemporaries, vUserVarNames, vInternals );
			break;
				
		case TOKEN_TYPE_STDIN_IDENTIFIER:
			vTheTemp = vTemporaries.GetNewTemp();
			
			vThisInstr.mCaller = this;
			vThisInstr.mInstructionProc = HyperGetStdinInstruction;
			vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mResult.mLocation = vTheTemp;
			
			vCode.push_back( vThisInstr );	// Add instruction.
			
			vResult = vThisInstr.mResult;
			vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
			break;
		
		case TOKEN_TYPE_THE_IDENTIFIER:
		{
			++iterator;
			vToken = (*iterator);
			
			// Sort out worst mistakes for "the":
			switch( vToken->GetType() )
			{
				case TOKEN_TYPE_INTEGER:
				case TOKEN_TYPE_OPERATOR:
				case TOKEN_TYPE_STRING:
					throw parse_error( "Expected identifier after \"the\".", vToken->GetStartChar(), vToken->GetEndChar() );
					break;
			}
			
			// Now fetch name of function/property/whatever behind "the" and do additional checks:
			TextMunger						vOneArgFcnName( (char*) vToken->GetValue() );
			bool							vFoundSomething = false;
			
			vOneArgFcnName.RecalcHash();	// Makes compares faster.
			
			switch( vToken->GetType() )
			{
				case TOKEN_TYPE_NUMBER_IDENTIFIER:
					vResult = ParseNumberOfFunction( iterator, vCode, vTemporaries, vUserVarNames,
											vInternals, doCreate, vEndToken, vFoundSomething );	// vFoundSomething is passed by REFERENCE!
					if( vFoundSomething )
						break;	// Found something? No need to go on in this switch statement anymore.
					else
						;	// Drop through if the function above couldn't parse a "number of <chunkType>s of ..." expression.
				
				default:
				{
					if( vEndToken != NULL && vEndToken->GetType() == TOKEN_TYPE_OF_IDENTIFIER )
						break;	// Don't try parsing function or property when our caller wants the next "of".
					
					++iterator;
					if( (*iterator)->GetType() == TOKEN_TYPE_OF_IDENTIFIER )
					{
						TalkFunctionList::iterator		fli;
						
						// Look for a one-argument function of this name:
						fli = sOneArgFunctions.find( vOneArgFcnName );
						if( fli != sOneArgFunctions.end() )	// We found something!
						{
							vResult = ParseValue( iterator, vCode, vTemporaries, vUserVarNames, vInternals, false );
							
							// Now generate instruction for this function call:
							vThisInstr.mCaller = this;
							vThisInstr.mInstructionProc = (*fli).second;	// Proc comes from One Arg Function List.
							vThisInstr.mParamCount = 1;		// Only a single param.
							vThisInstr.mParam[0] = vResult;
							vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
							vThisInstr.mResult.mLocation = vTemporaries.GetNewTemp();
						
							vCode.push_back( vThisInstr );	// Append instruction to our code.
							
							vFoundSomething = true;
							vResult = vThisInstr.mResult;
							
							vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
						}
						else	// Try properties here before erroring:
						{
							// Get value to get property from:
							vResult = ParseValue( iterator, vCode, vTemporaries, vUserVarNames, vInternals, false );
							
							// Now generate instruction that retrieves the property:
							vThisInstr.mCaller = this;
							vThisInstr.mInstructionProc = HyperGetPropInstruction;
							vThisInstr.mParamCount = 2;
							vThisInstr.mParam[0] = vResult;
							vThisInstr.mParam[1].mType = MEM_LOCATION_TYPE_IMMEDIATE;
							vThisInstr.mParam[1].mValue = new TalkVarValue( new TextMunger( vOneArgFcnName ) );	// Munger is taken over by value.
							vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
							vThisInstr.mResult.mLocation = vTemporaries.GetNewTemp();
							
							vCode.push_back( vThisInstr );	// Append instruction to our code.
							
							vFoundSomething = true;
							vResult = vThisInstr.mResult;
							
							vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
						}
					}
					else
						--iterator;
					break;
				}
			}
			
			if( !vFoundSomething )	// No "of" or no property or one-argument function? Try special variables and global properties:
			{
				TalkGlobalPropertyList::iterator	gpli;
				
				gpli = sGlobalProperties.find( vOneArgFcnName );
				if( gpli != sGlobalProperties.end() )	// Found global property or special var.
					vResult = (*gpli).second.GetResult( this, vCode, vTemporaries, vUserVarNames, vInternals );
				else	// None in list? It's property syntax on default object for properties:
				{
					vThisInstr.mCaller = this;
					vThisInstr.mInstructionProc = HyperGetPropInstruction;
					vThisInstr.mParamCount = 2;
					vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_IMMEDIATE;
					vThisInstr.mParam[0].mValue = new TalkVarValue( GetPropertyDefaultObject() );
					vThisInstr.mParam[1].mType = MEM_LOCATION_TYPE_IMMEDIATE;
					vThisInstr.mParam[1].mValue = new TalkVarValue( new TextMunger( vOneArgFcnName ) );	// Munger is taken over by value.
					vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
					vThisInstr.mResult.mLocation = vTemporaries.GetNewTemp();
					
					vCode.push_back( vThisInstr );	// Append instruction to our code.
					
					vFoundSomething = true;
					vResult = vThisInstr.mResult;
					
					vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
				}
			}
			break;
		}
		
		case TOKEN_TYPE_ENTRY_IDENTIFIER:
		{
			HyperToken		vTokenToEnd( 0,0, TOKEN_TYPE_OF_IDENTIFIER );
			
			if( vEndToken != NULL && vEndToken->GetType() == vTokenToEnd.GetType() )	// Can't get an "of" because our caller wants it?
			{
				vResult = ParseAnyIdentifier( iterator, vCode, vTemporaries, vUserVarNames, vInternals,
												doCreate, vEndToken );
				break;
			}
			
			// Parse index:
			vTheTemp = vTemporaries.GetNewTemp();
			
			ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals, vTheTemp, &vTokenToEnd );
			vThisInstr.mCaller = this;
			vThisInstr.mInstructionProc = HyperGetEntryInstruction;
			vThisInstr.mParam[1].mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mParam[1].mLocation = vTheTemp;
			vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mResult.mLocation = vTheTemp;
			
			++iterator;
			vToken = (*iterator);
			if( vToken->GetType() != TOKEN_TYPE_OF_IDENTIFIER )
				throw parse_error( "Expected \"of\" after entry index.", vToken->GetStartChar(), vToken->GetEndChar() );
			
			// Parse value to get entry from:
			vThisInstr.mParam[0] = ParseValue( iterator, vCode, vTemporaries, vUserVarNames, vInternals, true );
			
			vCode.push_back( vThisInstr );	// Add instruction.
			
			// Return the location where "Get Entry" instruction put its result:
			vResult = vThisInstr.mResult;
							
			vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
			break;
		}
		
		case TOKEN_TYPE_URL_IDENTIFIER:
			// Parse index:
			vTheTemp = vTemporaries.GetNewTemp();
			
			ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals, vTheTemp );
			vThisInstr.mCaller = this;
			vThisInstr.mInstructionProc = HyperGetURLInstruction;
			vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mParam[0].mLocation = vTheTemp;
			vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mResult.mLocation = vTheTemp;
			
			vCode.push_back( vThisInstr );	// Add instruction.
			
			// Return the location where "Get URL" instruction put its result:
			vResult = vThisInstr.mResult;
							
			vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
			break;
		
		case TOKEN_TYPE_CHAR_IDENTIFIER:
		case TOKEN_TYPE_WORD_IDENTIFIER:
		case TOKEN_TYPE_ITEM_IDENTIFIER:
		case TOKEN_TYPE_LINE_IDENTIFIER:
		case TOKEN_TYPE_BYTE_IDENTIFIER:
		{
			HyperToken		vEndingTokenForIndex( 0, 0, TOKEN_TYPE_OF_IDENTIFIER );
			
			vThisInstr.mParamCount = 4;
			
			switch( vToken->GetType() )
			{
				case TOKEN_TYPE_WORD_IDENTIFIER:
					vChunkType = CHUNK_TYPE_WORD;
					break;
				
				case TOKEN_TYPE_ITEM_IDENTIFIER:
					vThisInstr.mParam[4].mType = MEM_LOCATION_TYPE_STACK;
					vThisInstr.mParam[4].mLocation = vUserVarNames.GetVarByName( "the itemdelimiter", false );
					if( vThisInstr.mParam[4].mLocation == MEM_INDEX_ACCU )	// This var doesn't exist?
					{
						vThisInstr.mParam[4].mType = MEM_LOCATION_TYPE_IMMEDIATE;
						vThisInstr.mParam[4].mValue = new TalkVarValue(new TextMunger(""));	// Varvalue takes over munger.
					}
					vThisInstr.mParamCount = 5;	// Make sure this param is known.
					vChunkType = CHUNK_TYPE_ITEM;
					break;
				
				case TOKEN_TYPE_LINE_IDENTIFIER:
					vChunkType = CHUNK_TYPE_LINE;
					break;
				
				case TOKEN_TYPE_BYTE_IDENTIFIER:
					vChunkType = CHUNK_TYPE_BYTE;
					break;
				
				case TOKEN_TYPE_CHAR_IDENTIFIER:
				default:
					vChunkType = CHUNK_TYPE_CHARACTER;
					break;
			}
			
			// Parse index:
			vTheTemp = vTemporaries.GetNewTemp();
			
			ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals, vTheTemp, &vEndingTokenForIndex );
			vThisInstr.mCaller = this;
			vThisInstr.mInstructionProc = HyperGetChunkOffsInstruction;
			vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mParam[0].mLocation = vTheTemp;
			
			// Check if this is range, if yes, put end of range expression into mParam[1]:
			++iterator;
			vToken = (*iterator);
			if( vToken->GetType() == TOKEN_TYPE_TO_IDENTIFIER )
			{
				vTheTemp = vTemporaries.GetNewTemp();
				ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals, vTheTemp, &vEndingTokenForIndex );
				
				vThisInstr.mParam[1].mType = MEM_LOCATION_TYPE_STACK;
				vThisInstr.mParam[1].mLocation = vTheTemp;
				++iterator;
			}
			else	// No range? Set mParam[1] to 0 (zero) to indicate this:
			{
				vThisInstr.mParam[1].mType = MEM_LOCATION_TYPE_LONG;
				vThisInstr.mParam[1].mLong = 0;
			}
			
			vToken = (*iterator);
			if( vToken->GetType() != TOKEN_TYPE_OF_IDENTIFIER )
				throw parse_error( "Expected \"to\" or \"of\" after chunk start.", vToken->GetStartChar(), vToken->GetEndChar() );
			
			// Parse value to get entry from:
			vThisInstr.mParam[2] = ParseValue( iterator, vCode, vTemporaries, vUserVarNames, vInternals, true );
			
			/* Now check whether parsed value was a chunk, if yes we remove its HyperGetChunkRef
				instruction so we don't overwrite its chunk: */
			vPrevInstr = vCode.back();
			if( vPrevInstr.mInstructionProc == HyperGetChunkRefInstruction )	// Was 'get ref' instr?
			{
				vAreSubChunk = true;	// Remember.
				vCode.pop_back();		// Remove from code.
				vThisInstr.mParam[2] = vPrevInstr.mParam[0];	// Make sure it parses the actual value.
			}
			
			// Set chunk type:
			vThisInstr.mParam[3].mType = MEM_LOCATION_TYPE_LONG;
			vThisInstr.mParam[3].mLong = vChunkType;
			
			vCode.push_back( vThisInstr );	// Add offset calculating instruction.
			vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
			
			// ----- Now make actual chunking instruction:
			vThisInstr.mCaller = this;
			vThisInstr.mInstructionProc = HyperGetChunkRefInstruction;
			if( vAreSubChunk )
				vThisInstr.mParam[0] = vPrevInstr.mParam[0];
			else
				vThisInstr.mParam[0] = vThisInstr.mParam[2];	// Param 0 is value to get entry from. We simply take this from the previously pushed instruction.
			vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mResult.mLocation = vTheTemp;	// We can re-use last temp since we don't need offsets anymore.
			
			vCode.push_back( vThisInstr );	// Add instruction that actually does the chunking based on previously calculated offsets.
			
			// Return the location where "GetChunkRef" instruction put its result:
			vResult = vThisInstr.mResult;
			
			vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
			vPrevInstr.ClearAllFields( false );	// Same here.
			break;
		}
		
		default:	// Treat any other token as any unknown identifier:
		{
			OperatorTypeEnum		vOpType;
			
			vOpType = TalkTokenizer::GetOperatorTokenKind(vToken);
			switch( vOpType )
			{
				case OPERATOR_CHAR_OPEN_BRACE:	// Opening bracket?
				{
					if( doCreate )
						throw parse_error( "Expected container, found expression.", vToken->GetStartChar(), vToken->GetEndChar() );
					
					vResult.mType = MEM_LOCATION_TYPE_STACK;
					vResult.mLocation = vTemporaries.GetNewTemp();
					
					ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals, vResult.mLocation );
					
					++iterator;
					vToken = *iterator;
					
					if( TalkTokenizer::GetOperatorTokenKind( vToken ) != OPERATOR_CHAR_CLOSE_BRACE )	// After "(" we need a closing one.
						throw parse_error( "Expected \")\" here.", vToken->GetStartChar(), vToken->GetEndChar() );
					break;
				}
				
				case OPERATOR_CHAR_INVALID:	// Not an operator?
					vResult = ParseAnyIdentifier( iterator, vCode, vTemporaries, vUserVarNames, vInternals,
													doCreate, vEndToken );
					break;
				
				default:	// Is an operator? Look it up:
				{
					if( doCreate )	// We need a container? Then it's probably just a variable that looks like one of our operators:
					{
						switch( vToken->GetType() )
						{
							case TOKEN_TYPE_INTEGER:
							case TOKEN_TYPE_OPERATOR:
							case TOKEN_TYPE_STRING:
								throw parse_error( "Expected container, found operator/value.", vToken->GetStartChar(), vToken->GetEndChar() );
								break;
							
							default:
								return( ParseAnyIdentifier( iterator, vCode, vTemporaries, vUserVarNames, vInternals,
																doCreate, vEndToken ) );
						}
						
					}
					
					TalkOperatorList::iterator	vCurrOp;
					
					// Find the unary operator in our list:
					vCurrOp = sUnaryOperators.find( vOpType );
					if( vCurrOp == sUnaryOperators.end() )
						throw parse_error( "Expected value or unary operator, found binary operator.", vToken->GetStartChar(), vToken->GetEndChar() );
					
					// Parse the value to use the operator on (may contain another unary operator):
					vResult = ParseValue( iterator, vCode, vTemporaries, vUserVarNames, vInternals, false );
					
					// Add an instruction that does the operator's magic:
					vThisInstr.mCaller = this;
					vThisInstr.mInstructionProc = vCurrOp->second;	// Instruction for this unary operator.
					vThisInstr.mParam[0] = vResult;
					vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
					vThisInstr.mResult.mLocation = vTemporaries.GetNewTemp();
					vThisInstr.mParamCount = 1;
					
					vCode.push_back( vThisInstr );	// Add instruction.
					
					vResult = vThisInstr.mResult;
					vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
				}
			}
			break;
		}
	}
	
	return( vResult );
}


/* --------------------------------------------------------------------------------
	ParseStringToken:
		Make the correct instructions for a string immediate value from a token
		that has already been identified as TOKEN_TYPE_STRING or
		TOKEN_TYPE_IDENTIFIER or another identifier token.
	
	TAKES:
		iterator		-	An iterator into the token list of the script being
							parsed. This must point to the token that might be a
							variable.
		vCode			-	The handler to add any code to.
		vTemporaries	-	List of temporary variables where we can request a
							temporary when we need one.
		vUserVarNames	-	Names of user variables + mapping to their stack
							location. Use this to look up variable names.
		vInternals		-	List of internal variables where we can request and
							release one as needed.
		doCreate		-	A boolean whether we should create variables or make
							unknown identifiers unquoted literals. This is TRUE
							when the parsing context requires a container.
	
	GIVES:
		TalkMemLocation	-	A stack location of a temp or immediate with the
							result of what we parsed.
	
	REVISIONS:
		1999-12-18	UK		Created.
   ----------------------------------------------------------------------------- */

TalkMemLocation	HyperTalk::ParseStringToken( HTLIterator &iterator, TalkHandler& vCode,
												TalkTempVarList& vTemporaries,
												TalkVarNameList& vUserVarNames,
												TalkInternalVarList& vInternals )
{
#pragma unused(vCode)
#pragma unused(vTemporaries)
#pragma unused(vUserVarNames)
#pragma unused(vInternals)
	TalkMemLocation		vResult;
	
	vResult.mType = MEM_LOCATION_TYPE_IMMEDIATE;
	vResult.mValue = new TalkVarValue( new TextMunger( (char*) (*iterator)->GetValue() ) );
	
	return vResult;
}


/* --------------------------------------------------------------------------------
	ParseObjectDescriptorToken:
		Add instructions to the script that match one of the objects we support.
	
	TAKES:
		iterator		-	An iterator into the token list of the script being
							parsed. This must point to the token that might be a
							variable.
		vCode			-	The handler to add any code to.
		vTemporaries	-	List of temporary variables where we can request a
							temporary when we need one.
		vUserVarNames	-	Names of user variables + mapping to their stack
							location. Use this to look up variable names.
		vInternals		-	List of internal variables where we can request or
							release one as needed.
		doCreate		-	A boolean whether the parsing context requires a
							container or is satisfied with a value.
	
	GIVES:
		TalkMemLocation	-	A reference to an entity.
	
	REVISIONS:
		2000-12-22	UK		Added support for plug-in object types using
							RegisterObjectDescriptor().
		2000-11-14	UK		Created.
   ----------------------------------------------------------------------------- */

TalkMemLocation	HyperTalk::ParseObjectDescriptorToken( HTLIterator &iterator, TalkHandler& vCode,
												TalkTempVarList& vTemporaries,
												TalkVarNameList& vUserVarNames,
												TalkInternalVarList& vInternals, bool doCreate )
{
#pragma unused(doCreate)

	HyperToken*			vToken;
	TalkMemLocation		vParam;
	HyperInstruction	vThisInstr;
		
	vToken = *iterator;
	vThisInstr.mCaller = this;
	vThisInstr.mInstructionProc = NULL;
	vParam.mType = MEM_LOCATION_TYPE_INVALID;
	
	TalkObjectDescriptorList::iterator	foundObjSyntax;
	bool								vParamsSet[TALK_INSTR_MAX_PARAMS+1];
	TalkCommandEntry*					vFoundEntityType;
	short								x,
										vDestParam;
	TalkMemLocation*					vDestination;
	
	/* Clear array. 0 is result, 1 to TALK_INSTR_MAX_PARAMS are params.
		The array is used to keep track of which params we did already set. */
	for( x = 0; x <= TALK_INSTR_MAX_PARAMS; x++ )
		vParamsSet[x] = false;
	
	// Make sure we have somewhere to put our result:
	vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
	vThisInstr.mResult.mLocation = vTemporaries.GetNewTemp();
	
	// Look up proper entry for this starting token:
	foundObjSyntax = sObjectDescriptors.find( vToken->GetType() );
	if( foundObjSyntax == sObjectDescriptors.end() )
	{
		vParam.mType = MEM_LOCATION_TYPE_INVALID;	// Return saying we failed.
		return vParam;
	}
	
	vFoundEntityType = foundObjSyntax->second;
	vThisInstr.mInstructionProc = vFoundEntityType->mParam[0].mCommandInstruction;
	vThisInstr.mRefCon = vFoundEntityType->mRefCon;
	
	--iterator; 	// Step back one so we can get the object name & direct param with the same code used for params.
	
	// Parse the syntax entry for our object and its params:
	for( x = 0; x < vFoundEntityType->mParamCount; x++ )
	{
		vDestParam = vFoundEntityType->mParam[x].mDestParam;
		if( vFoundEntityType->mParam[x].mParamType != TALK_PARAM_TYPE_INVALID )	// We're not supposed to ignore this param's label?
		{
			switch( vDestParam )
			{
				case TALK_PARAM_RESULT:
					vDestination = &vThisInstr.mResult;
					break;
				
				case TALK_PARAM_IT:
					throw logic_error( "Invalid internal command definition." );
					break;
				
				default:
					if( vParamsSet[vDestParam] == false )	// Param not already specified?
					{
						vParamsSet[vDestParam] = true;
						vDestination = &vThisInstr.mParam[vDestParam-1];
						vThisInstr.mParamCount = max<short>(vDestParam, vThisInstr.mParamCount);
					}
					else	// It is? Just skip this one.
						continue;
			}
		}
		
		++iterator;
		vToken = *iterator;
		
		// Check whether "label" of this parameter matches:
		if( iterator == mTokens.end()
			|| (vFoundEntityType->mParam[x].mTokenType != vToken->GetType()	// No?
			&& vFoundEntityType->mParam[x].mTokenType != TOKEN_TYPE_INVALID) )	// And it's also not an unlabeled param?
		{
			if( vFoundEntityType->mParam[x].mOptional != TALK_PARAM_IS_OPTIONAL )	// Required param? Error!
				throw parse_error( "Wrong identifier in object descriptor.", vToken->GetStartChar(), vToken->GetEndChar() );
			else	// Optional param?
			{
				switch( vFoundEntityType->mParam[x].mParamType )
				{
					case TALK_PARAM_TYPE_QUALIFIER:	// Qualifier? Set to FALSE since it isn't present:
						vDestination->mType = MEM_LOCATION_TYPE_IMMEDIATE;
						vDestination->mValue = new TalkVarValue( (bool) false );
						break;
					
					case TALK_PARAM_TYPE_INVALID:	// Syntactic sugar or no direct argument? Just ignore.
						break;
					
					default:	// Any other value? Set to empty to indicate it isn't present:
						vDestination->mType = MEM_LOCATION_TYPE_IMMEDIATE;
						vDestination->mValue = new TalkVarValue( TextMunger("") );
				}
				// Make sure the next param can try whether this token fits.
				--iterator;
			}
		}
		else	// Label match!
		{
			if( vFoundEntityType->mParam[x].mTokenType == TOKEN_TYPE_INVALID )	// Unlabeled param?
				--iterator;	// Make sure token we took for label gets parsed.
			
			// This labeled param changes the kind of command generated?
			if( vFoundEntityType->mParam[x].mCommandInstruction != NULL )	// Change it:
				vThisInstr.mInstructionProc = vFoundEntityType->mParam[x].mCommandInstruction;
			
			switch( vFoundEntityType->mParam[x].mParamType )	// Parse whatever param goes after the label:
			{
				case TALK_PARAM_TYPE_INVALID:	// Invalid? Syntactic sugar or no direct arg. Just ignore.
					break;
				
				case TALK_PARAM_TYPE_QUALIFIER:	// Qualifier is also syntactic sugar but sets the param to TRUE if present and to FALSE if not:
					vDestination->mType = MEM_LOCATION_TYPE_IMMEDIATE;
					vDestination->mValue = new TalkVarValue( (bool) true );
					break;
				
				case TALK_PARAM_TYPE_EXPRESSION:
					if( iterator == mTokens.end() )
						throw runtime_error( "Expected expression but found end of script." );
					vDestination->mType = MEM_LOCATION_TYPE_STACK;
					vDestination->mLocation = vTemporaries.GetNewTemp();
					ParseExpression( iterator, vCode, vTemporaries, vUserVarNames, vInternals,
										vDestination->mLocation );
					break;
				
				case TALK_PARAM_TYPE_VALUE:
					if( iterator == mTokens.end() )
						throw runtime_error( "Expected value but found end of script." );
					*vDestination = ParseValue( iterator, vCode, vTemporaries, vUserVarNames,
												vInternals, false );
					break;
				
				case TALK_PARAM_TYPE_CONTAINER:
					if( iterator == mTokens.end() )
						throw runtime_error( "Expected container but found end of script." );
					*vDestination = ParseValue( iterator, vCode, vTemporaries, vUserVarNames,
												vInternals, true );
					if( vDestination->mType == MEM_LOCATION_TYPE_IMMEDIATE )
						throw parse_error( "Expected container in object descriptor.", vToken->GetStartChar(), vToken->GetEndChar() );
					break;
				
				case TALK_PARAM_TYPE_IDENTIFIER:
					if( iterator == mTokens.end() )
						throw runtime_error( "Expected identifier but found end of script." );
					++iterator;
					vToken = *iterator;
					if( iterator == mTokens.end() )
						throw runtime_error( "Expected: identifier. Found: end of script." );
					switch( vToken->GetType() )
					{
						case TOKEN_TYPE_INTEGER:
						case TOKEN_TYPE_STRING:
						case TOKEN_TYPE_OPERATOR:
						case TOKEN_TYPE_NEWLINE:
							throw parse_error( "Expected identifier at this point in object descriptor.", vToken->GetStartChar(), vToken->GetEndChar() );
							break;
						
						default:
							vDestination->mType = MEM_LOCATION_TYPE_IMMEDIATE;
							vDestination->mValue = new TalkVarValue( new TextMunger( (char*) vToken->GetValue() ) );
					}
					break;
			}
		}
	}
	
	if( vThisInstr.mInstructionProc == NULL )	// Nothing found?
	{
		vParam.mType = MEM_LOCATION_TYPE_INVALID;	// Return saying we failed.
		return vParam;
	}
	
	vCode.push_back( vThisInstr );	// Add instruction.
	
	vParam = vThisInstr.mResult;
	vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
	
	return vParam;
}


/* --------------------------------------------------------------------------------
	ParseVariableToken:
		Make the correct instructions for a stack variable from a token
		that has already been identified as TOKEN_TYPE_IDENTIFIER or another
		identifier token.
	
	TAKES:
		iterator		-	An iterator into the token list of the script being
							parsed. This must point to the token that might be a
							variable.
		vCode			-	The handler to add any code to.
		vTemporaries	-	List of temporary variables where we can request a
							temporary when we need one.
		vUserVarNames	-	Names of user variables + mapping to their stack
							location. Use this to look up variable names.
		vInternals		-	List of internal variables we can use and later
							release.
		doCreate		-	A boolean whether we should create variables or make
							unknown identifiers unquoted literals. This is TRUE
							when the parsing context requires a container.
	
	GIVES:
		TalkMemLocation	-	The stack location of the variable this parsed. If
							this returns the accu no variable could be found.
	
	REVISIONS:
		1999-12-18	UK		Created.
   ----------------------------------------------------------------------------- */

TalkMemLocation	HyperTalk::ParseVariableToken( HTLIterator &iterator, TalkHandler& vCode,
												TalkTempVarList& vTemporaries,
												TalkVarNameList& vUserVarNames,
												TalkInternalVarList& vInternals,
												bool doCreate )
{
#pragma unused(vCode)
#pragma unused(vTemporaries)
#pragma unused(vInternals)

	TalkMemLocation		vResult;
	
	vResult.mType = MEM_LOCATION_TYPE_STACK;
	vResult.mLocation = vUserVarNames.GetVarByName( (char*) (*iterator)->GetValue(), doCreate );
	
	return vResult;
}


/* --------------------------------------------------------------------------------
	ParseNumberOfFunction:
		This parses the "number of <chunkType>s of <value>" construct. To still
		allow for the "number of <object>" property, or a variable named "number",
		this carefully backtracks through the token stream if it can't parse a
		proper "number of <chunkType>s of <value>" expression and returns a
		boolean value indicating this.
	
	TAKES:
		iterator		-	An iterator into the token list of the script being
							parsed. This must point to the token that might be a
							variable.
		vCode			-	The handler to add any code to.
		vTemporaries	-	List of temporary variables where we can request a
							temporary when we need one.
		vUserVarNames	-	Names of user variables + mapping to their stack
							location. Use this to look up variable names.
		vInternals		-	List of internal variables we can use and later
							release.
		doCreate		-	A boolean whether we should create variables or make
							unknown identifiers unquoted literals. This is TRUE
							when the parsing context requires a container.
		vEndToken		-	A token to stop parsing at.
	
	GIVES:
		TalkMemLocation	-	The stack location of a temporary holding the result
							of the count instruction. If this wasn't successful
							(as indicated by vFoundSomething) this is set to the
							Accu.
		vFoundSomething	-	This is set to TRUE if this successfully parsed the
							construct. If this is FALSE, everything is as it was
							before the call and you should try parsing this
							differently.
	
	REVISIONS:
		2001-02-13	UK		Created.
   ----------------------------------------------------------------------------- */

TalkMemLocation	HyperTalk::ParseNumberOfFunction( HTLIterator &iterator, TalkHandler& vCode,
													TalkTempVarList& vTemporaries,
													TalkVarNameList& vUserVarNames,
													TalkInternalVarList& vInternals, bool doCreate,
													HyperToken* vEndToken, bool &vFoundSomething )
{
	TalkMemLocation		vResult;
	HyperToken			*vToken;
	long				vChunkType;
	HyperInstruction	vThisInstr;
	
	++iterator;
	vToken = (*iterator);
	if( vToken->GetType() != TOKEN_TYPE_OF_IDENTIFIER )
	{
		--iterator;
		vFoundSomething = false;
		vResult.mType = MEM_LOCATION_TYPE_STACK;
		vResult.mLocation = MEM_INDEX_ACCU;
		return vResult;
	}
	
	++iterator;
	vToken = (*iterator);
	switch( vToken->GetType() )
	{
		case TOKEN_TYPE_CHARS_IDENTIFIER:
			vChunkType = CHUNK_TYPE_CHARACTER;
			break;
		
		case TOKEN_TYPE_WORDS_IDENTIFIER:
			vChunkType = CHUNK_TYPE_WORD;
			break;
		
		case TOKEN_TYPE_LINES_IDENTIFIER:
			vChunkType = CHUNK_TYPE_LINE;
			break;
		
		case TOKEN_TYPE_BYTES_IDENTIFIER:
			vChunkType = CHUNK_TYPE_BYTE;
			break;
		
		case TOKEN_TYPE_ITEMS_IDENTIFIER:
			vThisInstr.mParam[2].mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mParam[2].mLocation = vUserVarNames.GetVarByName( "the itemdelimiter", false );
			if( vThisInstr.mParam[2].mLocation != MEM_INDEX_ACCU )	// This var exists?
			{
				vThisInstr.mParam[2].mType = MEM_LOCATION_TYPE_IMMEDIATE;
				vThisInstr.mParam[2].mValue = new TalkVarValue(new TextMunger(""));	// Varvalue takes over munger.
			}
			vChunkType = CHUNK_TYPE_ITEM;
			break;
		
		default:	// Ooops. Was a property, I suppose:
			vChunkType = CHUNK_TYPE_INVALID;	// Make sure code below doesn't try to chunk again.
			--iterator; --iterator;	// Step back so global prop/etc. parser below gets this.
			vFoundSomething = false;	// Make sure we return failure.
			break;
	}
	
	if( vChunkType != CHUNK_TYPE_INVALID )
	{
		++iterator;
		vToken = (*iterator);
		if( vToken->GetType() != TOKEN_TYPE_OF_IDENTIFIER )
			throw parse_error( "Expected \"of\" after \"the number of <chunkType>s\".", vToken->GetStartChar(), vToken->GetEndChar() );
		
		vResult = ParseValue( iterator, vCode, vTemporaries, vUserVarNames, vInternals, false );
		
		vThisInstr.mCaller = this;
		vThisInstr.mInstructionProc = HyperCountChunkInstruction;
		vThisInstr.mParamCount = (vChunkType == CHUNK_TYPE_ITEM) ? 3 : 2;
		vThisInstr.mParam[0] = vResult;
		vThisInstr.mParam[1].mType = MEM_LOCATION_TYPE_LONG;
		vThisInstr.mParam[1].mLocation = vChunkType;
		vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
		vThisInstr.mResult.mLocation = vTemporaries.GetNewTemp();
		
		vCode.push_back( vThisInstr );	// Add instruction.
		
		// Return the location where "count" instruction put its result:
		vFoundSomething = true;
		vResult = vThisInstr.mResult;
		vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
	}
	
	return vResult;
}


/* --------------------------------------------------------------------------------
	ParseAnyIdentifier:
		We encountered one of the identifier tokens. Try a look-ahead to determine
		whether it's a function call, array notation, property or just plain an
		identifier.
	
	TAKES:
		iterator		-	An iterator into the token list of the script being
							parsed. This must point to the token that might be a
							variable.
		vCode			-	The handler to add any code to.
		vTemporaries	-	List of temporary variables where we can request a
							temporary when we need one.
		vUserVarNames	-	Names of user variables + mapping to their stack
							location. Use this to look up variable names.
		vInternals		-	List of internal vars we can request and release as
							needed.
		doCreate		-	A boolean whether we should create variables or make
							unknown identifiers unquoted literals. This is TRUE
							when the parsing context requires a container.
	
	GIVES:
		TalkMemLocation	-	A stack location of a temp or immediate with the
							result of what we parsed.
	
	REVISIONS:
		1999-12-18	UK		Created.
   ----------------------------------------------------------------------------- */

TalkMemLocation	HyperTalk::ParseAnyIdentifier( HTLIterator &iterator, TalkHandler& vCode,
												TalkTempVarList& vTemporaries,
												TalkVarNameList& vUserVarNames,
												TalkInternalVarList& vInternals,
												bool doCreate, HyperToken* vEndToken )
{
	HyperToken*			vToken;
	TalkMemLocation		vResult;
	char*				vFcnName;
	HyperInstruction	vThisInstr;
	
	vToken = (*iterator);	// Remember this token.
	
	// Now look ahead to check whether it's a property, function call etc.:
	++iterator;
	if( iterator == mTokens.end() )	// No more token?
		--iterator;	// Step back and parse as single identifier.
	else	// Identifier with follow-up?
	{
		// Now examine the next token to find out what it is:
		switch( (*iterator)->GetType() )
		{
			case TOKEN_TYPE_OPERATOR:
				switch( (char) (*iterator)->GetValue() )
				{
					case OPERATOR_CHAR_OPEN_BRACE:		// Function call.
						if( vEndToken != NULL
							&& vEndToken->GetType() == TOKEN_TYPE_OPERATOR
							&& vEndToken->GetValue() == OPERATOR_CHAR_OPEN_BRACE )
						{
							--iterator;
							break;	// Don't try parsing this when caller wants next brace.
						}
						vFcnName = (char*) vToken->GetValue();
						iterator++;
						vResult.mType = MEM_LOCATION_TYPE_STACK;
						vResult.mLocation = vTemporaries.GetNewTemp();
						ParseFunctionCall( vFcnName, iterator, vCode, vTemporaries, vUserVarNames,
											vInternals, vResult );
						return vResult;
						break;
					
					case OPERATOR_CHAR_OPEN_ABRACE:		// Array notation. Exit gracefully, not yet supported.
						--iterator;
						break;
					
					default:
						--iterator;
				}
				break;
			
			case TOKEN_TYPE_OF_IDENTIFIER:		// Property notation or function call.
			{
				if( vEndToken != NULL && vEndToken->GetType() == TOKEN_TYPE_OF_IDENTIFIER )
				{
					--iterator;
					break;	// Don't try parsing function or property when our caller wants the next "of".
				}
				
				if( vToken->GetType() == TOKEN_TYPE_NUMBER_IDENTIFIER )
				{
					bool		vFoundSomething = false;
					
					--iterator;	// Step back one identifier so "of" can be parsed by the call below:
					vResult = ParseNumberOfFunction( iterator, vCode, vTemporaries, vUserVarNames,
													vInternals, doCreate, vEndToken, vFoundSomething );	// vFoundSomething is passed by REFERENCE!
					if( vFoundSomething )
						return vResult;	// Found something? No need to go on in this function anymore.
					else
						++iterator;	// Skip "of", then drop through if the function above couldn't parse a "number of <chunkType>s of ..." expression.
				}
				
				TalkFunctionList::iterator		fli;
				TextMunger						vOneArgFcnName( (char*) vToken->GetValue() );
				
				vOneArgFcnName.RecalcHash();	// Makes compares faster.
				
				// Look for a one-argument function of this name:
				fli = sOneArgFunctions.find( vOneArgFcnName );
				if( fli != sOneArgFunctions.end() )	// We found something!
				{
					vResult = ParseValue( iterator, vCode, vTemporaries, vUserVarNames,
											vInternals, false );
					
					// Now generate instruction for this function call:
					vThisInstr.mCaller = this;
					vThisInstr.mInstructionProc = (*fli).second;	// Proc comes from One Arg Function List.
					vThisInstr.mParamCount = 1;
					vThisInstr.mParam[0] = vResult;
					vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
					vThisInstr.mResult.mLocation = vTemporaries.GetNewTemp();
					
					vCode.push_back( vThisInstr );	// Append instruction to our code.
					vResult = vThisInstr.mResult;
					vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
					
					return vResult;
				}
				else
				{
					// Get value to get property from:
					vResult = ParseValue( iterator, vCode, vTemporaries, vUserVarNames,
											vInternals, false );
					
					// Now generate instruction that retrieves the property:
					vThisInstr.mCaller = this;
					vThisInstr.mInstructionProc = HyperGetPropInstruction;
					vThisInstr.mParamCount = 2;
					vThisInstr.mParam[0] = vResult;
					vThisInstr.mParam[1].mType = MEM_LOCATION_TYPE_IMMEDIATE;
					vThisInstr.mParam[1].mValue = new TalkVarValue( new TextMunger( vOneArgFcnName ) );	// Munger is taken over by value.
					vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
					vThisInstr.mResult.mLocation = vTemporaries.GetNewTemp();
					
					vCode.push_back( vThisInstr );	// Append instruction to our code.
					vResult = vThisInstr.mResult;
					vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
					
					return vResult;
				}
				
				--iterator;// Found nothing, backtrack and let code below parse single identifier.
				break;
			}
			
			default:
				--iterator;
		}
	}
	
	// If we get here, our look-ahead resulted in nothing useful. Try looking for an Entity:
	vResult = ParseObjectDescriptorToken( iterator, vCode, vTemporaries, vUserVarNames,
											vInternals, doCreate );
	if( vResult.mType != MEM_LOCATION_TYPE_INVALID )
		return vResult;
	
	// Try variable:
	vResult = ParseVariableToken( iterator, vCode, vTemporaries, vUserVarNames,
									vInternals, doCreate );
	if( vResult.mLocation != MEM_INDEX_ACCU )	// No var of this type?
		return vResult;
	
	// Try constant:
	if( !doCreate )	// We aren't looking for a constant?
	{
		TalkConstantList::iterator		cli;
		TextMunger						vConstantName( (char*) vToken->GetValue() );
		
		vConstantName.RecalcHash();	// Makes compares faster.
		
		cli = sConstants.find( vConstantName );
		if( cli != sConstants.end() )	// We found one!
		{
			vResult.mType = MEM_LOCATION_TYPE_IMMEDIATE;
			vResult.mValue = new TalkVarValue( (long) 0 );
			(*cli).second->CopyValueTo( *vResult.mValue, true );	// Make copy of the constant.
			return vResult;
		}
	}
	
	// No variable? Try unquoted literals:
	if( sOptions[OPT_ALLOW_UNQUOTED_LITERALS] == true )
	{
		if( sOptions[OPT_VARS_AS_UNQUOTED_LITERALS] == true )
		{
			char*		vVarName;
			
			vVarName = (char*) (*iterator)->GetValue();
			
			// Create variable of that name:
			vResult = ParseVariableToken( iterator, vCode, vTemporaries, vUserVarNames,
											vInternals, true );
			
			/* Create instruction that puts variable's name into its value, so
				it is just the same as a constant to the user: */
			vThisInstr.mCaller = this;
			vThisInstr.mInstructionProc = HyperPutInstruction;
			vThisInstr.mParamCount = 1;
			vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_IMMEDIATE;
			vThisInstr.mParam[0].mValue = new TalkVarValue( new TextMunger( vVarName ) );	// TextMunger is taken over by value.
			vThisInstr.mResult = vResult;
			
			/* Now it gets complicated:
				If we put this instruction at the back of the code and it's inside a loop condition,
				it will be executed every time around, which gives us the same result as if it was a constant.
				To fix this, we need to insert this initialization instruction at the beginning of our handler.
				But if we do that, all our jump offsets become invalid. Since we don't want to loop over all
				instructions and increase the offsets every time this happens, Joker simply stores all jump
				offsets relative to the end of the init code. All we need to do when adding an initialization
				statement at the top of a handler is increase the "init code size" member of the handler.
				Goto() will automatically take care of adding that to any jump offsets to calculate the
				absolute jump address it needs. */
			vCode.push_front( vThisInstr );
			vCode.SetInitCodeSize( vCode.GetInitCodeSize() +1 );	// Make sure the init code that was moved down is still seen as init code by increasing size of init code block.
			vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
		}
		else
			vResult = ParseStringToken( iterator, vCode, vTemporaries, vUserVarNames,
										vInternals );	// Make it a string literal.
	}
	else
		throw parse_error( "Expected value, found identifier.", (*iterator)->GetStartChar(), (*iterator)->GetEndChar() );
	
	return vResult;
}


/* --------------------------------------------------------------------------------
	ParseFunctionCall:
		Parse the script that was tokenized before for a function's parameters.
	
	TAKES:
		vHandlerToCall	-	A zero-terminated C-string containing the name of the
							function to call.
		iterator		-	An iterator into the token list of the script being
							parsed.
		vCode			-	The handler to add any code to.
		vTemporaries	-	List of temporary variables where we can request a
							temporary when we need one.
		vUserVarNames	-	Names of user variables + mapping to their stack
							location. Use this to look up variable names.
		vInternals		-	List of internal variables qhich we can request or
							release as needed.
		vDest			-	The stack location of the variable we should put
							the result in. This defaults to the accu but is another
							variable if expressions are part of a value like with
							arrays.
		
	REVISIONS:
		2000-10-20	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::ParseFunctionCall( char* vHandlerToCall, HTLIterator &iterator, TalkHandler& vCode,
									TalkTempVarList &vTemporaries, TalkVarNameList &vUserVarNames,
									TalkInternalVarList& vInternals,
									TalkMemLocation vDest )
{
	HyperToken*				vToken;
	unsigned long			vParamCount = 0;
	bool					vHaveParams = true;
	HyperInstruction		vThisInstr;
	HyperToken				vEndToken( 0, 0, TOKEN_TYPE_OPERATOR, OPERATOR_CHAR_COMMA );
	TextMunger*				vHandlerName;
	
	vToken = (*iterator);
	
	if( TalkTokenizer::GetOperatorTokenKind( vToken ) == OPERATOR_CHAR_CLOSE_BRACE )
		vHaveParams = false;
	
	// Push params onto the stack:
	if( vHaveParams )	// We have params.
	{
		iterator--;	// Make sure first token of 1st param gets parsed.
		
		// Get us a temporary we can use to store our params in:
		vThisInstr.mParam[0].mLocation = vTemporaries.GetNewTemp();
		
		do
		{
			ParseExpression( iterator, vCode, vTemporaries, vUserVarNames,
							vInternals, vThisInstr.mParam[0].mLocation, &vEndToken );	// Make this stop at commas.
			
			vThisInstr.mCaller = this;
			vThisInstr.mInstructionProc = HyperPushParamInstruction;
			vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;
			//vThisInstr.mResult = 0;	// Unused.
			
			vParamCount++;	// Increase param count.
		
			vCode.push_back( vThisInstr );	// Append instruction to our code.
			vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
			
			iterator++;
			vToken = (*iterator);
		}
		while( TalkTokenizer::GetOperatorTokenKind( vToken ) == OPERATOR_CHAR_COMMA );
		
		if( TalkTokenizer::GetOperatorTokenKind( vToken ) != OPERATOR_CHAR_CLOSE_BRACE )
			throw parse_error( "Expected closing brace after function parameter list.", vToken->GetStartChar(), vToken->GetEndChar() );
	}
	
	// Now push param count onto stack:
	vThisInstr.mCaller = this;
	vThisInstr.mInstructionProc = HyperPushParamInstruction;
	vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_LONG;
	vThisInstr.mParam[0].mLong = vParamCount;	// Param count as LHS.
	//vThisInstr.mResult = 0;	// Unused.

	vCode.push_back( vThisInstr );	// Append instruction to our code.
	vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
	
	// Generate JSR instruction to jump to proper function/command message handler:
	vHandlerName = new TextMunger( vHandlerToCall );
	if( sOptions[OPT_DISTINGUISH_FCN_CMD] == true )	// Perform name mangling to distinguish functions & commands.
	{
		vHandlerName->SetOffset(0);
		vHandlerName->Insert<char>( '\t' );	// Tab is impossible to enter as part of a function name.
	}
	
	vThisInstr.mInstructionProc = HyperJSRInstruction;
	vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_IMMEDIATE;
	vThisInstr.mParam[0].mValue = new TalkVarValue( vHandlerName );	// vHandlerName is taken over by the value.
	vThisInstr.mResult = vDest;
	
	vCode.push_back( vThisInstr );	// Add JSR instruction.
	vThisInstr.ClearAllFields( false );	// Make sure params aren't disposed by destructor.
}


/* --------------------------------------------------------------------------------
	ParseExpression:
		Parse the script that was tokenized before for one expression, that is,
		one or more values concatenated by operators.
	
	TAKES:
		iterator		-	An iterator into the token list of the script being
							parsed.
		vCode			-	The handler to add any code to.
		vTemporaries	-	List of temporary variables where we can request a
							temporary when we need one.
		vUserVarNames	-	Names of user variables + mapping to their stack
							location. Use this to look up variable names.
		vInternals		-	List of internal vars we can request or release
							as needed.
		vDest			-	The stack location of the variable we should put
							the result in. This defaults to the accu but is another
							variable if expressions are part of a value like with
							arrays.
		vEndToken		-	A token at which, if it is encountered, parsing should
							stop. Usually you would pass some kind of operator here
							e.g. when parsing parameters. Most other tokens cause
							expression parsing to end anyway.
		
		NOTE:
			Operator instructions *must* have 2 arguments, no more, no less, where
			the LHS goes into mParam[0] and the RHS goes into mParam[1]. Since
			this code works by hijacking arguments from instructions, you can
			royally mess up parsing if your code works otherwise. BUT you may
			have multiple operators represented by the same instruction if you
			put a flag in mParam[2].
	
	REVISIONS:
		2000-10-22	UK		Added support for ending token.
		1999-11-29	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::ParseExpression( HTLIterator &iterator, TalkHandler& vCode,
									TalkTempVarList &vTemporaries, TalkVarNameList &vUserVarNames,
									TalkInternalVarList& vInternals,
									ValStackLocation vDest, HyperToken* vEndToken )
{
	OpPrecEnum					vLastOperatorPrecedence = OP_PREC_END,
								vThisOperatorPrecedence;
	HyperInstrProcPtr			vThisInstructionProc;
	HyperToken*					vToken;
	HyperInstruction			vThisInstr,
								vPrevInstr;
	bool						vWantToContinue = true;	// Used by switch statement to exit loop, as break doesn't work there.
	
	if( iterator == mTokens.end() )
		throw parse_error( "Expected expression, found EOF.", 0, 0 );
	
	// Get first value to operate on:
	vPrevInstr.mCaller = this;
	vPrevInstr.mParam[1] = ParseValue( iterator, vCode, vTemporaries, vUserVarNames,
										vInternals, false, vEndToken );
	
	// Generate NULL instruction with value.
	vPrevInstr.mInstructionProc = HyperStoreInstruction;	// Simply transfers RHS to result.
	vPrevInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
	vPrevInstr.mResult.mLocation = vDest;	// Accu as result address.
	
	vCode.push_back( vPrevInstr );
	vPrevInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
	
	// Now start parsing operators until we're finished:
	++iterator;
	
	for( ; (iterator != mTokens.end()) && vWantToContinue; ++iterator )
	{
		char		vTokenChar;
		        
        vToken = (*iterator);
        
		// Check whether we encountered ending token:
		if( vEndToken != NULL
			&& vToken->GetType() == vEndToken->GetType()
			&& vToken->GetValue() == vEndToken->GetValue() )
		{
			iterator--;	// Make sure ending token can be parsed by caller.
			break;
		}
		
		vTokenChar = TalkTokenizer::GetOperatorTokenKind( vToken );
		if( vTokenChar == OPERATOR_CHAR_INVALID )	// Not an operator? Expression is finished.
		{
			--iterator;	// Make sure this token gets a chance to be parsed.
			break;	// Bust it!
		}
		
		switch( vTokenChar )		// ADD NEW OPERATORS HERE!
		{
			case OPERATOR_CHAR_EQUAL:
				vThisOperatorPrecedence = OP_PREC_EQUAL;
				vThisInstructionProc = HyperCompareInstruction;
				break;
			
			case OPERATOR_CHAR_NOT_EQUAL:
				vThisOperatorPrecedence = OP_PREC_NOT_EQUAL;
				vThisInstructionProc = HyperCompareNotInstruction;
				break;
				
			case OPERATOR_CHAR_LESS_THAN:
				vThisOperatorPrecedence = OP_PREC_LT;
				vThisInstructionProc = HyperLessThanInstruction;
				break;
				
			case OPERATOR_CHAR_GREATER_THAN:
				vThisOperatorPrecedence = OP_PREC_GT;
				vThisInstructionProc = HyperGreaterThanInstruction;
				break;
			
			case OPERATOR_CHAR_CONCAT:
				vThisOperatorPrecedence = OP_PREC_CONCAT;
				vThisInstructionProc = HyperConcatInstruction;
				break;
			
			case OPERATOR_CHAR_SPACE_CONCAT:
				vThisOperatorPrecedence = OP_PREC_CONCAT_SPACE;
				vThisInstructionProc = HyperConcatSpaceInstruction;
				break;
			
			case OPERATOR_CHAR_PLUS:
				vThisOperatorPrecedence = OP_PREC_PLUS;
				vThisInstructionProc = HyperAddInstruction;
				break;
			
			case OPERATOR_CHAR_MINUS:
				vThisOperatorPrecedence = OP_PREC_MINUS;
				vThisInstructionProc = HyperSubInstruction;
				break;
			
			case OPERATOR_CHAR_MUL:
				vThisOperatorPrecedence = OP_PREC_MULTIPLY;
				vThisInstructionProc = HyperMulInstruction;
				break;
			
			case OPERATOR_CHAR_DIV:
				vThisOperatorPrecedence = OP_PREC_DIVIDE;
				vThisInstructionProc = HyperDivInstruction;
				break;
			
			case OPERATOR_CHAR_AND:
				vThisOperatorPrecedence = OP_PREC_AND;
				vThisInstructionProc = HyperLogicalAndInstruction;
				break;
			
			case OPERATOR_CHAR_OR:
				vThisOperatorPrecedence = OP_PREC_OR;
				vThisInstructionProc = HyperLogicalOrInstruction;
				break;
			
			case OPERATOR_CHAR_APOSTROPHE:
				vThisOperatorPrecedence = OP_PREC_GENITIVE;
				vThisInstructionProc = HyperGetPropInstruction;
				++iterator;
				if( (*iterator)->GetType() != TOKEN_TYPE_S_IDENTIFIER )
					throw runtime_error( "You forgot an \"s\" in your \"'s\" operator." );
				break;
			
			default:
				--iterator;
				vWantToContinue = false;
				break;
		}
		
		if( !vWantToContinue )	// Needed since we can't "break" out of the loop inside a switch.
			break;
		
		// Parse for second argument:
		TalkMemLocation		vNextValue;
		
		vNextValue = ParseValue( iterator, vCode, vTemporaries, vUserVarNames,
									vInternals, vEndToken );
		
		if( vLastOperatorPrecedence < vThisOperatorPrecedence )
		{
			vPrevInstr = vCode.back();
			vCode.pop_back();
			
			// Make our instruction into a temp:
			vThisInstr.mCaller = this;
			vThisInstr.mInstructionProc = vThisInstructionProc;
			vThisInstr.mParam[0] = vPrevInstr.mParam[1];	// prev's RHS as LHS.
			vThisInstr.mParam[1] = vNextValue;		// next as RHS.
			vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mResult.mLocation = vTemporaries.GetNewTemp();	// result to a new Temp.
			
			// Have prev instruction use this temp as argument:
			// prev's LHS stays LHS.
			vPrevInstr.mParam[1] = vThisInstr.mResult;	// Temp as prev's RHS.
			// result to Accu.
			
			// Insert this new instruction before the previous one:
			vCode.push_back( vThisInstr );
			vThisInstr.ClearAllFields( false );
			vCode.push_back( vPrevInstr );	// We just popped it, which means it's not in there anymore.
			vPrevInstr.ClearAllFields( false );
			
			// vLastOperatorPrecedence stays the same as we just inserted our statement before the previous one.
		}
		else
		{
			vThisInstr.mCaller = this;
			vThisInstr.mInstructionProc = vThisInstructionProc;
			vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_STACK;	// Accu as LHS.
			vThisInstr.mParam[0].mLocation = vDest;
			vThisInstr.mParam[1] = vNextValue;	// next as RHS.
			vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mResult.mLocation = vDest;	// result to Accu.
			
			vCode.push_back( vThisInstr );
			vThisInstr.ClearAllFields( false );	// Clear fields but don't kill values that belong to instruction on stack now.
			
			vLastOperatorPrecedence = vThisOperatorPrecedence;
		}
	}
}


/* --------------------------------------------------------------------------------
	RegisterGlobPropOrVar:
		Register a global property or special variable ("the result" etc.) that
		can be referenced by using "the" with the parser.
	
	TAKES:
		index	-	The name of the part behind the "the", e.g. "result" for the
					result.
		tpt		-	A property type, either TALK_PROP_TYPE_GLOBAL_PROP if you are
					registering a global property or TALK_PROP_TYPE_LOCAL_VARIABLE
					for a special variable.
		vName	-	The name of the property or variable. For global properties
					this is likely the same as the index (barring any short forms)
					but for special variables it usually includes the "the" and a
					space to avoid clashes with user-defined variable names.
	
	REVISIONS:
		2000-10-23	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::RegisterGlobPropOrVar( const TextMunger& index, TalkPropTypeEnum tpt,
											const TextMunger& vName )
{
	TalkPropOrVar		mPropOrVar( tpt, vName );
	
	index.RecalcHash();
	
	sGlobalProperties[index] = mPropOrVar;
}


/* --------------------------------------------------------------------------------
	* CONSTRUCTOR:
		Takes over the params passed to HyperTalk::RegisterGlobPropOrVar as tpt
		and vName and initialized the TalkPropOrVar entry with it.
	
	REVISIONS:
		2000-10-23	UK		Created.
   ----------------------------------------------------------------------------- */

TalkPropOrVar::TalkPropOrVar( TalkPropTypeEnum a, const TextMunger& b )
	: mName(b)
{
	mPropType = a;
	mName.RecalcHash();
}


/* --------------------------------------------------------------------------------
	GetResult:
		Called when this TalkPropOrVar instance has been looked up in the registry
		to generate the instructions needed to return the specified value or
		reference.
	
	TAKES:
		vCode			-	The script being parsed so we can append instructions.
		vTemporaries	-	Temporary variable list in case we need to generate
							code that uses temporaries.
		vUserVarNames	-	Variable name --> Stack index mapping list so we can
							look up or create local variables.
		vInternals		-	List of internal variables which we can request and
							release as needed.
	
	REVISIONS:
		2000-10-23	UK		Created.
   ----------------------------------------------------------------------------- */

TalkMemLocation		TalkPropOrVar::GetResult( HyperTalk* caller, TalkHandler& vCode, TalkTempVarList &vTemporaries,
												TalkVarNameList &vUserVarNames, TalkInternalVarList& vInternals )
{
#pragma unused(vCode)
#pragma unused(vTemporaries)

	TalkMemLocation		vResult;
	HyperInstruction	vThisInstr;

	switch( mPropType )
	{
		case TALK_PROP_TYPE_GLOBAL_PROP:
			vThisInstr.mCaller = caller;
			vThisInstr.mInstructionProc = HyperGetGlobalPropInstruction;
			vThisInstr.mParam[0].mType = MEM_LOCATION_TYPE_IMMEDIATE;
			vThisInstr.mParam[0].mValue = new TalkVarValue( new TextMunger( mName ) );
			vThisInstr.mResult.mType = MEM_LOCATION_TYPE_STACK;
			vThisInstr.mResult.mLocation = vTemporaries.GetNewTemp();
			
			vCode.push_back( vThisInstr );
			break;
		
		case TALK_PROP_TYPE_LOCAL_VARIABLE:
			vResult.mType = MEM_LOCATION_TYPE_STACK;
			vResult.mLocation = vUserVarNames.GetVarByName( mName, true );
			break;
	}
	
	return vResult;
}
























