/*	***************************************************************************

	PROJECT:	HyperTalk
	
	FILE:		HyperTalk.h
	
	PURPOSE:	HyperTalk interpreter.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-02-14	UK		Created.
				
	************************************************************************ */

#ifndef HYPERTALK_H
#define HYPERTALK_H

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */
   
#include	"TextMunger.h"
#include	"TalkVarValue.h"
#include	"ValueStack.h"
#include	"HyperToken.h"
#include	"TalkMemLocation.h"
#include	"TalkGlobalsList.h"
#include	"TalkInstructions.h"
#include	"TalkScriptBlock.h"
#include	<deque>
#include	<map>
#include	<stack>


#pragma mark [Constants]

/* --------------------------------------------------------------------------------
	Constants:
   ----------------------------------------------------------------------------- */

// Operator precedences:
typedef enum OpPrecEnum
{
	OP_PREC_START			= 1,	// This must be first!
	OP_PREC_AND				= 2,	// Logical and.
	OP_PREC_OR				= 2,	// Logical or.
	OP_PREC_EQUAL			= 3,	// Logical equalness.
	OP_PREC_NOT_EQUAL		= 3,	// Logical not.
	OP_PREC_LT				= 4,	// <
	OP_PREC_GT				= 4,	// >
	OP_PREC_CONCAT			= 5,	// &
	OP_PREC_CONCAT_SPACE	= 5,	// &&
	OP_PREC_PLUS			= 6,	// Addition (+)
	OP_PREC_MINUS			= 6,	// Subtraction (-)
	OP_PREC_MULTIPLY		= 7,	// Multiplication (*)
	OP_PREC_DIVIDE			= 7,	// Division (/)
	OP_PREC_GENITIVE		= 8,	// "'s" genitive-style constructs for properties. Should be pretty high.
	OP_PREC_END			// This must be last!
} OpPrecEnum;

// Chunk types:
enum
{
	CHUNK_TYPE_INVALID	= 0,	// Used in identifying bad ends in parsing for "number of chars" etc.
	CHUNK_TYPE_CHARACTER,		// Characters. This is 1 byte for Ascii, or 2 or 4 bytes for different variants of Unicode.
	CHUNK_TYPE_ITEM,			// Items, using the item delimiter.
	CHUNK_TYPE_LINE,			// Lines, like items but uses CR or LF as the delimiter.
	CHUNK_TYPE_WORD,			// Words, separator is any number of returns or spaces.
	CHUNK_TYPE_BYTE				// Bytes. A single byte as opposed to a character, which, since Unicode, may be of different size.
};


/* Compiler options:
	These constants are indexes into our array of options. */
enum
{
	OPT_ALLOW_UNQUOTED_LITERALS = 0,	/* This entry is true or false to turn on/off support
											for unquoted string literals. Default is TRUE (HC). */
	OPT_DISTINGUISH_FCN_CMD,			/* This entry is true or false to turn on/off distinction
											of message and function handlers. (HC uses TRUE here) */
	OPT_VARS_AS_UNQUOTED_LITERALS,		/* True if you want unquoted literals to be created as a
											variable with its name in it, false if you really want
											them to be constants. (HC uses TRUE here) */
	OPT_ESCAPE_CHARS_IN_STRINGS,		/* True if you want to support C-style escaped characters
											in strings to insert returns and quotes in strings. If
											this is on, returns in multi-line string tokens will be
											turned into spaces. (HC uses FALSE here) */
	OPTIONS_LAST	// Must be last, used for array size.
};


#pragma mark [Data types]

/* --------------------------------------------------------------------------------
	Data types:
   ----------------------------------------------------------------------------- */

class	TalkTempVarList;		// Forward.
class	TalkVarNameList;		// Forward.
class	TalkInternalVarList;	// Forward.
class	HyperTalk;				// Forward.

/* The "compiled" code for each handler in a script is kept in a TalkHandler instance.
	
	Note that to allow for unquoted string literals that turn into variables later
	down the script, mInitCodeSize contains the size of the initialization code
	added to the top of the script that sets the value of such a variable to their
	name.
	
	All jump addresses are specified relative to the end of the initialization
	code block, so they don't need to change when more initialization code is
	added. */

class TalkHandler : public deque<HyperInstruction>
{
protected:
	bool			mIsFunction;		// TRUE if fcn, FALSE if message.
	size_t			mStackSpaceNeeded;	// Stack space (in variables) this handler needs.
	size_t			mInitCodeSize;		// Size (in instructions) of the init code added at the top of this handler.

public:
	TalkHandler() : deque<HyperInstruction>() { mStackSpaceNeeded = 0; mInitCodeSize = 0; };
	
	size_t		GetStackNeed()				{ return mStackSpaceNeeded; };
	void		SetStackNeed( size_t n )	{ mStackSpaceNeeded = n; };
	
	size_t		GetInitCodeSize()			{ return mInitCodeSize; };
	void		SetInitCodeSize( size_t n )	{ mInitCodeSize = n; };
	
	bool		GetIsFunction()				{ return mIsFunction; };
	void		SetIsFunction( bool n )		{ mIsFunction = n; };
};
typedef map<TextMunger,TalkHandler>			TalkScript;
typedef map<TextMunger,HyperInstrProcPtr>	TalkFunctionList;		// List of function instruction procs.

/* The constants and classes below are used for the command registry. Just fill out a TalkCommandEntry
	struct with the appropriate fields and it will generate the appropriate TalkInstruction once you
	have registered it with the interpreter.
	
	HyperTalk purists:	(Possible "FIX ME!" here)
	Joker currently generates an instruction for built-in commands. This is not what HyperCard does.
	If you want HyperTalk-like behaviour, you have to change the HyperInstrProcPtr in TalkCommandEntry
	into a TextMunger that holds the name of the message to send and change
	HyperTalk::ParseBuiltInCommand() to generate push instructions and a JSR instead of a custom
	instruction. */

typedef enum TalkEntryParamTypeEnum
{
	TALK_PARAM_TYPE_INVALID = 0,	// Use this to skip direct argument or to designate syntactic sugar.
	TALK_PARAM_TYPE_EXPRESSION,		// Any expression. Not editable.
	TALK_PARAM_TYPE_VALUE,			/* A value that isn't editable. Used e.g. for
										entry <n> of <variable> to avoid a precedence mess. */
	TALK_PARAM_TYPE_CONTAINER,		// A value that is editable, e.g. variables or entries etc.
	TALK_PARAM_TYPE_QUALIFIER,		// If the label for this param is present, the param is true, else false.
	TALK_PARAM_TYPE_IDENTIFIER		// Any unquoted identifier.
} TalkEntryParamTypeEnum;

// Possible values for mOptional:
#define TALK_PARAM_IS_OPTIONAL	true
#define TALK_PARAM_IS_REQUIRED	false

// Values with special meaning for mDestParam:
#define TALK_PARAM_RESULT		-1			// Store in result.

// Values with special meaning for mResultParam:
// TALK_PARAM_RESULT			-1			// Store in "the result". Will take care "the result" is defined.
#define TALK_PARAM_IT			-2			// Store in "it". Will take care "it" is defined.
#define TALK_PARAM_IGNORE		-3			// Ignore function result.

struct TalkCommandEntryParam
{
	TokenTypeEnum			mTokenType;	/* The label to identify this param. This allows arbitrary
											order or optional params. For direct argument this is the
											command name. Specify TOKEN_TYPE_INVALID here for unlabeled
											parameters. */
	TalkEntryParamTypeEnum	mParamType;	/* Invalid to omit direct argument, else the kind of value to
											Be parsed after above token has been encountered. */
	short					mDestParam;	// What param of the instruction to store this value in (one-based).
	bool					mOptional;	/* Set to empty if this parameter is missing and don't complain?
											Be careful here: This is set to an empty string constant, so
											changing a missing optional parameter will likely result in odd
											behaviour. */
	HyperInstrProcPtr		mCommandInstruction;	/* The instruction to generate for this. Set to NULL to
														use the one most recently set. The command entry *must*
														set this, all other params should only set this if they
														cause an entirely different command to be executed. */
};

struct TalkCommandEntry
{
	short					mResultParam;			// If you want the instruction result to be stored in one of the parameters (e.g. for a "replace" command), set this to the param number. To ignore it, use TALK_PARAM_IGNORE.
	long					mRefCon;				/* Custom data assigned to the refCon of the generated instruction. */
	short					mParamCount;			// Number of parameters (including direct argument).
	TalkCommandEntryParam	mParam[8];				// One entry per parameter. mParam[0] is direct argument, i.e. its token is ignored.
	/* Note: Since this is used as a template for creating a TalkInstruction, no valid command may
		generate more than TALK_INSTR_MAX_PARAMS parameters! */
};

typedef map<TokenTypeEnum,TalkCommandEntry*>	TalkBuiltInCmdList;			// List of commands with english-like syntax.
typedef map<TokenTypeEnum,TalkCommandEntry*>	TalkObjectDescriptorList;	// List of identifiers and "parameters" for parsing an object type.

/* The constants and classes below are used by the global property and special variable registry,
	which contains everything that has the form of "the <identifier>" without an "of" following it.
	
	The class HyperTalk has a method to create TalkPropOrVar objects for certain variable/property
	names and add them to the registry in one go. */

typedef enum TalkPropTypeEnum
{
	TALK_PROP_TYPE_GLOBAL_PROP = 0,
	TALK_PROP_TYPE_LOCAL_VARIABLE
} TalkPropTypeEnum;

class TalkPropOrVar
{
protected:
	TalkPropTypeEnum	mPropType;
	TextMunger			mName;
	
public:
	TalkPropOrVar()	{};
	TalkPropOrVar( TalkPropTypeEnum a, const TextMunger& b );

	TalkMemLocation		GetResult( HyperTalk* caller, TalkHandler& vCode, TalkTempVarList &vTemporaries,
									TalkVarNameList &vUserVarNames, TalkInternalVarList& vInternals );
};
typedef map<TextMunger,TalkPropOrVar>		TalkGlobalPropertyList;	// List of global properties or special variables.

/* Entry used by "repeat" when it is parsed to keep track of point to jump
	back to etc. for (nested) repeats. */
typedef struct RepeatStackEntry
{
	TalkMemLocation*	mIfDestField;		// Here the "end" stuffs its location so the "if" can exit.
	unsigned long		mJumpBackAddress;	// This is the address of the "if" instruction so the "end" can jump back to the start.
	unsigned long		mStepWidth;			// Step width for "repeat with x = 1 to 5"-kind statements. If this is 0 it is another kind of repeat.
	ValStackLocation	mCounter;			// Stack location of counter variable (BP relative). Only valid if mStepWidth != 0.
} RepeatStackEntry;

/* Entry used by "if" when it is parsed to keep track of jump addresses of
	"else" etc. for (nested) ifs. */
typedef struct IfStackEntry
{
	TalkMemLocation*	mIfDestField;		// Address of the "if"'s jump address so "else" or "end if" can modify it.
	bool				mOneLineIf;			// TRUE if this is a "one-line-if" that ends on a line break and doesn't need an "end if".
	bool				mIfNotElse;			// Is this entry for an "if" (true) or an "else" (false)?
} IfStackEntry;

// Call results are the return values from SendMessage() and the likes:
typedef enum
{
	CALL_RESULT_INTERCEPTED,	// A handler caught this message.
	CALL_RESULT_PASSED,			// A handler caught this message but passed it on.
	CALL_RESULT_EXITED,			// A handler caught this message but exited.
	CALL_RESULT_EXCEPTION		// An exception (error) occurred during execution of a handler.
} TalkCallResult;


typedef map<TextMunger,TalkValue*>	TalkConstantList;	// List of names and values for parsing constants.

typedef map<OperatorTypeEnum,HyperInstrProcPtr>	TalkOperatorList;

// Script blocks are eg. handler definitions, all kinds of parts that make up a script:
typedef map<TokenTypeEnum,TalkScriptBlock*>		TalkScriptBlockList;

#pragma mark [Class Declaration]

/* --------------------------------------------------------------------------------
	Class declaration:
   ----------------------------------------------------------------------------- */

// The interpreter itself:
class	HyperTalk
{
friend void	HyperIfInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );	// Accesses Goto().

protected:
	unsigned long			mRefCount;			// Reference count to allow shared use of this object without killing data that's still in use.
	TalkScript				mScript;			// The actual script with all our handlers compiled in it.
	HyperTokenList			mTokens;			// The list of tokens generated by the tokenizer.
	TalkHandler::iterator*	mCurrIterator;		// The iterator to the instruction currently being executed.
	TalkHandler*			mCurrHandler;		// The handler currently executing.
	HyperTalk*				mBossInHierarchy;	// Our boss in the message-passing hierarchy, to which we pass messages we don't handle.
	TalkEntity*				mAssociatedObject;	// Object containing this script, or NULL if there is none (anymore?).
	
	static bool						sInited;				// Did we already init the static members?
	static TalkGlobalsList			sGlobals;				// List of all globals.
	static long						sOptions[OPTIONS_LAST];	// Compiler flags.
	static TalkFunctionList			sOneArgFunctions;		// List of built-in functions that take only one argument like abs() etc.
	static TalkFunctionList			sBuiltInFunctions;		// List of all built-in functions, except one-arg ones.
	static TalkFunctionList			sBuiltInMessages;		// List of all command messages with user-command syntax we support.
	static TalkBuiltInCmdList		sBuiltInCommands;		// List of all built-in commands with english-like syntax.
	static TalkGlobalPropertyList	sGlobalProperties;		// List of global props & special variables.
	static TalkObjectDescriptorList	sObjectDescriptors;		// List of syntaxes for entity parsing instructions.
	static TalkConstantList			sConstants;				// List of constants and what value they equal.
	static TalkOperatorList			sUnaryOperators;		// Unary operators like the prefix minus sign.
    static TalkScriptBlockList		sScriptBlocks;			/* Objects that know how to parse portions of a script. There are e.g.
                                                                script blocks that parse function handlers and message handlers. */

public:
	HyperTalk();
	virtual ~HyperTalk();
	
	void				Retain()	{ mRefCount++; };
	void				Release()	{ if( --mRefCount == 0 ) delete this; };
	
	void				Tokenize( TextMunger& inScript );
	void				Parse();
	void				ParseAsHandler( const TextMunger& inName );
	TalkCallResult		SendMessage( const TextMunger& inName, ValueStack *vStack,
									TalkMemLocation vResult, bool isFcn, bool eatMessage = false );
	TalkCallResult		SendMessage( const TextMunger& inHandlerName, bool eatMessage = false );	// Command handler w/o params.
	TalkCallResult		Run( const TextMunger& inHandlerName, ValueStack &vStack, TalkMemLocation vResult, bool eatMessage = false );	// Push params + count on stack before calling this!
	TalkCallResult		CallBuiltInFunction( const TextMunger& inName, ValueStack &vStack,
											TalkMemLocation vResult, bool eatMessage );
	void				GetGlobal( TalkValue* &vValue, TextMunger& inGlobName, bool doCreate );
	TalkHandler*		GetCurrentHandler()		{ return mCurrHandler; };
	
    HyperTokenList&		GetTokenList()			{ return mTokens; };
    
	virtual HyperTalk*	GetBossInHierarchy()				{ return mBossInHierarchy; };
	virtual void		SetBossInHierarchy( HyperTalk* b )	{ mBossInHierarchy = b; };
	
	void				PrintAllTokens();	// Debugging only.
	void				PrintAllHandlers();	// Debugging only.
	void				PrintOneHandler( TalkHandler &vCode );	// Debugging only.
	
	void				SetAssociatedObject( TalkEntity* e )	{ mAssociatedObject = e; };
	TalkEntity*			GetAssociatedObject()					{ return mAssociatedObject; };
	
	virtual void		EmitHandlerNotFoundErr( bool isFcn, TextMunger& vRealHandlerName,
												ValueStack& vStack, TalkMemLocation vResult );
    void				ParseHandler( HTLIterator& iterator, bool isFcn, long &vCurrLineNumber );

	static void		ForceInitialize();	// Use when you're not sure static initializers work.
	static void		SetOption( long inOption, long inValue );	// Turn on/off compiler flags.
	static void		InitializeVocabulary();	// Init all language keywords. Called automatically by constructor if you don't do it.
	static void		TryToInitOptions();	// Init options to defaults. Called automatically by the constructor.
	static void		RegisterBuiltInCommands();		// Called by constructor. Defined in TalkBuiltInCommands.cpp.
	static void		RegisterObjectDescriptors();	// Called by constructor. Defined in JokerDummy.cpp.
	static void		RegisterOneArgFunction( const TextMunger& vFcnName, HyperInstrProcPtr vInstructionProc );
	static void		RegisterBuiltInFunction( const TextMunger& vFcnName, HyperInstrProcPtr vInstructionProc );
	static void		RegisterBuiltInMessage( const TextMunger& vFcnName, HyperInstrProcPtr vInstructionProc );
	static void		RegisterBuiltInCommand( TalkCommandEntry* vCommand );
	static void		RegisterObjectDescriptor( TalkCommandEntry* vDescriptor );
	static void		RegisterGlobPropOrVar( const TextMunger& index, TalkPropTypeEnum tpt, const TextMunger& vName );
	static void		RegisterConstant( const TextMunger& vName, TalkValue* vValue );
	static void		RegisterUnaryOperator( OperatorTypeEnum vOpType, HyperInstrProcPtr vInstructionProc );
    static void		RegisterScriptBlock( TokenTypeEnum vStartIdentifier, TalkScriptBlock* block );

protected:
	// Tokenizing:
	void			ProcessToken( TextMunger& inScript, size_t start, size_t end );
	void			MakeStringToken( TextMunger& inScript, size_t start, size_t end, TokenTypeEnum type = TOKEN_TYPE_STRING );
	void			MakeNumberToken( TextMunger& inScript, size_t start, size_t end );
	void			AddToken( size_t start, size_t end, TokenTypeEnum tokenType, long tokenValue = NULL );
	
	// Parsing:
	void			ParseHandlerBody( TalkHandler& vCode, TextMunger& vNameOfHandler,
										HTLIterator &iterator,
										TalkTempVarList& vTemporaries,
										TalkVarNameList& vUserVarNames,
										TalkInternalVarList& vInternals,
										long& vCurrLineNum );
	void			ParserHandleEndIf( HTLIterator& currToken, TalkHandler& vCode, stack<IfStackEntry> &vIfStack );
	void			ParserMakeLineEndInstruction( long& vCurrLineNum, TalkHandler& vCode,
												HTLIterator& iterator, stack<IfStackEntry>& ifStack,
												stack<RepeatStackEntry>& repStack );
	void			ParseExpression( HTLIterator &iterator, TalkHandler &vCode, TalkTempVarList &vTemporaries,
									TalkVarNameList& vUserVarNames, TalkInternalVarList& vInternals,
									ValStackLocation vDest = MEM_INDEX_ACCU, HyperToken* vEndToken = NULL );
	TalkMemLocation	ParseValue( HTLIterator &iterator, TalkHandler& vCode, TalkTempVarList& vTemporaries,
									TalkVarNameList& vUserVarNames, TalkInternalVarList& vInternals,
									bool doCreate = false, HyperToken* vEndToken = NULL );
	TalkMemLocation	ParseStringToken( HTLIterator &iterator, TalkHandler& vCode,
										TalkTempVarList& vTemporaries, TalkVarNameList& vUserVarNames,
										TalkInternalVarList& vInternals );
	void			ParseFunctionCall( char* vHandlerToCall, HTLIterator &iterator, TalkHandler& vCode,
									TalkTempVarList &vTemporaries, TalkVarNameList &vUserVarNames,
									TalkInternalVarList& vInternals, TalkMemLocation vDest );
	TalkMemLocation	ParseNumberOfFunction( HTLIterator &iterator, TalkHandler& vCode,
											TalkTempVarList& vTemporaries, TalkVarNameList& vUserVarNames,
											TalkInternalVarList& vInternals, bool doCreate,
											HyperToken* vEndToken, bool &vFoundSomething );
	TalkMemLocation	ParseObjectDescriptorToken( HTLIterator &iterator, TalkHandler& vCode,
												TalkTempVarList& vTemporaries,
												TalkVarNameList& vUserVarNames,
												TalkInternalVarList& vInternals, bool doCreate );
	TalkMemLocation	ParseIntegerValue( HTLIterator &iterator, TalkHandler& vCode,
										TalkTempVarList& vTemporaries,
										TalkVarNameList& vUserVarNames,
										TalkInternalVarList& vInternals, bool doCreate,
										HyperToken* vEndToken );
	TalkMemLocation	ParseVariableToken( HTLIterator &iterator, TalkHandler& vCode,
										TalkTempVarList& vTemporaries, TalkVarNameList& vUserVarNames,
										TalkInternalVarList& vInternals, bool doCreate );
	TalkMemLocation	ParseAnyIdentifier( HTLIterator &iterator, TalkHandler& vCode,
										TalkTempVarList& vTemporaries, TalkVarNameList& vUserVarNames,
										TalkInternalVarList& vInternals, bool doCreate,
										HyperToken* vEndToken );
	void			ParseBuiltInCommand( HTLIterator &iterator, TalkHandler& vCode,
										TalkTempVarList& vTemporaries,
										TalkVarNameList& vUserVarNames,
										TalkInternalVarList& vInternals );
	void			ParsePutCommand( HTLIterator &iterator, TalkHandler& vCode,
										TalkTempVarList& vTemporaries,
										TalkVarNameList& vUserVarNames,
										TalkInternalVarList& vInternals );

	// Running:
	void			Goto( long vAddress );	// Address is index into command array minus size of initialization block.
};


#pragma mark [Prototypes]

/* --------------------------------------------------------------------------------
	Prototypes:
   ----------------------------------------------------------------------------- */





#endif /*HYPERTALK_H*/



