/*	***************************************************************************

	PROJECT:	HyperToken
	
	FILE:		HyperToken.h
	
	PURPOSE:	One token in a HyperTalk script.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-02-14	UK		Created.
				
	************************************************************************ */

#ifndef HYPERTOKEN_H
#define HYPERTOKEN_H

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */
   
#include	"TextMunger.h"
#include	"TalkVarValue.h"
#include	"ValueStack.h"
#include	<vector>
#include	<deque>
#include	<map>


#pragma mark [Constants]

/* --------------------------------------------------------------------------------
	Constants:
   ----------------------------------------------------------------------------- */

// Possible token types:
enum TokenTypeActualEnum
{
	TOKEN_TYPE_INVALID	=	0,		// Used for special meanings wherever a token is specified.
	TOKEN_TYPE_STRING,				// A quoted string literal. Pointer to string w/o quotes is in value.
	TOKEN_TYPE_IDENTIFIER,			// An unknown identifier (constant, handler, variable name etc.). Pointer to string is in value.
	TOKEN_TYPE_OPERATOR,			// Any operator. Operator char is in value.
	TOKEN_TYPE_INTEGER,				// Any integer number.
	TOKEN_TYPE_NEWLINE,				// A newline character.
	TOKEN_TYPE_START_HANDLER,		// "on" identifier used to start handlers.
	TOKEN_TYPE_END,					// "end" identifier used to end lots of things.
	TOKEN_TYPE_START_FUNCTION,		// "function" identifier used to start functions.
	TOKEN_TYPE_PUT_COMMAND,			// "put" identifier to indicate the command "put".
	TOKEN_TYPE_INTO_IDENTIFIER,		// "into" identifier used to indicate 2nd param for "put".
	TOKEN_TYPE_IF_IDENTIFIER,		// "if" identifier used to indicate "if" structure.
	TOKEN_TYPE_THEN_IDENTIFIER,		// "then" identifier used to indicate start of commands for "if".
	TOKEN_TYPE_ELSE_IDENTIFIER,		// "else" identifier used to indicate start of commands for "if".
	TOKEN_TYPE_REPEAT_IDENTIFIER,	// "repeat" identifier used to indicate "repeat" command.
	TOKEN_TYPE_WHILE_IDENTIFIER,	// "while" identifier used to indicate condition to "repeat" command.
	TOKEN_TYPE_DOWN_IDENTIFIER,		// "down" identifier used to indicate loop direction for loops with counters.
	TOKEN_TYPE_PARAM_IDENTIFIER,	// "param" identifier used to get a function's parameters.
	TOKEN_TYPE_ENTRY_IDENTIFIER,	// "entry" identifier used to access array entries in variables.
	TOKEN_TYPE_OF_IDENTIFIER,		// "of" identifier used to indicate various params.
	TOKEN_TYPE_STDIN_IDENTIFIER,	// "stdin"/"standardInput" token used to get user input from console.
	TOKEN_TYPE_RETURN_IDENTIFIER,	// "return" token used to insert return characters in strings.
	TOKEN_TYPE_GLOBAL_IDENTIFIER,	// "global" identifier used for declaring global variables.
	TOKEN_TYPE_CHAR_IDENTIFIER,		// "char"/"character" identifier used for chunks.
	TOKEN_TYPE_WORD_IDENTIFIER,		// "word" identifier used for chunks.
	TOKEN_TYPE_ITEM_IDENTIFIER,		// "item" identifier used for chunks.
	TOKEN_TYPE_LINE_IDENTIFIER,		// "line" identifier used for chunks.
	TOKEN_TYPE_TO_IDENTIFIER,		// "to" identifier used for chunk ranges, counted loops and the "set" command.
	TOKEN_TYPE_THE_IDENTIFIER,		// "the" identifier used to indicate properties, out-of-path function calls and special variables.
	TOKEN_TYPE_RESULT_IDENTIFIER,	// "result" identifier used to indicate the variable "the result".
	TOKEN_TYPE_KEYS_IDENTIFIER,		// "keys" identifier used for the "the keys" built-in function.
	TOKEN_TYPE_NUMBER_IDENTIFIER,	// "number" identifier used for the "the number of <chunkType>s of" construct.
	TOKEN_TYPE_CHARS_IDENTIFIER,	// "chars"/"characters" identifier used for the "number of chars" construct.
	TOKEN_TYPE_WORDS_IDENTIFIER,	// "words" identifier used for the "number of words" construct.
	TOKEN_TYPE_ITEMS_IDENTIFIER,	// "items" identifier used for the "number of items" construct.
	TOKEN_TYPE_LINES_IDENTIFIER,	// "lines" identifier used for the "number of lines" construct.
	TOKEN_TYPE_URL_IDENTIFIER,		// "url" identifier used for file and URL access.
	TOKEN_TYPE_TRUE_IDENTIFIER,		// "true" identifier used for booleans.
	TOKEN_TYPE_FALSE_IDENTIFIER,	// "false" identifier used for booleans.
	TOKEN_TYPE_SET_COMMAND,			// "set" identifier used for "set" command.
	TOKEN_TYPE_ADD_COMMAND,			// "add" identifier used for "add" command.
	TOKEN_TYPE_SUBTRACT_COMMAND,	// "subtract" identifier used for "subtract" command.
	TOKEN_TYPE_MULTIPLY_COMMAND,	// "multiply" identifier used for "multiply" command.
	TOKEN_TYPE_DIVIDE_COMMAND,		// "divide" identifier used for "divide" command.
	TOKEN_TYPE_FROM_IDENTIFIER,		// "from" identifier used for "subtract" command.
	TOKEN_TYPE_BY_IDENTIFIER,		// "by" identifier used for "divide" & "multiply" commands.
	TOKEN_TYPE_REPLACE_COMMAND,		// "replace" identifier used for "replace" command.
	TOKEN_TYPE_WITH_IDENTIFIER,		// "with" identifier used for "replace", "request" & "reply" commands and counted loops.
	TOKEN_TYPE_IN_IDENTIFIER,		// "in" identifier used for "replace" command.
	TOKEN_TYPE_REPLY_COMMAND,		// "reply" identifier used for "reply" command.
	TOKEN_TYPE_KEYWORD_IDENTIFIER,	// "keyword" identifier used for "reply" & "request" commands.
	TOKEN_TYPE_ERROR_IDENTIFIER,	// "error" identifier used for "reply" command.
	TOKEN_TYPE_PASS_COMMAND,		// "pass" identifier used for "pass" command.
	TOKEN_TYPE_REQUEST_COMMAND,		// "request" identifier used for "request" command
	TOKEN_TYPE_APPLEEVT_IDENTIFIER,	// "ae" or "appleEvent" identifier used for "request" command.
	TOKEN_TYPE_LOCAL_COMMAND,		// "local" command for declaring variables.
	TOKEN_TYPE_CARD_IDENTIFIER,		// "card"/"cd" identifier used for referencing card-relative objects and cards.
	TOKEN_TYPE_BKGND_IDENTIFIER,	// "background", "bkgnd", "bkgd" or "bg" identifier used for referencing bg-relative objects and backgrounds.
	TOKEN_TYPE_ID_IDENTIFIER,		// "id" identifier used for referencing objects by ID number.
	TOKEN_TYPE_STACK_IDENTIFIER,	// "stack" identifier used for referencing stacks.
	TOKEN_TYPE_ASK_COMMAND,			// "ask" command for requesting textual information from the user.
	TOKEN_TYPE_THIS_IDENTIFIER,		// "this" identifier to refer to the current card or stack.
	TOKEN_TYPE_DEFINE_COMMAND,		// "define" command used to create user properties.
	TOKEN_TYPE_CREATE_COMMAND,		// "create" command used to make new stacks, menus etc.
	TOKEN_TYPE_MENU_IDENTIFIER,		// "menu" identifier used for referencing menu and menu item objects.
	TOKEN_TYPE_PART_IDENTIFIER,		// "part" identifier used for referencing card and background parts.
	TOKEN_TYPE_MENUITEM_IDENTIFIER,	// "menuitem" identifier used for referencing menu item objects.
	TOKEN_TYPE_IS_IDENTIFIER,		// "is" identifier used as a more readable alternative to the "=" operator.
	TOKEN_TYPE_DO_COMMAND,			// "do" identifier used for "do" command.
	TOKEN_TYPE_DEBUG_COMMAND,		// "debug" identifier used for "debug checkpoint" command.
	TOKEN_TYPE_CHECKPOINT_IDENTIFIER,	// "checkpoint" identifier used for "debug checkpoint" command.
	TOKEN_TYPE_WINDOW_IDENTIFIER,	// "window" identifier used for referencing windows.
	TOKEN_TYPE_QUIT_COMMAND,		// "quit" identifier used for "quit" command.
	TOKEN_TYPE_CENTER_COMMAND,		// "center" identifier used for "center <object>" command.
	TOKEN_TYPE_CLOSE_COMMAND,		// "close" identifier used for "close <object>" command.
	TOKEN_TYPE_SHOW_COMMAND,		// "show" identifier used for "show <object>" command.
	TOKEN_TYPE_HIDE_COMMAND,		// "hide" identifier used for "hide <object>" command.
	TOKEN_TYPE_DISABLE_COMMAND,		// "disable" identifier used for "disable <object>" command.
	TOKEN_TYPE_DOCUMENT_IDENTIFIER,	// "document" identifier used for referencing document windows.
	TOKEN_TYPE_PALETTE_IDENTIFIER,	// "palette" identifier used for referencing floating windows.
	TOKEN_TYPE_EXIT_COMMAND,		// "exit" identifier used for the exit to top/exit handler commands.
	TOKEN_TYPE_SETWINDOW_COMMAND,	// "setWindow" command used to specify the window for short object descriptors.
	TOKEN_TYPE_NOT_IDENTIFIER,		// "not" identifier used for "is not" and unary "not" operators.
	TOKEN_TYPE_S_IDENTIFIER,		// "s" identifier used for genitive-style property syntax ("btn 5's loc").
	TOKEN_TYPE_ME_IDENTIFIER,		// "me" identifier used for referring to the object containing the current script.
	TOKEN_TYPE_BUTTON_IDENTIFIER,	// "button" identifier used for referencing button parts.
	TOKEN_TYPE_SELECT_COMMAND,		// "select" identifier used for selecting parts.
	TOKEN_TYPE_DELETE_COMMAND,		// "delete" identifier used for getting rid of parts and text ranges.
	TOKEN_TYPE_AND_IDENTIFIER,		// "and" identifier used for logical "and" operator.
	TOKEN_TYPE_OR_IDENTIFIER,		// "or" identifier used for logical "or" operator.
	TOKEN_TYPE_BYTE_IDENTIFIER,		// "byte" identifier used for chunk expressions.
	TOKEN_TYPE_BYTES_IDENTIFIER,	// "bytes" identifier used for chunk expressions.
	TOKEN_TYPE_TRY_IDENTIFIER,		// "try" identifier used for exception catching.
	TOKEN_TYPE_CATCH_IDENTIFIER,	// "catch" identifier used for exception catching.
	TOKEN_TYPE_FINALLY_IDENTIFIER,	// "finally" identifier used for exception catching.
	TOKEN_TYPE_THROW_COMMAND,		// "throw" identifier used for the "throw" command.
	TOKEN_TYPE_LAST					// Must be last, used for getting IDs for client-defined tokens.
};
typedef long TokenTypeEnum;

// Operator characters:
typedef enum OperatorTypeEnum
{
	OPERATOR_CHAR_INVALID		= 0,	// Only used to indicate non-operator tokens.
	OPERATOR_CHAR_PLUS			= '+',
	OPERATOR_CHAR_MINUS			= '-',	// Duplicate for comment.
	OPERATOR_CHAR_MUL			= '*',
	OPERATOR_CHAR_DIV			= '/',
	OPERATOR_CHAR_OPEN_BRACE	= '(',
	OPERATOR_CHAR_CLOSE_BRACE	= ')',
	OPERATOR_CHAR_OPEN_ABRACE	= '[',
	OPERATOR_CHAR_CLOSE_ABRACE	= ']',
	OPERATOR_CHAR_MOD			= '%',
	OPERATOR_CHAR_LESS_THAN		= '<',	// May add =, < or >
	OPERATOR_CHAR_GREATER_THAN	= '>',	// May add = or >
	OPERATOR_CHAR_EQUAL			= '=',
	OPERATOR_CHAR_CONCAT		= '&',	// Duplicate for concat +space.
	OPERATOR_CHAR_COMMA			= ',',
	OPERATOR_CHAR_APOSTROPHE	= '\'',
	// Dummy operators for multi-char operators (only used in operator token's value):
	OPERATOR_CHAR_GT_EQUAL		= 'a',	// >=
	OPERATOR_CHAR_LT_EQUAL		= 'b',	// <=
	OPERATOR_CHAR_NOT_EQUAL		= 'c',	// <>
	OPERATOR_CHAR_BSL			= 'd',	// <<
	OPERATOR_CHAR_BSR			= 'e',	// >>
	OPERATOR_CHAR_SPACE_CONCAT	= 'f',	// &&
	OPERATOR_CHAR_NOT			= 'g',	// "not"
	OPERATOR_CHAR_AND			= 'h',	// "and"
	OPERATOR_CHAR_OR			= 'i'	// "or"
} OperatorTypeEnum;


#pragma mark [Class Declaration]

/* --------------------------------------------------------------------------------
	Class declaration:
   ----------------------------------------------------------------------------- */

// One token in our script's tokenized version:
class	HyperToken
{
protected:
	TokenTypeEnum	mTokenType;		// Token type.
	long			mTokenValue;	// Token value (depends on type).
	size_t			mStartInText;	// Start of token in text (used for reporting errors).
	size_t			mEndInText;		// End of token in text (used for reporting errors).

public:
	HyperToken( size_t start, size_t end, TokenTypeEnum type, long val = NULL )	{ mTokenType = type; mTokenValue = val; mStartInText = start; mEndInText = end; };
	
	TokenTypeEnum	GetType()		{ return mTokenType; };
	long			GetValue()		{ return mTokenValue; };
	size_t			GetStartChar()	{ return mStartInText; };
	size_t			GetEndChar()	{ return mEndInText; };
	
	void			PrintToken();		// Debugging only.
	
// For use of the map class (not suitable for comparing tokens):
	virtual	bool		operator== ( const HyperToken& t ) const;
	virtual	bool		operator< ( const HyperToken& t ) const;
	virtual	bool		operator> ( const HyperToken& t ) const;
};


typedef vector<HyperToken*>			HyperTokenList;
typedef HyperTokenList::iterator	HTLIterator;








#endif /*HYPERTOKEN_H */