/*	***************************************************************************

	PROJECT:	Joker
	
	FILE:		common/JokerMain.cpp
	
	PURPOSE:	Main entry point for Joker.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		2000-10-26	UK		Removed Mac-specific details.
		1999-06-22	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"JokerMain.h"
#include	"TextMunger.h"
#include	"HyperTalk.h"
#include	"TalkVarValue.h"
#include	"parse_error.h"
#include	<iostream>
#include	<ctime>

#pragma mark [Globals]

/* --------------------------------------------------------------------------------
	Globals/Static Variables:
   ----------------------------------------------------------------------------- */

// Put these li'l critters here.


#pragma mark -
#pragma mark [Implementation]


/* --------------------------------------------------------------------------------
	JokerMain:
		Entry point for Joker. Since some OSs do not pass any parameters to an
		application's entry point, this is not named main. Just call this function
		from your actual main entry point.
		
	PARAMETERS TO JOKER:
		file Name	-	The name of a source file to execute.
        		
		In addition to that, you can turn on or off one of the following compiler
		flags:
		
		-u			-	Activate or deactivate support for unquoted string
						literals. This means, if an unknown identifier is
						encountered while looking for a value, it will be
						considered a string constant.
		-v			-	If on, Joker will generate a variable when it
						encounters an unquoted string literal of the same name
						as the identifer and will set the variable's value to
						the name of the identifier. This means that if e.g.
						in the body of a loop some command uses this identifier
						as a variable, it will be re-interpreted to be a
						variable. Turn this off to generate a real constant
						instead.
		-f			-	Have Joker distinguish between command and function
						message handlers. If this is off, you can call a
						user-defined function using command syntax and vice
						versa.
		-e			-	Have Joker support escaped character sequences inside
						string constants. Default is off.
        -i			-	Have Joker output version information and Copyright.
                        This isn't followed by "on" or "off".
		
		Examples:
			Joker myscript.txt -u on
			Joker otherscript.txt -u off
			Joker script.txt -u			-- this is the same as -u on
            Joker -u on -? script.txt
        
        It is unimportant where you put the file name in the parameter list, as
        long as it isn't immediately following a "-u" or between a "-u" and its
        "on" or "off" attribute.
	
	TAKES:
		argc	-	A positive integer specifying the number of entries in argv[].
		agv		-	An array of argc+1 string pointers. The first entry, argv[0],
					points at a string that contains this application's file path
					and argv[1] through argv[argc-1] contain any parameters
					specified on the command line. Finally, argv[argc] is a NULL
					pointer.
	
	GIVES:
		-
	
	REVISIONS:
        2001-07-29	UK		Improved parameter parsing and added parameters.
		2000-10-26	UK		Changed to take parameters, made more cross-platform.
		1999-06-22	UK		Created.
   ----------------------------------------------------------------------------- */

int		JokerMain( int argc, char* argv[] )
{
	char*			vData = NULL;
	FILE*			vFile;
	size_t			vLength = 0;
	//time_t			vStartTime, vEndTime;
    HyperTalk		interpreter;
    TextMunger		script;
    
	// Process params:
	if( argc > 1 )
	{
		int		x;
		
		for( x = 1; x < argc; x++ )		// x is zero-based index, argc is one-based.
		{
			if( strcmp( argv[x], "-u" ) == 0 )
			{
				bool		vBool = true;
				if( ++x < argc )
					vBool = strcmp( argv[x], "on" ) == 0;
				HyperTalk::SetOption( OPT_ALLOW_UNQUOTED_LITERALS, vBool );
			}
			else if( strcmp( argv[x], "-f" ) == 0 )
			{
				bool		vBool = true;
				if( ++x < argc )
					vBool = strcmp( argv[x], "on" ) == 0;
				HyperTalk::SetOption( OPT_DISTINGUISH_FCN_CMD, vBool );
			}
			else if( strcmp( argv[x], "-v" ) == 0 )
			{
				bool		vBool = true;
				if( ++x < argc )
					vBool = strcmp( argv[x], "on" ) == 0;
				HyperTalk::SetOption( OPT_VARS_AS_UNQUOTED_LITERALS, vBool );
			}
			else if( strcmp( argv[x], "-e" ) == 0 )
			{
				bool		vBool = true;
				if( ++x < argc )
					vBool = strcmp( argv[x], "on" ) == 0;
				HyperTalk::SetOption( OPT_ESCAPE_CHARS_IN_STRINGS, vBool );
			}
            else if( strcmp( argv[x], "-i" ) == 0 )
            {
                std::cout << "Joker " << JOKER_VERSION_NUMBER << endl;
                if( JOKER_PLATFORM_FIX_NUMBER > 0 )
                    std::cout << "\tFixlevel " << JOKER_PLATFORM_FIX_NUMBER << endl;
                std::cout << "\tBased on the Joker distribution from " << JOKER_RELEASE_DATE << endl;
                std::cout << "\t(c) Copyright 1999-2001 by M. Uli Kusterer." << endl;
                std::cout << JOKER_PLATFORM_NAME << " port by " << JOKER_PLATFORM_COPYRIGHT << endl << endl;
                std::cout.flush();
            }
            else if( strcmp( argv[x], "-s" ) == 0 )
            {
                if( ++x < argc )
                    script.CopyFromString( argv[x] );
            }
            else
            {
                // Open file:
                vFile = fopen( argv[x], "r" );
                if( !vFile )
                {
                    std::cerr << "Error: Couldn't open source file \"" << argv[x] << "\"." << endl;
                    return 0;
                }
                
                // Get length:
                if( fseek( vFile, 0, SEEK_END ) != 0 )
                {
                    std::cerr << "Error: Couldn't determine size of source file \"" << argv[x] << "\"." << endl;
                    return 0;
                }

                vLength = ftell( vFile );
                fseek( vFile, 0, SEEK_SET );	// Move mark back to start.
                
                // Allocate memory for file's contents & read them:
                vData = (char*) malloc( vLength );
                if( !vData )
                {
                    std::cerr << "Error: Out of memory opening source file \"" << argv[x] << "\"." << endl;
                    return 0;
                }
                
                if( fread( vData, sizeof(char), vLength, vFile ) != vLength )
                {
                    std::cerr << "Error: Couldn't read in entire source file \"" << argv[x] << "\"." << endl;
                    return 0;
                }
                
                script.TakeOverText( vData, vLength );	// Takes care of disposing of vData.
            }
		}
	}
	
	ValueStack		vCallStack;
	long			vErrLine;
	HyperTalk*		errS;
		
	try {
        if( script.GetLength() == 0 )	// No script loaded? Fetch script from stdin.
        {
            script.SetOffset(0);
            
            while( true )
            {
                char		vChar;
                cin.get(vChar);
                if( cin.eof() )
                    break;	// Otherwise we get last char twice.
                script.Insert<char>(vChar);
            }
        }
        
		// Now do the interesting stuff:
        interpreter.Retain();	// Must call this or when Run() calls Release() it will try to call delete on our stack object.
		interpreter.Tokenize( script );
		interpreter.Parse();

	  #if JOKER_DUMP_INSTRUCTIONS
		interpreter.PrintAllHandlers();
	  #endif

		TalkMemLocation		vResult;
		vResult.mType = MEM_LOCATION_TYPE_INVALID;
		
		// Push param count (none) onto stack:
		TalkVarValue*	vValue = new TalkVarValue( (long) 0 );	// Will be disposed by Run(name,stack) below.
		if( !vValue )
			throw bad_alloc();
		vCallStack.push_back( vValue );	// Push!
		interpreter.SendMessage( TextMunger( "startUp" ), &vCallStack, vResult, false );
	}
	catch( parse_error& pe )
	{
		char		errStr[256];
		
		std::cerr << "\nError at characters " << pe.GetStartChar() << " to ";
		std::cerr << pe.GetEndChar() << ": " << pe.what() << "\n";
		
		script.SetOffset( pe.GetStartChar() );
		script.ReadData( errStr, pe.GetEndChar() -pe.GetStartChar() );
		errStr[pe.GetEndChar() -pe.GetStartChar()] = 0;
		
		std::cerr << "\t\"" << errStr << "\"\n\n";
        
        return EXIT_SUCCESS;
	}
	catch( bad_alloc& be )
	{
		vCallStack.GetErrorSource( vErrLine, errS );
		
		std::cerr << "\nOut of Memory in Line " << vErrLine << ": " << be.what() << "\n";
	}
	catch( exception& e )
	{
		vCallStack.GetErrorSource( vErrLine, errS );
		
		std::cerr << "\nLine " << vErrLine << ": " << e.what() << "\n";
	}
    catch( ... )
    {
        vCallStack.GetErrorSource( vErrLine, errS );
		
		std::cerr << "\nLine " << vErrLine << ": Unknown Error.\n";
    }
	
	return EXIT_SUCCESS; //EXIT_FAILURE;
}







