/*	***************************************************************************

	PROJECT:	Joker
	
	FILE:		TalkInstructions.h
	
	PURPOSE:	Instructions in a script.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-02-14	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkInstructions.h"
#include	"HyperTalk.h"
#include	"TalkVarValue.h"
#include	"TalkURLValue.h"
#include	"TalkExtendableEntity.h"
#include	"JokerMain.h"
#include	"SerialNumber.h"
#include	"JokerPackage.h"
#include	<iostream>
#include	<memory>


#pragma mark -
#pragma mark [Implementation]


/* --------------------------------------------------------------------------------
	PrintOneHandler:
		Output some handler's code to the console in some halfways readable format.
	
	TAKES:
		vCode	-	The code of the handler to print.
	
	REVISIONS:
		1999-11-29	UK		Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::PrintOneHandler( TalkHandler &vCode )
{
	TalkHandler::iterator		dbiterator;
	
	// Debugging info:
	if( vCode.GetIsFunction() )
		std::cout << "Function";
	else
		std::cout << "Message";
	
	std::cout << " needs " << vCode.GetStackNeed() << " variables on stack.\nPlus 1 slot for its result, plus 1 slot for the param count.\n\n";
	
	std::cout << "Parsed form of script:\n";
	
	for( dbiterator = vCode.begin(); dbiterator != vCode.end(); ++dbiterator )
		(*dbiterator).Dump();
	
	std::cout << "\n";
}


/* --------------------------------------------------------------------------------
	PrintAllHandlers:
		Loop over all handlers in this script and call PrintParsedCode() on them.
		Use this to debug parsing.
	
	REVISIONS:
		1999-12-05	UK	Created.
   ----------------------------------------------------------------------------- */

void	HyperTalk::PrintAllHandlers()
{
	TalkScript::iterator		dbiterator;
		
	for( dbiterator = mScript.begin(); dbiterator != mScript.end(); ++dbiterator )
	{
		std::cout << "\nHandler: " << const_cast<TextMunger&>((*dbiterator).first).String() << endl;
		
		PrintOneHandler( (*dbiterator).second );
	}
	
	std::cout << endl;
}


#pragma mark -

/* --------------------------------------------------------------------------------
	* CONSTRUCTOR:
		Initialize the data structure to some empty default values.
	
	REVISIONS:
		2001-01-05	UK	Moved here from header, added initialization of params.
   ----------------------------------------------------------------------------- */

HyperInstruction::HyperInstruction()
{
	mParamCount = 0;
	mRefCon = NULL;
	mInstructionProc = NULL;	// HyperTalk will skip instructions with NULL procPtrs.
	short		x;
	
	for( x = 0; x < TALK_INSTR_MAX_PARAMS; x++ )
	{
		mParam[x].mType = MEM_LOCATION_TYPE_INVALID;
	}
}


/* --------------------------------------------------------------------------------
	* DESTRUCTOR:
		Dispose of any immediates we got as parameters.
	
	REVISIONS:
		2001-01-05	UK	Created.
   ----------------------------------------------------------------------------- */

HyperInstruction::~HyperInstruction()
{
	ClearAllFields( true );
}


/* --------------------------------------------------------------------------------
	ClearAllFields:
		Clear all fields in this data structure. If desired we also dispose of
		any immediate values in our parameters.
		
	TAKES:
		doDispose	-	If TRUE, any immediate values in the mParam array will
						be deleted.
	
	REVISIONS:
		2001-01-05	UK	Created.
   ----------------------------------------------------------------------------- */

void	HyperInstruction::ClearAllFields( bool doDispose )
{
	short		x;
	
	if( mParamCount != INSTRUCTION_PARAM_COUNT_ON_STACK &&
		mParamCount <= TALK_INSTR_MAX_PARAMS )
	{
		for( x = 0; x < mParamCount; x++ )	// Changed from TALK_INSTR_MAX_PARAMS to mParamCount
		{
			if( mParam[x].mType == MEM_LOCATION_TYPE_IMMEDIATE
				&& mParam[x].mValue != NULL )	// Just to play it safe.
			{
				if( doDispose )
					delete mParam[x].mValue;
				mParam[x].mValue = NULL;
			}
		}
	}
	
	mParamCount = 0;
	mInstructionProc = NULL;
	mResult.mType = MEM_LOCATION_TYPE_INVALID;
	//mCaller = NULL;	// Don't clear so we don't have to set it when re-using a local variable.
}


/* --------------------------------------------------------------------------------
	Dump:
		Print an instruction to the console. Used for debugging.
	
	REVISIONS:
		2000-10-23	UK	Extracted from PrintOneHandler().
   ----------------------------------------------------------------------------- */

void	HyperInstruction::Dump()
{
	ValueStorage		vValStor;
	
	if( mInstructionProc == HyperStoreInstruction )
		std::cout << "STORE( ";
	else if( mInstructionProc == HyperPutInstruction )
		std::cout << "Put( ";
	else if( mInstructionProc == HyperReturnInstruction )
		std::cout << "Return( ";
	else if( mInstructionProc == HyperBreakpointInstruction )
		std::cout << "DEBUG_CHECKPOINT( ";
	else if( mInstructionProc == HyperReplaceInstruction )
		std::cout << "Replace( ";
	else if( mInstructionProc == HyperAddInstruction )
		std::cout << "Add( ";
	else if( mInstructionProc == HyperSubInstruction )
		std::cout << "Sub( ";
	else if( mInstructionProc == HyperMulInstruction )
		std::cout << "Mul( ";
	else if( mInstructionProc == HyperDivInstruction )
		std::cout << "Div( ";
	else if( mInstructionProc == HyperAddCommand )
		std::cout << "CmdAdd( ";
	else if( mInstructionProc == HyperSubCommand )
		std::cout << "CmdSub( ";
	else if( mInstructionProc == HyperMulCommand )
		std::cout << "CmdMul( ";
	else if( mInstructionProc == HyperDivCommand )
		std::cout << "CmdDiv( ";
	else if( mInstructionProc == HyperPushParamInstruction )
		std::cout << "Push( ";
	else if( mInstructionProc == HyperOutputInstruction )
		std::cout << "PutMsg( ";
	else if( mInstructionProc == HyperJSRInstruction )
		std::cout << "JSR( ";
	else if( mInstructionProc == HyperGetParamInstruction )
		std::cout << "Param( ";
	else if( mInstructionProc == HyperIfInstruction )
		std::cout << "If( ";
	else if( mInstructionProc == HyperCompareInstruction )
		std::cout << "Cmp( ";
	else if( mInstructionProc == HyperCompareNotInstruction )
		std::cout << "CmpNot( ";
	else if( mInstructionProc == HyperGreaterThanInstruction )
		std::cout << "CmpGT( ";
	else if( mInstructionProc == HyperLessThanInstruction )
		std::cout << "CmpLT( ";
	else if( mInstructionProc == HyperGetEntryInstruction )
		std::cout << "Entry( ";
	else if( mInstructionProc == HyperGetKeysInstruction )
		std::cout << "EntryKeys( ";
	else if( mInstructionProc == HyperConcatInstruction )
		std::cout << "Concatenate( ";
	else if( mInstructionProc == HyperConcatSpaceInstruction )
		std::cout << "ConcatenateSpace( ";
	else if( mInstructionProc == HyperGetStdinInstruction )
		std::cout << "Stdin( ";
	else if( mInstructionProc == HyperDeclareGlobalInstruction )
		std::cout << "Global( ";
	else if( mInstructionProc == HyperGetChunkOffsInstruction )
		std::cout << "ChunkOffs( ";
	else if( mInstructionProc == HyperGetChunkRefInstruction )
		std::cout << "ChunkRef( ";
	else if( mInstructionProc == HyperCountChunkInstruction )
		std::cout << "ChunkCount( ";
	else if( mInstructionProc == HyperNumToCharInstruction )
		std::cout << "NumToChar( ";
	else if( mInstructionProc == HyperCharToNumInstruction )
		std::cout << "CharToNum( ";
	else if( mInstructionProc == HyperDebuggerLineInstruction )
		std::cout << "NEW LINE(";
	else if( mInstructionProc == HyperDoInstruction )
		std::cout << "doScript( ";
	else if( mInstructionProc == HyperMakePackageInstruction )
		std::cout << ".makepackage( ";
	else if( mInstructionProc == HyperGetPropInstruction )
		std::cout << "GetProp( ";
	else if( mInstructionProc == HyperSetPropInstruction )
		std::cout << "SetProp( ";
	else if( mInstructionProc == HyperDefinePropInstruction )
		std::cout << "DefineProp( ";
	else if( mInstructionProc == HyperUndefinePropInstruction )
		std::cout << "UndefineProp( ";
	else if( mInstructionProc == HyperGetGlobalPropInstruction )
		std::cout << "GetGlobalProp( ";
	else if( mInstructionProc == HyperExitInstruction )
		std::cout << "EXIT( ";
	else if( mInstructionProc == HyperPassInstruction )
		std::cout << "Pass( ";
	else if( mInstructionProc == HyperLogicalOrInstruction )
		std::cout << "LogicalOr( ";
	else if( mInstructionProc == HyperLogicalAndInstruction )
		std::cout << "LogicalAnd( ";
	else if( mInstructionProc == HyperFindOffsetInstruction )
		std::cout << "Offset( ";
	else if( mInstructionProc == HyperUnaryMinusInstruction )
		std::cout << "UnaryMinus( ";
	else if( mInstructionProc == HyperUnaryNotInstruction )
		std::cout << "UnaryNot( ";
	else if( mInstructionProc == HyperLessThanEqualInstruction )
		std::cout << "CmpLTEq( ";
	else if( mInstructionProc == HyperGreaterThanEqualInstruction )
		std::cout << "CmpGTEq( ";
	else
		std::cout << "*unknown*( ";
	
	if( mInstructionProc != HyperStoreInstruction
		&& mInstructionProc != HyperGetStdinInstruction
		&& mInstructionProc != HyperDebuggerLineInstruction )
	{
		switch( mParam[0].mType )
		{
			case MEM_LOCATION_TYPE_STACK:
				std::cout << "[" << mParam[0].mLocation << "], ";
				break;
			
			case MEM_LOCATION_TYPE_IMMEDIATE:
				mParam[0].mValue->GetValue( vValStor, VALUE_TYPE_TEXT );
				std::cout << vValStor.textType->String() << ", ";
				delete vValStor.textType;	// TextMunger returned belongs to us.
				break;
			
			case MEM_LOCATION_TYPE_LONG:
				std::cout << mParam[0].mLong << ", ";
				break;
            
            case MEM_LOCATION_TYPE_INVALID:
                std::cout << "*INVALID*, ";
                break;
		}
	}
	
	if( mInstructionProc != HyperPutInstruction
		&& mInstructionProc != HyperOutputInstruction
		&& mInstructionProc != HyperGetStdinInstruction
		&& mInstructionProc != HyperPushParamInstruction
		&& mInstructionProc != HyperJSRInstruction
		&& mInstructionProc != HyperGetParamInstruction
		&& mInstructionProc != HyperDeclareGlobalInstruction
		&& mInstructionProc != HyperGetChunkRefInstruction
		&& mInstructionProc != HyperDebuggerLineInstruction
		&& mInstructionProc != HyperReturnInstruction
		&& mInstructionProc != HyperGetKeysInstruction
		&& mInstructionProc != HyperNumToCharInstruction
		&& mInstructionProc != HyperCharToNumInstruction
		&& mInstructionProc != HyperGetURLInstruction
		&& mInstructionProc != HyperDoInstruction
		&& mInstructionProc != HyperMakePackageInstruction
		&& mInstructionProc != HyperGetGlobalPropInstruction
		&& mInstructionProc != HyperPassInstruction )
	{
		switch( mParam[1].mType )
		{
			case MEM_LOCATION_TYPE_STACK:
				std::cout << "[" << mParam[1].mLocation << "] ) ";
				break;
			
			case MEM_LOCATION_TYPE_IMMEDIATE:
				mParam[1].mValue->GetValue( vValStor, VALUE_TYPE_TEXT );
				std::cout << vValStor.textType->String() << " ) ";
				delete vValStor.textType;	// TextMunger returned belongs to us.
				break;
			
			case MEM_LOCATION_TYPE_LONG:
				std::cout << mParam[1].mLong << " ) ";
				break;
			
			case MEM_LOCATION_TYPE_INVALID:
				std::cout << "*INVALID* ) ";
				break;
		}
	}
	else
		std::cout << " ) ";
	
	if( mInstructionProc != HyperPushParamInstruction
		&& mInstructionProc != HyperOutputInstruction
		&& mInstructionProc != HyperIfInstruction
		&& mInstructionProc != HyperGetChunkOffsInstruction
		&& mInstructionProc != HyperDebuggerLineInstruction
		&& mInstructionProc != HyperReturnInstruction
		&& mInstructionProc != HyperDoInstruction
		&& mInstructionProc != HyperMakePackageInstruction
		&& mInstructionProc != HyperDefinePropInstruction
		&& mInstructionProc != HyperUndefinePropInstruction
		&& mInstructionProc != HyperExitInstruction
		&& mInstructionProc != HyperPassInstruction )
	{
		switch( mResult.mType )
		{
			case MEM_LOCATION_TYPE_STACK:
				std::cout << "--> [" << mResult.mLocation << "]\n";
				break;
			
			case MEM_LOCATION_TYPE_LONG:
				std::cout << "--> " << mResult.mLong << " *** INVALID! ***\n";
				break;
			
			case MEM_LOCATION_TYPE_IMMEDIATE:
				mParam[1].mValue->GetValue( vValStor, VALUE_TYPE_TEXT );
				std::cout << "--> \"" << vValStor.textType->String() << "\"\n";
				delete vValStor.textType;	// TextMunger returned belongs to us.
				break;
			
			case MEM_LOCATION_TYPE_INVALID:
				std::cout << "--> *INVALID*\n";
				break;
		}
	}
	else
		std::cout << "\n";
}


#pragma mark -
#pragma mark [Instruction Procs]

/* --------------------------------------------------------------------------------
	All instruction procedures take two parameters, an "HI" parameter that contains
	the instruction's parameters and destination for the result along with other
	info like a pointer to the object currently running, and an "s" parameter
	that points to the stack with the variables for this instruction.
	
	Note that when returning a result from instructions that implement built-in
	functions, you must not follow any references, as function return values
	are output to temporaries which may contain other values.
   ----------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------------
	HyperStoreInstruction:
		A dummy instruction that simply copies the RHS to the result. This is used
		for transferring immediates into temps before commands are executed.
	
	HI's FIELDS:
		0		-	*Undefined*
		1		-	Value
		Result	-	[1]
   ----------------------------------------------------------------------------- */

void	HyperStoreInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	TalkValue*		vValue;
		
	hi.mResult.GetValue( vValue, s );	// Get destination.
	hi.mParam[1].CopyValueTo( (*vValue), s, false, false );	// RHS --> destination.
}


/* --------------------------------------------------------------------------------
	HyperDoInstruction:
		The "do" command that lets you execute HyperTalk statements inside
		variables or text files.
	
	HI's FIELDS:
		0		-	script to run
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperDoInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vScript;
	
	hi.mParam[0].GetValue( vScript, s, VALUE_TYPE_TEXT );		// Our TexMunAPtr disposes this.
	TexMunAPtr	vScriptPtr(vScript.textType);
	
	HyperTalk		vTalkScript;
	
	vTalkScript.Retain();
	vTalkScript.Tokenize( *vScriptPtr );
	vTalkScript.ParseAsHandler( "do" );
	if( vTalkScript.SendMessage( "do", true ) == CALL_RESULT_EXITED )
	{
		vCallRec.mExitHandler = true;
		vCallRec.mExitCompletely = true;
	}
}


/* --------------------------------------------------------------------------------
	HyperIfInstruction:
		Instruction for implementing conditionals ("if" structure).
	
	HI's FIELDS:
		0		-	Condition
		1		-	Jump address, it jumps here if LHS=FALSE, else this is just
					a no-op.
		Result	-	*undefined*
   ----------------------------------------------------------------------------- */

void	HyperIfInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vAddress;
	
	if( hi.mParam[0].mType == MEM_LOCATION_TYPE_LONG		// This is a "goto" instruction.
		&& hi.mParam[0].mLong == 1 )
		vAddress.boolType = false;
	else
		hi.mParam[0].GetValue( vAddress, s, VALUE_TYPE_BOOL );
	
	/* If "if" evaluates TRUE, it simply continues until it hits the goto,
		else it jumps to the location where the "else" commands start or
		where the "if" commands end. */
	if( !vAddress.boolType )
	{
		hi.mParam[1].GetValue( vAddress, s, VALUE_TYPE_LONG );
		hi.mCaller->Goto( vAddress.longType );
	}
}


/* --------------------------------------------------------------------------------
	HyperMakePackageInstruction:
		Instruction for implementing the (undocumented) instruction to create a
        package document ("bundle").
	
	HI's FIELDS:
		0		-	File search path for the package to create.
		Result	-	*undefined*
   ----------------------------------------------------------------------------- */

void	HyperMakePackageInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vPath;
	
    hi.mParam[0].GetValue( vPath, s, VALUE_TYPE_TEXT );
	TexMunAPtr		vKiller( vPath.textType );
    
    if( MakePackage( vPath.textType->String() ) != 0 )
        throw runtime_error( "Couldn't create package." );
}


/* --------------------------------------------------------------------------------
	HyperPushParamInstruction:
		Push a parameter onto the stack prior to doing a JSR. This is also used
		to push the param count as the last value onto the stack.
	
	HI's FIELDS:
		0		-	Value to push on stack
		Result	-	*undefined*
   ----------------------------------------------------------------------------- */

void	HyperPushParamInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	TalkVarValue*	vValue;
	
	vValue = new TalkVarValue( (long) 0 );
	if( !vValue )
		throw bad_alloc();
	hi.mParam[0].CopyValueTo( (*vValue), s );
	s.push_back( vValue );
}


/* --------------------------------------------------------------------------------
	HyperJSRInstruction:
		Actually perform a JSR after all parameters have been pushed on the stack.
	
	HI's FIELDS:
		0		-	Name of handler to call.
		Result	-	Function or handler result.
   ----------------------------------------------------------------------------- */

void	HyperJSRInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage		vHandlerName;
	
	hi.mParam[0].GetValue( vHandlerName, s, VALUE_TYPE_TEXT );
	TexMunAPtr	vKiller( vHandlerName.textType );	// The TextMunger returned is our responsibility.
	if( hi.mCaller->Run( (*vHandlerName.textType), s, hi.mResult ) == CALL_RESULT_EXITED )
	{
		vCallRec.mExitHandler = true;
		vCallRec.mExitCompletely = true;
	}
}


/* --------------------------------------------------------------------------------
	HyperPutInstruction:
		The 'put' command. Takes the LHS and copies it to the result.
	
	HI's FIELDS:
		0		-	Value to transfer
		Result	-	Value
   ----------------------------------------------------------------------------- */

void	HyperPutInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	TalkValue*		vValue;
		
	hi.mResult.GetValue( vValue, s );			// Get destination.
	hi.mParam[0].CopyValueTo( (*vValue), s, true, true );	// LHS --> destination. TRUE means if dest. is a ref. we follow it.
}


/* --------------------------------------------------------------------------------
	HyperGetPropInstruction:
		Fetch a properties' value.
	
	HI's FIELDS:
		0		-	Entity to get property from.
		1		-	Name of property.
		Result	-	Value
   ----------------------------------------------------------------------------- */

void	HyperGetPropInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	TalkValue*		vValue;
	TalkValue*		vParamZero;
	ValueStorage	vName;
	
	hi.mParam[1].GetValue( vName, s, VALUE_TYPE_TEXT );	// We're responsible for disposing of this.
	hi.mResult.GetValue( vValue, s );			// Get destination.
	hi.mParam[0].GetValue( vParamZero, s );
	vParamZero->GetPropertyValue( *vName.textType, (*vValue) );
	
	delete vName.textType;
}


/* --------------------------------------------------------------------------------
	HyperGetGlobalPropInstruction:
		Fetch a global properties' value.
	
	HI's FIELDS:
		0		-	Name of property.
		Result	-	Value
   ----------------------------------------------------------------------------- */

void	HyperGetGlobalPropInstruction( struct HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec )
{
	TalkValue*		vValue;
	ValueStorage	vName;
	
	hi.mParam[0].GetValue( vName, s, VALUE_TYPE_TEXT );	// We're responsible for disposing of this.
	hi.mResult.GetValue( vValue, s );			// Get destination.
	GetPropertyDefaultObject()->GetPropertyValue( *vName.textType, (*vValue) );
	
	delete vName.textType;
}



/* --------------------------------------------------------------------------------
	HyperSetPropInstruction:
		Fetch a properties' value.
	
	HI's FIELDS:
		0		-	Name of property (without "the").
		1		-	Entity to get property from (optional).
		2		-	Value to set property to.
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperSetPropInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	TalkValue*		vValue;
	TalkValue*		vEntity;
	ValueStorage	vName;
	
	hi.mParam[0].GetValue( vName, s, VALUE_TYPE_TEXT );	// We're responsible for disposing of this.
	hi.mParam[2].GetValue( vValue, s );					// Get value.
	hi.mParam[1].GetValue( vEntity, s );
	if( vEntity->GetNativeType() == VALUE_TYPE_TEXT )	// Param specifying entity to get property from was omitted -- is empty string.
		GetPropertyDefaultObject()->SetPropertyValue( *vName.textType, (*vValue) );
	else
		vEntity->SetPropertyValue( *vName.textType, (*vValue) );
	
	delete vName.textType;
}


/* --------------------------------------------------------------------------------
	HyperDefinePropInstruction:
		Add a user property to an object.
	
	HI's FIELDS:
		0		-	Name of property.
		1		-	Entity to add property to.
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperDefinePropInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage			vParamZero,
							vName;
	TalkExtendableEntity*	vPropObject;
	
	hi.mParam[0].GetValue( vName, s, VALUE_TYPE_TEXT );	// We're responsible for disposing of this.
	hi.mParam[1].GetValue( vParamZero, s, VALUE_TYPE_ENTITY );
	
	vPropObject = dynamic_cast<TalkExtendableEntity*>( vParamZero.entityType );
	if( vPropObject == NULL )
	{
		delete vName.textType;
		throw runtime_error( "Can't define user properties for this object." );
	}
	
	vPropObject->DefineProperty( *vName.textType );
	
	delete vName.textType;
}


/* --------------------------------------------------------------------------------
	HyperUndefinePropInstruction:
		Remove a user property from an object.
	
	HI's FIELDS:
		0		-	Name of property.
		1		-	Entity to add property to.
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperUndefinePropInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage			vParamZero,
							vName;
	TalkExtendableEntity*	vPropObject;
	
	hi.mParam[0].GetValue( vName, s, VALUE_TYPE_TEXT );	// We're responsible for disposing of this.
	hi.mParam[1].GetValue( vParamZero, s, VALUE_TYPE_ENTITY );
	
	vPropObject = dynamic_cast<TalkExtendableEntity*>( vParamZero.entityType );
	if( vPropObject == NULL )
	{
		delete vName.textType;
		throw runtime_error( "This kind of object has no user properties." );
	}
	
	vPropObject->UndefineProperty( *vName.textType );
	
	delete vName.textType;
}


/* --------------------------------------------------------------------------------
	HyperCheckSumInstruction:
		Calculate a check sum for a string.
	
	HI's FIELDS:
		0		-	String to make check sum from.
		Result	-	Check sum for the string.
   ----------------------------------------------------------------------------- */

void	HyperCheckSumInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage			vName;
	char					vSerial[10];
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The checkSum() function takes only one argument." );
	
	// Get string to determine check sum from:
	hi.mParam[0].GetValue( vName, s, VALUE_TYPE_TEXT );	// We're responsible for disposing of this.
	
	// Calc check sum:
	CalcSerialNumber( vName.textType->String(), vSerial );
	
	// Copy serial number to Text Munger.
	vName.textType->CopyFromString( vSerial );
	
	// Return Text Munger's value:
	hi.mResult.SetValue( vName, s, VALUE_TYPE_TEXT );
	
	delete vName.textType;
}


/* --------------------------------------------------------------------------------
	HyperReturnInstruction:
		The 'return' command. Takes the LHS and copies it to this handler's
		result.
	
	HI's FIELDS:
		0		-	Value to transfer
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperReturnInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	TalkValue*	vValue;
	
	/* Store the result value passed in the stack's last entry, which was reserved for
		this purpose. */
	vValue = s.back();	// Handler result is always at back of stack.
	hi.mParam[0].CopyValueTo( (*vValue), s, true, true );	// LHS --> handler result. TRUE means if dest. is a ref. we follow it.
	
	// Now exit our current handler:
	vCallRec.mExitHandler = true;
	
  #if JOKER_DEBUG
	std::cout << "\nStack after 'return' instruction:\n";
	s.Dump();
  #endif
}


/* --------------------------------------------------------------------------------
	HyperExitInstruction:
		The 'exit' command.
	
	HI's FIELDS:
		0		-	Handler name
		1		-	TRUE if we're exiting to HyperCard, FALSE otherwise.
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperExitInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	TalkValue*		vValue;
	ValueStorage	vResult;
	
	vResult.textType = new TextMunger("");
	TexMunAPtr		vEmptyStrPtr( vResult.textType );	// Make sure this is released on exit or exceptions alike.
	
	/* Store an empty string in the stack's last entry, which is the result value of
		this handler. */
	vValue = s.back();	// Handler result is always at back of stack.
	vValue->SetValue( vResult, VALUE_TYPE_TEXT, true );	// TRUE means if dest. is a ref. we follow it.
	
	hi.mParam[1].GetValue( vResult, s, VALUE_TYPE_BOOL );
	
	// Now exit our current handler:
	vCallRec.mExitHandler = true;
	vCallRec.mExitCompletely = vResult.boolType;
	
  #if JOKER_DEBUG
	std::cout << "\nStack after 'exit' instruction:\n";
	s.Dump();
  #endif
}


/* --------------------------------------------------------------------------------
	HyperPassInstruction:
		The 'pass' command. Simply sets some flags to exit at this point and let
		the next object in the hierarchy have this message.
	
	HI's FIELDS:
		0		-	Name of handler.
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperPassInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	// Now exit our current handler:
	vCallRec.mExitHandler = true;
	vCallRec.mPassFlag = true;
}


/* --------------------------------------------------------------------------------
	HyperDebuggerLineInstruction:
		This command is inserted after each script line has been parsed. This is
		very useful in debugging and error reporting.
	
	HI's FIELDS:
		0		-	line number
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperDebuggerLineInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLineNum;
		
	hi.mParam[0].GetValue( vLineNum, s, VALUE_TYPE_LONG );
	s.SetErrorSource( vLineNum.longType, hi.mCaller );	// Make sure call stack remembers what line and object had an error if a runtime error happens.
	
  #if JOKER_STACK_TRACES_LINEWISE
	std::cout << "\nStack Dump:\n";
	s.Dump();
  #endif
	
  #if JOKER_DEBUG_OUTPUT_LINES
	std::cout << "\n----- Line " << vLineNum.longType << " -----\n";
  #endif /*JOKER_DEBUG*/
  
	JokerSpendTime();	// Give application time to handle events, other threads etc.
}


/* --------------------------------------------------------------------------------
	HyperGetParamInstruction:
		Copy a parameter into a variable. Used for stuffing parameters in local
		variables.
		
		This instruction is generated automatically for the parameter list after
		function and message handler definitions, and is also registered as a
		one-argument function with the parser.
	
	HI's FIELDS:
		0		-	Number of param to get
		Result	-	Param value
   ----------------------------------------------------------------------------- */

void	HyperGetParamInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS;
	ValueStorage		vParamCount;
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The param() function takes only one argument." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
	s[ s.GetBasePointer() -1 ]->GetValue( vParamCount, VALUE_TYPE_LONG );
	if( vParamCount.longType >= vLHS.longType )	// Param in range, return value:
	{
		TalkValue*		vValue;
		
		hi.mResult.GetValue( vValue, s );
		s[ s.GetBasePointer() -1 -vParamCount.longType +vLHS.longType -1 ]->CopyValueTo( (*vValue) );
	}
	else	// Param out of range, set result to empty string.
	{
		ValueStorage		vEmptyStr;
		
		vEmptyStr.textType = new TextMunger( "" );
		
		hi.mResult.SetValue( vEmptyStr, s, VALUE_TYPE_TEXT );
		
		delete vEmptyStr.textType;
	}
}


/* --------------------------------------------------------------------------------
	HyperNumToHexInstruction:
		This one-argument function Computes the hexadecimal equivalent of an
		integer.
	
	HI's FIELDS:
		0		-	number to convert.
		Result	-	hex representation of this number.
   ----------------------------------------------------------------------------- */

void	HyperNumToHexInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS;
	ValueStorage		vResult;
	TalkValue*			vValue;
	char				vNumStr[10];
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The numToHex() function takes only one argument." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
	
	sprintf( vNumStr, "%lX", vLHS.longType );
	vResult.textType = new TextMunger( vNumStr );
	TexMunAPtr		vKiller( vResult.textType );
	
	hi.mResult.GetValue( vValue, s );
	vValue->SetValue( vResult, VALUE_TYPE_TEXT );
}


/* --------------------------------------------------------------------------------
	HyperHexToNumInstruction:
		This one-argument function Computes the integer equivalent of a
		hexadecimal number.
	
	HI's FIELDS:
		0		-	hex to convert.
		Result	-	number equivalent of this hex.
   ----------------------------------------------------------------------------- */

void	HyperHexToNumInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS;
	ValueStorage		vResult;
	TalkValue*			vValue;
	char*				endPtr;
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The hexToNum() function takes only one argument." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );
	TexMunAPtr		vKiller( vLHS.textType );	
	
	vResult.longType = strtol( vLHS.textType->String(), &endPtr, 16 );
	
	hi.mResult.GetValue( vValue, s );
	vValue->SetValue( vResult, VALUE_TYPE_LONG );
}


/* --------------------------------------------------------------------------------
	HyperBitAndInstruction:
		Perform a bitwise AND on two numbers.
	
	HI's FIELDS:
		0		-	number 1.
		1		-	number 2.
		Result	-	number 1 AND number 2.
   ----------------------------------------------------------------------------- */

void	HyperBitAndInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS,
						vRHS;
	ValueStorage		vResult;
	TalkValue*			vValue;
	
	if( hi.mParamCount != 2 )
		throw runtime_error( "The bitAnd() function takes exactly two arguments." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
	hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
	
	vResult.longType = vLHS.longType & vRHS.longType;
	
	hi.mResult.GetValue( vValue, s );
	vValue->SetValue( vResult, VALUE_TYPE_LONG );
}


/* --------------------------------------------------------------------------------
	HyperBitOrInstruction:
		Perform a bitwise OR on two numbers.
	
	HI's FIELDS:
		0		-	number 1.
		1		-	number 2.
		Result	-	number 1 OR number 2.
   ----------------------------------------------------------------------------- */

void	HyperBitOrInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS,
						vRHS;
	ValueStorage		vResult;
	TalkValue*			vValue;
	
	if( hi.mParamCount != 2 )
		throw runtime_error( "The bitAnd() function takes exactly two arguments." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
	hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
	
	vResult.longType = vLHS.longType | vRHS.longType;
	
	hi.mResult.GetValue( vValue, s );
	vValue->SetValue( vResult, VALUE_TYPE_LONG );
}


/* --------------------------------------------------------------------------------
	HyperBitNotInstruction:
		Perform a bitwise OR on two numbers.
	
	HI's FIELDS:
		0		-	number.
		Result	-	NOT number.
   ----------------------------------------------------------------------------- */

void	HyperBitNotInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS;
	ValueStorage		vResult;
	TalkValue*			vValue;
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The bitNot() function takes only one argument." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
	
	vResult.longType = ~vLHS.longType;
	
	hi.mResult.GetValue( vValue, s );
	vValue->SetValue( vResult, VALUE_TYPE_LONG );
}



/* --------------------------------------------------------------------------------
	HyperLongToNumInstruction:
		This one-argument function Computes the numeric equivalent of a
		four-character value.
	
	HI's FIELDS:
		0		-	4-char-code to convert.
		Result	-	numeric value of these characters.
   ----------------------------------------------------------------------------- */

void	HyperLongToNumInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS;
	ValueStorage		vResult;
	TalkValue*			vValue;
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The longToNum() function takes only one argument." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );	// TexMunAPtr disposes of this.
	TexMunAPtr	vLHSPtr( vLHS.textType );

	if( vLHSPtr->GetLength() < 4 )	// Less than 4 bytes? Pad it on the left side with NULL chars.
	{
		vLHSPtr->SetOffset( 0 );
		while( vLHSPtr->GetLength() < 4 )
			vLHSPtr->Insert<char>( 0 );
	}
	else if( vLHSPtr->GetLength() > 4 )
		throw runtime_error( "The longToNum() function exspects a value of four characters or less." );
	
	vLHSPtr->SetOffset(0);
	vResult.longType = vLHSPtr->Read<long>();
	
	hi.mResult.GetValue( vValue, s );
	vValue->SetValue( vResult, VALUE_TYPE_LONG );
}


/* --------------------------------------------------------------------------------
	HyperShortToNumInstruction:
		This one-argument function Computes the numeric equivalent of a
		two-character value.
	
	HI's FIELDS:
		0		-	2-char-code to convert.
		Result	-	numeric value of these characters.
   ----------------------------------------------------------------------------- */

void	HyperShortToNumInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS;
	ValueStorage		vResult;
	TalkValue*			vValue;
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The shortToNum() function takes only one argument." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );	// TexMunAPtr disposes of this.
	TexMunAPtr	vLHSPtr( vLHS.textType );

	if( vLHSPtr->GetLength() < 2 )	// Less than 2 bytes? Pad it on the left side with NULL chars.
	{
		vLHSPtr->SetOffset( 0 );
		while( vLHSPtr->GetLength() < 2 )
			vLHSPtr->Insert<char>( 0 );
	}
	else if( vLHSPtr->GetLength() > 2 )
		throw runtime_error( "The shortToNum() function exspects a value of two characters or less." );
	
	vLHSPtr->SetOffset(0);
	vResult.longType = vLHSPtr->Read<short>();
	
	hi.mResult.GetValue( vValue, s );
	vValue->SetValue( vResult, VALUE_TYPE_LONG );
}


/* --------------------------------------------------------------------------------
	HyperCharToNumInstruction:
		This one-argument function returns the ASCII value of a character.
	
	HI's FIELDS:
		0		-	Character to convert.
		Result	-	ASCII number of the character.
   ----------------------------------------------------------------------------- */

void	HyperCharToNumInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS;
	ValueStorage		vResult;
	TalkValue*			vValue;
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The charToNum() function takes only one argument." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );	// TexMunAPtr disposes of this.
	TexMunAPtr	vLHSPtr( vLHS.textType );

	if( vLHSPtr->GetLength() != 1 )
		throw runtime_error( "CharToNum() expects a single character as its parameter." );
	
	vLHSPtr->SetOffset(0);
	vResult.longType = vLHSPtr->Read<char>();
	
	hi.mResult.GetValue( vValue, s );
	vValue->SetValue( vResult, VALUE_TYPE_LONG );
}


/* --------------------------------------------------------------------------------
	HyperNumToCharInstruction:
		This one-argument function returns the character corresponding to an ASCII
		value.
	
	HI's FIELDS:
		0		-	ASCII number to convert.
		Result	-	The character with this ASCII value.
   ----------------------------------------------------------------------------- */

void	HyperNumToCharInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS;
	char				vChar;
	ValueStorage		vResult;
	TalkValue*			vValue;
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The numToChar() function takes only one argument." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
	
	if( vLHS.longType < 0 || vLHS.longType > 255 )
		throw runtime_error( "NumToChar() expects an integer from 0 to 255 as its parameter." );
	
	vChar = vLHS.longType;
	vResult.textType = new TextMunger( &vChar, 1 );	// TexMunAPtr disposes of this.
	TexMunAPtr	vResultTxPtr( vResult.textType );
	
	hi.mResult.GetValue( vValue, s );
	vValue->SetValue( vResult, VALUE_TYPE_TEXT );
}


/* --------------------------------------------------------------------------------
	HyperNumToLongInstruction:
		This one-argument function returns the 4 characters corresponding to an
		integer value.
	
	HI's FIELDS:
		0		-	number to convert.
		Result	-	The 4 characters representing this number value.
   ----------------------------------------------------------------------------- */

void	HyperNumToLongInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS;
	long				vChar;
	ValueStorage		vResult;
	TalkValue*			vValue;
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The numToLong() function takes only one argument." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
	
	vChar = vLHS.longType;
	vResult.textType = new TextMunger( (char*) &vChar, 4 );	// TexMunAPtr disposes of this.
	TexMunAPtr	vResultTxPtr( vResult.textType );
	
	hi.mResult.GetValue( vValue, s );
	vValue->SetValue( vResult, VALUE_TYPE_TEXT );
}


/* --------------------------------------------------------------------------------
	HyperNumToShortInstruction:
		This one-argument function returns the 2 characters corresponding to an
		integer value.
	
	HI's FIELDS:
		0		-	number to convert.
		Result	-	The 2 characters representing this number value.
   ----------------------------------------------------------------------------- */

void	HyperNumToShortInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vLHS;
	short				vChar;
	ValueStorage		vResult;
	TalkValue*			vValue;
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The numToShort() function takes only one argument." );
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
	
	vChar = vLHS.longType;
	vResult.textType = new TextMunger( (char*) &vChar, 2 );	// TexMunAPtr disposes of this.
	TexMunAPtr	vResultTxPtr( vResult.textType );
	
	hi.mResult.GetValue( vValue, s );
	vValue->SetValue( vResult, VALUE_TYPE_TEXT );
}


/* --------------------------------------------------------------------------------
	HyperFindOffsetInstruction:
		This implements the "offset" function that can be used to find the
		offset of a string inside another string.
	
	HI's FIELDS:
		0		-	Pattern to match.
		1		-	String to find pattern in.
		2		-	Optional offset to start at.
		Result	-	Number of character if found, zero if not found.
   ----------------------------------------------------------------------------- */

void	HyperFindOffsetInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vPattern,
						vString,
						vStartOffs,
						vResult;
	TalkValue*			vValue;
	
	if( hi.mParamCount == INSTRUCTION_PARAM_COUNT_ON_STACK )
		throw runtime_error( "The offset() function takes 2 or 3 arguments, no more, no less." );
	else if( hi.mParamCount < 2 )
		throw runtime_error( "The offset() function needs at least 2 arguments." );
	else if( hi.mParamCount > 3 )
		throw runtime_error( "The offset() function takes at most 3 arguments." );
	
	hi.mParam[0].GetValue( vPattern, s, VALUE_TYPE_TEXT );
	hi.mParam[1].GetValue( vString, s, VALUE_TYPE_TEXT );
	if( hi.mParamCount == 3 )
		hi.mParam[2].GetValue( vStartOffs, s, VALUE_TYPE_LONG );
	else
		vStartOffs.longType = 0;
	try
	{
		hi.mResult.GetValue( vValue, s );
		
		if( vPattern.textType->GetLength() == 0
			|| vString.textType->GetLength() == 0
			|| vStartOffs.longType >= vString.textType->GetLength() )
			vResult.longType = 0;	// Nothing to find, or nothing to search in.
		else
		{
			// Make sure we start searching at beginning:
			vPattern.textType->SetOffset(0);
			vString.textType->SetOffset(vStartOffs.longType);
			
			char	vFirstChar = tolower(vPattern.textType->Read<char>());
			vResult.longType = 0;
			
			// Now look for first char and whenever we find it try going on with our compare:
			while( vResult.longType == 0
				&& vString.textType->GetOffset() < vString.textType->GetLength() )
			{
				if( tolower(vString.textType->Read<char>()) == vFirstChar )
				{
					size_t		vOldOffs = vString.textType->GetOffset();
					
					vResult.longType = vOldOffs;	// Since xTalk is 1-based, this is fine.
					
					vPattern.textType->SetOffset(1);
					while( vPattern.textType->GetOffset() < vPattern.textType->GetLength()
						&& vString.textType->GetOffset() < vString.textType->GetLength() )
					{
						if( tolower(vPattern.textType->Read<char>())
							!= tolower(vString.textType->Read<char>()) )
						{
							vString.textType->SetOffset( vOldOffs );	// Go back to start of compare.
							vResult.longType = 0;	// Make sure we keep searching.
							break;	// Abort compare and go on searching.
						}
					}
					
					if( vPattern.textType->GetOffset() != vPattern.textType->GetLength()	// Didn't compare whole pattern? Premature end of string; not found!
						&& vResult.longType != 0 )	// If zero, it was aborted due to mismatch.
					{
						vResult.longType = 0;	// Return "not found" result.
						break;	// Exit search loop.
					}
				}
			}
		}
		
		delete vPattern.textType;
		delete vString.textType;
		
		if( vResult.longType != 0 )	// Found something?
			vResult.longType -= vStartOffs.longType;	// Make this a relative offset to starting offset, as MC and SC do it.
		vValue->SetValue( vResult, VALUE_TYPE_LONG );
	}
	catch( exception& err )
	{
		delete vPattern.textType;
		delete vString.textType;
		throw;
	}
}


/* --------------------------------------------------------------------------------
	HyperReplaceInstruction:
		This implements the "replace" command that can be used to replace all
		occurrences of a pattern string inside some text through a new string.
	
	HI's FIELDS:
		0		-	Pattern to match.
		1		-	String to replace pattern with.
		2		-	String to find pattern in.
		Result	-	Text with the values replaced.
   ----------------------------------------------------------------------------- */

void	HyperReplaceInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vPattern,
						vString,
						vNewStr,
						vResult;
	TalkValue*			vValue;
		
	hi.mParam[0].GetValue( vPattern, s, VALUE_TYPE_TEXT );
	hi.mParam[1].GetValue( vNewStr, s, VALUE_TYPE_TEXT );
	hi.mParam[2].GetValue( vString, s, VALUE_TYPE_TEXT );
	
	try
	{
		hi.mResult.GetValue( vValue, s );
		
		if( vPattern.textType->GetLength() == 0
			|| vString.textType->GetLength() == 0 )
			vValue->SetValue( vString, VALUE_TYPE_TEXT, true );	// Nothing to search/replace? Just return string unchanged.
		else
		{
			vString.textType->SetOffset(0);
			
			while( vString.textType->GetOffset() < vString.textType->GetLength() )
			{
				// Make sure we start searching at beginning:
				vPattern.textType->SetOffset(0);
				
				char	vFirstChar = tolower(vPattern.textType->Read<char>());
				vResult.longType = 0;
				
				// Now look for first char and whenever we find it try going on with our compare:
				while( vResult.longType == 0
					&& vString.textType->GetOffset() < vString.textType->GetLength() )
				{
					if( tolower(vString.textType->Read<char>()) == vFirstChar )
					{
						size_t		vOldOffs = vString.textType->GetOffset();
						
						vResult.longType = vOldOffs;	// We balance for this below.
						
						vPattern.textType->SetOffset(1);
						while( vPattern.textType->GetOffset() < vPattern.textType->GetLength()
							&& vString.textType->GetOffset() < vString.textType->GetLength() )
						{
							if( tolower(vPattern.textType->Read<char>())
								!= tolower(vString.textType->Read<char>()) )
							{
								vString.textType->SetOffset( vOldOffs );	// Go back to start of compare.
								vResult.longType = 0;	// Make sure we keep searching.
								break;	// Abort compare and go on searching.
							}
						}
						
						if( vPattern.textType->GetOffset() != vPattern.textType->GetLength()	// Didn't compare whole pattern? Premature end of string; not found!
							&& vResult.longType != 0 )	// If zero, it was aborted due to mismatch.
						{
							vResult.longType = 0;	// Return "not found" result.
							break;	// Exit search loop.
						}
					}
				}
				
				if( vResult.longType != 0 )	// Found something?
				{
					// First delete pattern string:
					vString.textType->SetOffset( vResult.longType-1 );
					vString.textType->DeleteData( vPattern.textType->GetLength() );
					
					// Then insert new string where pattern used to start.
					vString.textType->InsertMunger( *vNewStr.textType );
					
					// The offset has been moved past the new string by InsertMunger(). We needn't care about endless replaces.
				}
			}
		}
		
		delete vPattern.textType;
		delete vNewStr.textType;
		
		vValue->SetValue( vString, VALUE_TYPE_TEXT, true );	// Return the modified text.
		delete vString.textType;
	}
	catch( exception& err )
	{
		delete vPattern.textType;
		delete vNewStr.textType;
		delete vString.textType;
		throw;
	}
}

/* --------------------------------------------------------------------------------
	HyperGetStdinInstruction:
		Get user input from the console.
	
	HI's FIELDS:
		Result	-	User input
   ----------------------------------------------------------------------------- */

void	HyperGetStdinInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage		vEmptyStr;
	char				vUserInputString[512];		// 0.5k should be enough for even long lines
	
	std::cin.getline( vUserInputString, 512 );			// Read one line from console.
	
	vEmptyStr.textType = new TextMunger( vUserInputString );
	std::cout << "INPUT RECEIVED:" << vUserInputString << endl;
		
	hi.mResult.SetValue( vEmptyStr, s, VALUE_TYPE_TEXT );
	
	delete vEmptyStr.textType;	// Get rid of this text munger.
}


/* --------------------------------------------------------------------------------
	HyperOutputInstruction:
		The 'put' command w/o destination. Takes the LHS and prints it to the
		console.
	
	HI's FIELDS:
		0		-	Value to output
		Result	-	*undefined*
   ----------------------------------------------------------------------------- */

void	HyperOutputInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS;
	
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );
	std::cout << (*vLHS.textType).String();
	cout.flush();
	
	delete vLHS.textType;	// The TextMunger returned by GetValue() belongs to us.
}


/* --------------------------------------------------------------------------------
	HyperUnaryMinusInstruction:
		Mathematical negation operator.
	
	HI's FIELDS:
		0		-	RHS argument.
		Result	-	-LHS + RHS
   ----------------------------------------------------------------------------- */

void	HyperUnaryMinusInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage	vRHS,
					vResult;
	
	// Get 1st arg:
	hi.mParam[0].GetValue( vRHS, s, VALUE_TYPE_LONG );
	
	// Calculate and store result:
	vResult.longType = -vRHS.longType;
	hi.mResult.SetValue( vResult, s, VALUE_TYPE_LONG );
}


/* --------------------------------------------------------------------------------
	HyperUnaryNotInstruction:
		Logical negation operator.
	
	HI's FIELDS:
		0		-	RHS argument.
		Result	-	-RHS
   ----------------------------------------------------------------------------- */

void	HyperUnaryNotInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage	vRHS,
					vResult;
	
	// Get 1st arg:
	hi.mParam[0].GetValue( vRHS, s, VALUE_TYPE_BOOL );
	
	// Calculate and store result:
	vResult.boolType = !vRHS.boolType;
	hi.mResult.SetValue( vResult, s, VALUE_TYPE_BOOL );
}


/* --------------------------------------------------------------------------------
	HyperAddInstruction:
		The beef that makes the add operator tick.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS + RHS
   ----------------------------------------------------------------------------- */

void	HyperAddInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vTypes;
	
	/* Do a bitwise AND on the type bits. What's left in vTypes
		is the types they both share. Since integers can always
		be turned into doubles, there's always at least floating
		point maths available if it's a number at all: */
	vTypes = hi.mParam[0].GetAvailableTypes(s);
	vTypes &= hi.mParam[1].GetAvailableTypes(s);
	
	// Try integer maths first -- faster:
	if( (vTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		// Calculate and store result:
		vResult.longType = vLHS.longType + vRHS.longType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_LONG );
	}
	else if( (vTypes & VALUE_TYPE_DOUBLE) == VALUE_TYPE_DOUBLE )	// Otherwise try floating point maths:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_DOUBLE );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_DOUBLE );
		
		// Calculate and store result:
		vResult.doubleType = vLHS.doubleType + vRHS.doubleType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_DOUBLE );
	}
	else
		throw runtime_error("Expected number here.");
}


/* --------------------------------------------------------------------------------
	HyperLogicalOrInstruction:
		The beef that makes the or operator tick.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS or RHS
   ----------------------------------------------------------------------------- */

void	HyperLogicalOrInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	
	// Get 1st arg:
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_BOOL );
		
	// Get 2nd arg:
	hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_BOOL );
		
	// Calculate and store result:
	vResult.boolType = vLHS.boolType || vRHS.boolType;
	hi.mResult.SetValue( vResult, s, VALUE_TYPE_BOOL );
}


/* --------------------------------------------------------------------------------
	HyperLogicalAndInstruction:
		The beef that makes the and operator tick.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS and RHS
   ----------------------------------------------------------------------------- */

void	HyperLogicalAndInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	
	// Get 1st arg:
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_BOOL );
		
	// Get 2nd arg:
	hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_BOOL );
		
	// Calculate and store result:
	vResult.boolType = vLHS.boolType && vRHS.boolType;
	hi.mResult.SetValue( vResult, s, VALUE_TYPE_BOOL );
}


/* --------------------------------------------------------------------------------
	HyperAddCommand:
		The beef that makes the add command tick.
		
		The command needs to follow refs, but that's all that makes this different
		from the operator.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS + RHS
   ----------------------------------------------------------------------------- */

void	HyperAddCommand( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vTypes;
	
	/* Do a bitwise AND on the type bits. What's left in vTypes
		is the types they both share. Since integers can always
		be turned into doubles, there's always at least floating
		point maths available if it's a number at all: */
	vTypes = hi.mParam[0].GetAvailableTypes(s);
	vTypes &= hi.mParam[1].GetAvailableTypes(s);
	
	// Try integer maths first -- faster:
	if( (vTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		// Calculate and store result:
		vResult.longType = vLHS.longType + vRHS.longType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_LONG, true );
	}
	else if( (vTypes & VALUE_TYPE_DOUBLE) == VALUE_TYPE_DOUBLE )	// Otherwise try floating point maths:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_DOUBLE );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_DOUBLE );
		
		// Calculate and store result:
		vResult.doubleType = vLHS.doubleType + vRHS.doubleType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_DOUBLE, true );
	}
	else
		throw runtime_error("Expected number here.");
}


/* --------------------------------------------------------------------------------
	HyperSubInstruction:
		The beef that makes the subtract operator tick.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS - RHS
   ----------------------------------------------------------------------------- */

void	HyperSubInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vTypes;
	
	/* Do a bitwise AND on the type bits. What's left in vTypes
		is the types they both share. Since integers can always
		be turned into doubles, there's always at least floating
		point maths available if it's a number at all: */
	vTypes = hi.mParam[0].GetAvailableTypes(s);
	vTypes &= hi.mParam[1].GetAvailableTypes(s);
	
	// Try integer maths first -- faster:
	if( (vTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		// Calculate and store result:
		vResult.longType = vLHS.longType - vRHS.longType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_LONG );
	}
	else if( (vTypes & VALUE_TYPE_DOUBLE) == VALUE_TYPE_DOUBLE )	// Otherwise try floating point maths:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_DOUBLE );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_DOUBLE );
		
		// Calculate and store result:
		vResult.doubleType = vLHS.doubleType - vRHS.doubleType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_DOUBLE );
	}
	else
		throw runtime_error("Expected number here.");
}


/* --------------------------------------------------------------------------------
	HyperSubCommand:
		The beef that makes the subtract command tick.
		
		The command needs to follow refs, but that's all that makes this different
		from the operator.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS - RHS
   ----------------------------------------------------------------------------- */

void	HyperSubCommand( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vTypes;
	
	/* Do a bitwise AND on the type bits. What's left in vTypes
		is the types they both share. Since integers can always
		be turned into doubles, there's always at least floating
		point maths available if it's a number at all: */
	vTypes = hi.mParam[0].GetAvailableTypes(s);
	vTypes &= hi.mParam[1].GetAvailableTypes(s);
	
	// Try integer maths first -- faster:
	if( (vTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		// Calculate and store result:
		vResult.longType = vLHS.longType - vRHS.longType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_LONG, true );
	}
	else if( (vTypes & VALUE_TYPE_DOUBLE) == VALUE_TYPE_DOUBLE )	// Otherwise try floating point maths:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_DOUBLE );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_DOUBLE );
		
		// Calculate and store result:
		vResult.doubleType = vLHS.doubleType - vRHS.doubleType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_DOUBLE, true );
	}
	else
		throw runtime_error("Expected number here.");
}


/* --------------------------------------------------------------------------------
	HyperMulInstruction:
		The beef that makes the multiply operator tick.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS * RHS
   ----------------------------------------------------------------------------- */

void	HyperMulInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vTypes;
	
	/* Do a bitwise AND on the type bits. What's left in vTypes
		is the types they both share. Since integers can always
		be turned into doubles, there's always at least floating
		point maths available if it's a number at all: */
	vTypes = hi.mParam[0].GetAvailableTypes(s);
	vTypes &= hi.mParam[1].GetAvailableTypes(s);
	
	// Try integer maths first -- faster:
	if( (vTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		// Calculate and store result:
		vResult.longType = vLHS.longType * vRHS.longType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_LONG );
	}
	else if( (vTypes & VALUE_TYPE_DOUBLE) == VALUE_TYPE_DOUBLE )	// Otherwise try floating point maths:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_DOUBLE );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_DOUBLE );
		
		// Calculate and store result:
		vResult.doubleType = vLHS.doubleType * vRHS.doubleType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_DOUBLE );
	}
	else
		throw runtime_error("Expected number here.");
}


/* --------------------------------------------------------------------------------
	HyperMulCommand:
		The beef that makes the multiply command tick.
		
		The command needs to follow refs, but that's all that makes this different
		from the operator.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS * RHS
   ----------------------------------------------------------------------------- */

void	HyperMulCommand( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vTypes;
	
	/* Do a bitwise AND on the type bits. What's left in vTypes
		is the types they both share. Since integers can always
		be turned into doubles, there's always at least floating
		point maths available if it's a number at all: */
	vTypes = hi.mParam[0].GetAvailableTypes(s);
	vTypes &= hi.mParam[1].GetAvailableTypes(s);
	
	// Try integer maths first -- faster:
	if( (vTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		// Calculate and store result:
		vResult.longType = vLHS.longType * vRHS.longType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_LONG, true );
	}
	else if( (vTypes & VALUE_TYPE_DOUBLE) == VALUE_TYPE_DOUBLE )	// Otherwise try floating point maths:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_DOUBLE );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_DOUBLE );
		
		// Calculate and store result:
		vResult.doubleType = vLHS.doubleType * vRHS.doubleType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_DOUBLE, true );
	}
	else
		throw runtime_error("Expected number here.");
}


/* --------------------------------------------------------------------------------
	HyperDivInstruction:
		The beef that makes the div operator tick.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS / RHS
   ----------------------------------------------------------------------------- */

void	HyperDivInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vTypes;
	
	/* Do a bitwise AND on the type bits. What's left in vTypes
		is the types they both share. Since integers can always
		be turned into doubles, there's always at least floating
		point maths available if it's a number at all: */
	vTypes = hi.mParam[0].GetAvailableTypes(s);
	vTypes &= hi.mParam[1].GetAvailableTypes(s);
	
	// Try integer maths first -- faster:
	if( (vTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		if( vRHS.longType == 0 )
			throw runtime_error( "I'd love to see _you_ divide by zero..." );
		
		// Calculate and store result:
		vResult.longType = vLHS.longType / vRHS.longType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_LONG );
	}
	else if( (vTypes & VALUE_TYPE_DOUBLE) == VALUE_TYPE_DOUBLE )	// Otherwise try floating point maths:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_DOUBLE );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_DOUBLE );
		
		if( vRHS.doubleType == 0.0 )
			throw runtime_error( "I'd love to see _you_ divide by zero..." );
		
		// Calculate and store result:
		vResult.doubleType = vLHS.doubleType / vRHS.doubleType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_DOUBLE );
	}
	else
		throw runtime_error("Expected number here.");
}


/* --------------------------------------------------------------------------------
	HyperDivCommand:
		The beef that makes the divide command tick.
		
		The command needs to follow refs, but that's all that makes this different
		from the operator.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS / RHS
   ----------------------------------------------------------------------------- */

void	HyperDivCommand( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vTypes;
	
	/* Do a bitwise AND on the type bits. What's left in vTypes
		is the types they both share. Since integers can always
		be turned into doubles, there's always at least floating
		point maths available if it's a number at all: */
	vTypes = hi.mParam[0].GetAvailableTypes(s);
	vTypes &= hi.mParam[1].GetAvailableTypes(s);
	
	// Try integer maths first -- faster:
	if( (vTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		if( vRHS.longType == 0 )
			throw runtime_error( "I'd love to see _you_ divide by zero..." );
		
		// Calculate and store result:
		vResult.longType = vLHS.longType / vRHS.longType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_LONG, true );
	}
	else if( (vTypes & VALUE_TYPE_DOUBLE) == VALUE_TYPE_DOUBLE )	// Otherwise try floating point maths:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_DOUBLE );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_DOUBLE );
		
		if( vRHS.doubleType == 0.0 )
			throw runtime_error( "I'd love to see _you_ divide by zero..." );
		
		// Calculate and store result:
		vResult.doubleType = vLHS.doubleType / vRHS.doubleType;
		hi.mResult.SetValue( vResult, s, VALUE_TYPE_DOUBLE, true );
	}
	else
		throw runtime_error("Expected number here.");
}


/* --------------------------------------------------------------------------------
	HyperCompareInstruction:
		Implements the = operator.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS == RHS
   ----------------------------------------------------------------------------- */

void	HyperCompareInstruction( HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vResult;
	
	vResult.boolType = CompareBeef( hi, s );
	
	// Store a boolean whether the arguments matched in the result:
	hi.mResult.SetValue( vResult, s, VALUE_TYPE_BOOL );
}


/* --------------------------------------------------------------------------------
	HyperCompareNotInstruction:
		Implements the <> operator.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS != RHS
   ----------------------------------------------------------------------------- */

void	HyperCompareNotInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vResult;
	
	vResult.boolType = !CompareBeef( hi, s );
	
	// Store a boolean whether the arguments matched in the result:
	hi.mResult.SetValue( vResult, s, VALUE_TYPE_BOOL );
}


/* --------------------------------------------------------------------------------
	HyperLessThanInstruction:
		< operator.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS < RHS
   ----------------------------------------------------------------------------- */

void	HyperLessThanInstruction( HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vAvailableTypes;
	
	// Get what types we can get from our arguments:
	vAvailableTypes = hi.mParam[0].GetAvailableTypes( s );
	vAvailableTypes &= hi.mParam[1].GetAvailableTypes( s );	// Mask out those they don't have in common.
	
	if( (vAvailableTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		// Calculate and store result:
		vResult.boolType = (vLHS.longType < vRHS.longType);
	}
	else	// Else compare as strings, which every value can export to:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_TEXT );
		
		// Calculate and store result:
		vResult.boolType = ((*vLHS.textType) < (*vRHS.textType));	// TextMunger overrides operator==.
		
		// The TextMungers returned are our responsibility:
		delete vLHS.textType;
		delete vRHS.textType;
	}
	
	// Store a boolean in the result:
	hi.mResult.SetValue( vResult, s, VALUE_TYPE_BOOL );
}


/* --------------------------------------------------------------------------------
	HyperLessThanqualInstruction:
		<= operator.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS <= RHS
   ----------------------------------------------------------------------------- */

void	HyperLessThanEqualInstruction( HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vAvailableTypes;
	
	// Get what types we can get from our arguments:
	vAvailableTypes = hi.mParam[0].GetAvailableTypes( s );
	vAvailableTypes &= hi.mParam[1].GetAvailableTypes( s );	// Mask out those they don't have in common.
	
	if( (vAvailableTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		// Calculate and store result:
		vResult.boolType = (vLHS.longType <= vRHS.longType);
	}
	else	// Else compare as strings, which every value can export to:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_TEXT );
		
		// Calculate and store result:
		vResult.boolType = ((*vLHS.textType) <= (*vRHS.textType));	// TextMunger overrides operator==.
		
		// The TextMungers returned are our responsibility:
		delete vLHS.textType;
		delete vRHS.textType;
	}
	
	// Store a boolean in the result:
	hi.mResult.SetValue( vResult, s, VALUE_TYPE_BOOL );
}


/* --------------------------------------------------------------------------------
	HyperGreaterThanInstruction:
		> operator.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS > RHS
   ----------------------------------------------------------------------------- */

void	HyperGreaterThanInstruction( HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vAvailableTypes;
	
	// Get what types we can get from our arguments:
	vAvailableTypes = hi.mParam[0].GetAvailableTypes( s );
	vAvailableTypes &= hi.mParam[1].GetAvailableTypes( s );	// Mask out those they don't have in common.
	
	if( (vAvailableTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		// Calculate and store result:
		vResult.boolType = (vLHS.longType > vRHS.longType);
	}
	else	// Else compare as strings, which every value can export to:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_TEXT );
		
		// Calculate and store result:
		vResult.boolType = ((*vLHS.textType) > (*vRHS.textType));	// TextMunger overrides operator==.
		
		// The TextMungers returned are our responsibility:
		delete vLHS.textType;
		delete vRHS.textType;
	}
	
	// Store a boolean in the result:
	hi.mResult.SetValue( vResult, s, VALUE_TYPE_BOOL );
}


/* --------------------------------------------------------------------------------
	HyperGreaterThanEqualInstruction:
		>= operator.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	LHS >= RHS
   ----------------------------------------------------------------------------- */

void	HyperGreaterThanEqualInstruction( HyperInstruction& hi, ValueStack& s,
											TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS,
					vResult;
	unsigned long	vAvailableTypes;
	
	// Get what types we can get from our arguments:
	vAvailableTypes = hi.mParam[0].GetAvailableTypes( s );
	vAvailableTypes &= hi.mParam[1].GetAvailableTypes( s );	// Mask out those they don't have in common.
	
	if( (vAvailableTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		// Calculate and store result:
		vResult.boolType = (vLHS.longType >= vRHS.longType);
	}
	else	// Else compare as strings, which every value can export to:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_TEXT );
		
		// Calculate and store result:
		vResult.boolType = ((*vLHS.textType) >= (*vRHS.textType));	// TextMunger overrides operator==.
		
		// The TextMungers returned are our responsibility:
		delete vLHS.textType;
		delete vRHS.textType;
	}
	
	// Store a boolean in the result:
	hi.mResult.SetValue( vResult, s, VALUE_TYPE_BOOL );
}


/* --------------------------------------------------------------------------------
	HyperConcatInstruction:
		Implements the & operator.
		
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	RHS appended to LHS.
   ----------------------------------------------------------------------------- */

void	HyperConcatInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS;
	
	// Get 1st arg:
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );
	
	// Get 2nd arg:
	hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_TEXT );
	
	// Merge the two strings:
	vLHS.textType->SetOffset( vLHS.textType->GetLength() );
	vLHS.textType->InsertMunger( (*vRHS.textType) );
	
	// The TextMungers returned are our responsibility:
	delete vRHS.textType;
	
	// Set the result's value to ours:
	hi.mResult.SetValue( vLHS, s, VALUE_TYPE_TEXT );
	
	delete vLHS.textType;
}


/* --------------------------------------------------------------------------------
	HyperConcatSpaceInstruction:
		Implements the && operator.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	RHS appended to LHS with a space inserted in between.
   ----------------------------------------------------------------------------- */

void	HyperConcatSpaceInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS;
	
	// Get 1st arg:
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );
	
	// Get 2nd arg:
	hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_TEXT );
	
	// Merge the two strings:
	vLHS.textType->SetOffset( vLHS.textType->GetLength() );
	vLHS.textType->Insert<char>( ' ' );	// Insert space between arguments.
	vLHS.textType->InsertMunger( (*vRHS.textType) );
	
	// The TextMungers returned are our responsibility:
	delete vRHS.textType;
	
	// Set the result's value to ours:
	//s[ s.GetBasePointer() +hi.mResult.mLocation ]->SetValue( vLHS, VALUE_TYPE_TEXT );
	hi.mResult.SetValue( vLHS, s, VALUE_TYPE_TEXT );
	
	delete vLHS.textType;
}


/* --------------------------------------------------------------------------------
	HyperGetEntryInstruction:
		This implements array access.
		
		Watch out: The order of the parameters is the reverse of the syntax of
		the "entry" statement.
		
	HI's FIELDS:
		0		-	Array.
		1		-	Array index.
		Result	-	Value at <index> in <array>.
   ----------------------------------------------------------------------------- */

void	HyperGetEntryInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS;
	TalkValue*		vEntry;
	
	// Get index:
	hi.mParam[1].GetValue( vLHS, s, VALUE_TYPE_TEXT );
	TexMunAPtr	vKiller( vLHS.textType );	// This takes care it's disposed.
	
	// Get entry of that index from value:
	hi.mParam[0].GetListEntry( vEntry, s, (*vLHS.textType) );
	vRHS.valueType = vEntry;
	
	/* Set the result's value to be a reference to this entry:
		We pass FALSE so the variable's mValue.valueType is set. */
	hi.mResult.SetValue( vRHS, s, VALUE_TYPE_VALUE );
}


/* --------------------------------------------------------------------------------
	HyperGetURLInstruction:
		This implements file and URL access.
		
	HI's FIELDS:
		0		-	URL.
		Result	-	The contents of the file.
   ----------------------------------------------------------------------------- */

void	HyperGetURLInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS;
	TalkValue*		vFile;
	
	// Get URL:
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );
	
	// Get entry of that index from value:
	vFile = new TalkURLValue( *vLHS.textType );
	delete vLHS.textType;	// Text munger returned is our responsibility.
	vRHS.valueType = vFile;
	
	/* Set the result's value to be a reference to this entry:
		We pass FALSE so the variable's mValue.valueType is set. */
	hi.mResult.SetValue( vRHS, s, VALUE_TYPE_VALUE );
	
	// FIX ME! Leak! Need to keep a list of all URL objects we create so we can dispose of them again.
}


/* --------------------------------------------------------------------------------
	HyperGetKeysInstruction:
		Returns a return-delimited list of all keys an array has.
		
		This is a one-argument function registered with the parser at startup.
		
	HI's FIELDS:
		0		-	Array.
		Result	-	Return-delimited text string of keys.
   ----------------------------------------------------------------------------- */

void	HyperGetKeysInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vRHS;
	TalkValue*		vArray;
	TextMunger*		vKeyString;
	
	if( hi.mParamCount != 1 )
		throw runtime_error( "The keys() function takes only one argument." );
	
	// Get array:
	hi.mParam[0].GetValue( vArray, s );
	
	// Get key list from array:
	vKeyString = vArray->GetKeys();
	vRHS.textType = vKeyString;
	
	// Store key list in our result:
	hi.mResult.SetValue( vRHS, s, VALUE_TYPE_TEXT );
	
	delete vKeyString;
}


/* --------------------------------------------------------------------------------
	HyperDeclareGlobalInstruction:
		This implements access to globals.
		
	HI's FIELDS:
		0		-	Name of global.
		Result	-	Reference to our global.
   ----------------------------------------------------------------------------- */

void	HyperDeclareGlobalInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	ValueStorage	vLHS,
					vRHS;
	TalkValue*		vGlobalValue;
	
	// Get index:
	hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );
	
	// Get entry of that index from value:
	hi.mCaller->GetGlobal( vGlobalValue, (*vLHS.textType), true );	// Get the global's value creating one if none exists.
	delete vLHS.textType;	// Text munger returned is our responsibility.
	vRHS.valueType = vGlobalValue;
	
	/* Set the result's value to be a reference to this global:
		We pass FALSE so the variable's mValue.valueType is set. */
	hi.mResult.SetValue( vRHS, s, VALUE_TYPE_VALUE );
}


/* --------------------------------------------------------------------------------
	HyperDeleteCommand:
		This implements the "delete" command.
		
	HI's FIELDS:
		0		-	Reference to container to delete.
		Result	-	*unused*.
   ----------------------------------------------------------------------------- */

void	HyperDeleteCommand( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	TalkVarValue		vDestValue(0L);
	
	hi.mParam[0].CopyValueTo( vDestValue, s, false );	// Don't follow any refs.
	vDestValue.Delete(0,0);
}


/* --------------------------------------------------------------------------------
	HyperGetChunkOffsInstruction:
		This is used once per nested chunk statement. A row of these must be
		finished off with a HyperGetChunkRefInstruction that actually extracts
		the chunk from the value.
		
	HI's FIELDS:
		0		-	start offset (one-based).
		1		-	end offset (one-based) or zero for chunk of length 1.
		2		-	Value to get chunk from.
		3		-	Chunk type to get.
		4		-	Item delimiter (items only). Usually this just points at
					the local variable "the itemdelimiter".
		Result	-	*undefined*
   ----------------------------------------------------------------------------- */

void	HyperGetChunkOffsInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage	vTargetValue,
					vStartOffs,
					vEndOffs,
					vChunkType,
					vItemDel;
	size_t			vCharStartOffs,
					vCharEndOffs,
					vCharSvStartOffs;
	
	hi.mParam[0].GetValue( vStartOffs, s, VALUE_TYPE_LONG );	// Chunk start.
	hi.mParam[1].GetValue( vEndOffs, s, VALUE_TYPE_LONG );		// Chunk end.
	hi.mParam[2].GetValue( vTargetValue, s, VALUE_TYPE_TEXT );	// Value to get chunk from.
	hi.mParam[3].GetValue( vChunkType, s, VALUE_TYPE_LONG );	// Chunk unit (chunk type).
	
	// Get caller's chunk range for this expression:
	vCharStartOffs = vCallRec.mChunkStartOffs;
	vCharEndOffs = vCallRec.mChunkEndOffs;
	if( vCharEndOffs == 0 )		// This offset is zero-based, if end is 0, this is unused.
	{
		vCharStartOffs = 0;
		vCharEndOffs = vTargetValue.textType->GetLength();
	}
	
	char		vChunkDelimiter = '\n';
	
	if( vTargetValue.textType->GetLength() == 0 )
		vCharStartOffs = vCharEndOffs = vTargetValue.textType->GetOffset();
	else
	{
		// Now parse this chunk within the range we are limited to:
		switch( vChunkType.longType )
		{
			case CHUNK_TYPE_CHARACTER:
			  #if JOKER_CHUNK_TYPES
				cout << "\nChunk type: Character";
			  #endif
				vCharStartOffs += vStartOffs.longType -1;	// Add start offset as zero-based offset to character start offset.
				
				if( vEndOffs.longType == 0 )	// Only 1 character
				{
					if( vCharEndOffs >= (vCharStartOffs +1) )	// Still in range?
						vCharEndOffs = vCharStartOffs +1;
					// Else leave end offset like it is.
				}
				else
				{
					vEndOffs.longType -= vStartOffs.longType -1;		// Calc length (remember, vStartOffs is one-based!).
					if( vCharEndOffs >= (vEndOffs.longType +vCharStartOffs) )	// Still in range?
					{
						vCharEndOffs = vEndOffs.longType +vCharStartOffs;
					}
					// Else leave end offset like it is.
				}
				break;
			
			case CHUNK_TYPE_BYTE:
			  #if JOKER_CHUNK_TYPES
				cout << "\nChunk type: Byte";
			  #endif
				vCharStartOffs += vStartOffs.longType -1;	// Add start offset as zero-based offset to character start offset.
				
				if( vEndOffs.longType == 0 )	// Only 1 character
				{
					if( vCharEndOffs >= (vCharStartOffs +1) )	// Still in range?
						vCharEndOffs = vCharStartOffs +1;
					// Else leave end offset like it is.
				}
				else
				{
					vEndOffs.longType -= vStartOffs.longType -1;		// Calc length (remember, vStartOffs is one-based!).
					if( vCharEndOffs >= (vEndOffs.longType +vCharStartOffs) )	// Still in range?
					{
						vCharEndOffs = vEndOffs.longType +vCharStartOffs;
					}
					// Else leave end offset like it is.
				}
				break;
			
			case CHUNK_TYPE_ITEM:
			  #if JOKER_CHUNK_TYPES
				cout << "\nChunk type: Item";
			  #endif
				// Item? Fetch delimiter char:
				if( hi.mParamCount > 4 )
				{
					hi.mParam[4].GetValue( vItemDel, s, VALUE_TYPE_TEXT );
					if( vItemDel.textType->GetLength() != 1 )	// Not set or unusable value? Default to comma.
						vChunkDelimiter = ',';
					else
					{
						vItemDel.textType->SetOffset(0);
						vChunkDelimiter = vItemDel.textType->Read<char>();
					}
					delete vItemDel.textType;
				}
				else
					vChunkDelimiter = ',';
				// Drop through!
				
			case CHUNK_TYPE_LINE:	// Uses default, which is return.
            {
			  #if JOKER_CHUNK_TYPES
				if( vChunkType.longType == CHUNK_TYPE_LINE )
					cout << "\nChunk type: Line";
			  #endif
				long		currItem = 1;
				bool		didStart = false;
				
				/* Since vCharEndOffs and vCharStartOffs already countain start & end of chunked
					value, we can leave them the way they are if an item number is too low or
					too large. I hope nobody minds that negative item lengths mean return all
					of the string. */
				
				if( vEndOffs.longType == 0 )	// Only one item?
					vEndOffs = vStartOffs;	// Make it same as start.
				
				// Now move to chunk start and parse for substring:
				vTargetValue.textType->SetOffset( vCharStartOffs );
				while( vTargetValue.textType->GetOffset() <= vCharEndOffs )
				{
					if( currItem == vStartOffs.longType
						&& !didStart )	// Found first item user wants?
					{
						vCharStartOffs = vTargetValue.textType->GetOffset();	// Remember offset to it.
						didStart = true;
					}
					
					if( vTargetValue.textType->Read<char>() == vChunkDelimiter )	// Found delimiter?
						currItem++;		// Increase item counter.
					
					if( currItem > vEndOffs.longType )	// We're just past last item user wanted?
					{
						vCharEndOffs = vTargetValue.textType->GetOffset() -1;	// Remember offset (-1 since we don't want trailing delimiter)
						if( !didStart )	// First item & empty?
						{
							vCharStartOffs = vCharEndOffs;
							didStart = true;
						}
						break;	// Exit loop.
					}
				}
				if( !didStart )	// Not enough items in string?
					vCharStartOffs = vCharEndOffs = vTargetValue.textType->GetOffset();
				break;
            }
			
			case CHUNK_TYPE_WORD:
			  #if JOKER_CHUNK_TYPES
				cout << "\nChunk type: Word";
			  #endif
				long		currWord = 0;
				bool		wdidStart = false,
							wdidEnd = false;
				
				/* Since vCharEndOffs and vCharStartOffs already countain start & end of chunked
					value, we can leave them the way they are if an item number is too low or
					too large. I hope nobody minds that negative item lengths mean return all
					of the string. */
				
				if( vEndOffs.longType == 0 )	// Only one item?
					vEndOffs = vStartOffs;	// Make it same as start.
				
				// Now move to chunk start and parse for substring:
				vTargetValue.textType->SetOffset( vCharStartOffs );
				vCharSvStartOffs = vCharStartOffs;	// Remember since we change vCharStartOffs below.
				while( vTargetValue.textType->GetOffset() <= vCharEndOffs )
				{
					switch( vTargetValue.textType->Read<char>() )
					{
						case ' ':
						case '\n':
						case '\t':
							if( !wdidEnd )	// If this is the first space after a word ...
								currWord++;		// ... increase item counter.
							wdidEnd = true;	// & make sure  we remember not to increase counter again.
							break;
						
						default:
							wdidEnd = false;
							if( vTargetValue.textType->GetOffset() == (vCharSvStartOffs +1) )	// This was first character?
								currWord++;	// Make sure this registers as first word.
							
							if( currWord == vStartOffs.longType
								&& !wdidStart )	// Found first word user wants?
							{
								vCharStartOffs = vTargetValue.textType->GetOffset() -1;	// Remember offset to it.
								wdidStart = true;
							}
							break;
					}
					
					if( currWord > vEndOffs.longType
						&& wdidEnd )	// We're just past last word user wanted?
					{
						vCharEndOffs = vTargetValue.textType->GetOffset() -1;	// Remember offset (-1 since we don't want trailing delimiter)
						break;	// Exit loop.
					}
				}
				break;
		}
	}
	
  #if JOKER_CHUNK_TYPES
	cout << "\nChunk range (chars): " << vCharStartOffs << "-" << vCharEndOffs;
  #endif
  
  	// Save changes to chunk range:
	vCallRec.mChunkStartOffs = vCharStartOffs;
	vCallRec.mChunkEndOffs = vCharEndOffs;
	
	// We're responsible of getting rid of the text munger returned.
	delete vTargetValue.textType;
}

/* --------------------------------------------------------------------------------
	HyperCountChunkInstruction:
		This counts the number of items, words, lines, characters or whatever in
		a string. Used by the "number of chars of <string>" construct and the
		likes.
		
		This is also registered as the one-argument function "length".
		
	HI's FIELDS:
		0		-	Value to get chunk from.
		1		-	Chunk type to get.
		2		-	Item delimiter (items only). Usually this just points at the
					local variable "the itemdelimiter"
		Result	-	Number of chars/items/lines/words
   ----------------------------------------------------------------------------- */

void	HyperCountChunkInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage	vTargetValue,
					vChunkType,
					vCountStorage,
					vItemDel;
	size_t			vCount;
	TalkValue*		vDestValue;
	
	hi.mParam[0].GetValue( vTargetValue, s, VALUE_TYPE_TEXT );	// Value to get chunk from.
	
	if( hi.mParamCount == 1 )	// Called as length() or the length of xy:
		vChunkType.longType = CHUNK_TYPE_CHARACTER;	// Force to characters.
	else if( hi.mParamCount == 2
			|| hi.mParamCount == 3 )
		hi.mParam[1].GetValue( vChunkType, s, VALUE_TYPE_LONG );	// Chunk unit (chunk type).
	else
		throw runtime_error( "The length() function takes one parameter only." );
		
	char		vChunkDelimiter = '\n';
	
	if( vTargetValue.textType->GetLength() == 0 )
		vCount = 0;
	else
	{
		// Now parse this chunk and count:
		switch( vChunkType.longType )
		{
			case CHUNK_TYPE_CHARACTER:
			  #if JOKER_CHUNK_TYPES
				cout << "\nChunk type: Character";
			  #endif
				vCount = vTargetValue.textType->GetLength();
				break;
			
			case CHUNK_TYPE_BYTE:
			  #if JOKER_CHUNK_TYPES
				cout << "\nChunk type: Byte";
			  #endif
				vCount = vTargetValue.textType->GetLength();
				break;
			
			case CHUNK_TYPE_ITEM:
			  #if JOKER_CHUNK_TYPES
				cout << "\nChunk type: Item";
			  #endif
				// Item? Fetch delimiter char:
				hi.mParam[2].GetValue( vItemDel, s, VALUE_TYPE_TEXT );
				if( vItemDel.textType->GetLength() != 1 )	// Not set or unusable value? Default to comma.
					vChunkDelimiter = ',';
				else
				{
					vItemDel.textType->SetOffset(0);
					vChunkDelimiter = vItemDel.textType->Read<char>();
				}
				delete vItemDel.textType;
				// Drop through!
				
			case CHUNK_TYPE_LINE:	// Uses default, which is return.
			  #if JOKER_CHUNK_TYPES
				if( vChunkType.longType == CHUNK_TYPE_LINE )
					cout << "\nChunk type: Line";
			  #endif
				vCount = 1;
				
				// Now parse from beginning:
				vTargetValue.textType->SetOffset( 0 );
				while( vTargetValue.textType->GetOffset() < vTargetValue.textType->GetLength() )
				{
					if( vTargetValue.textType->Read<char>() == vChunkDelimiter )	// Found delimiter?
						vCount++;		// Increase item counter.
				}
				break;
			
			case CHUNK_TYPE_WORD:
            {
			  #if JOKER_CHUNK_TYPES
				cout << "\nChunk type: Word";
			  #endif
				bool		wdidStart = false,
							wdidEnd = false;
				
				vCount = 0;
				
				// Now parse string and count words:
				vTargetValue.textType->SetOffset( 0 );
				while( vTargetValue.textType->GetOffset() < vTargetValue.textType->GetLength() )
				{
					switch( vTargetValue.textType->Read<char>() )
					{
						case ' ':
						case '\n':
						case '\t':
							if( !wdidEnd && wdidStart )	// If this is the first space after a word ...
							{
								vCount++;		// ... increase item counter.
								wdidStart = false;
							}
							wdidEnd = true;	// & make sure  we remember not to increase counter again.
							break;
						
						default:
							wdidEnd = false;
							wdidStart = true;
							break;
					}
				}
				// In case there's no space at the end, we need to nudge the counter by one:
				if( wdidStart && !wdidEnd )
					vCount++;
				break;
            }
			
			default:
				throw runtime_error( "Invalid chunk type passed to \"number of ...\" function." );
		}
	}
		
	// We're responsible of getting rid of the text munger returned.
	delete vTargetValue.textType;
	
	// Now assign count to result:
	vCountStorage.longType = vCount;
	hi.mResult.GetValue( vDestValue, s );
	vDestValue->SetValue( vCountStorage, VALUE_TYPE_LONG, true );
}


/* --------------------------------------------------------------------------------
	HyperGetChunkRefInstruction:
		After a row of HyperGetChunkOffsInstructions, this takes the chunk range
		they calculated and extracts the appropriate range from the chunked value.
		A reference to this sub-range is then returned.
		
	HI's FIELDS:
		0		-	The value to get the chunk from.
		Result	-	a subsection of mParam[0].
   ----------------------------------------------------------------------------- */

void	HyperGetChunkRefInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	TalkValue*		vDestValue;
	TalkValue*		vChunkedValue;
	ValueStorage	vValStor;
		
	hi.mResult.GetValue( vDestValue, s );
	hi.mParam[0].GetValue( vChunkedValue, s );
	
	vValStor.valueType = vChunkedValue;
	vDestValue->SetValue( vValStor, VALUE_TYPE_VALUE, false );	// Set its mValue member w/o following refs.
	
	vDestValue->SetChunkRange( vCallRec.mChunkStartOffs, vCallRec.mChunkEndOffs );
	
	// Reset chunk range for next use.
	vCallRec.mChunkStartOffs = vCallRec.mChunkEndOffs = 0;
}


/* --------------------------------------------------------------------------------
	HyperThrowCommand:
		"throw" an error string.
		
	HI's FIELDS:
		0		-	The error message.
		Result	-	*unused*.
   ----------------------------------------------------------------------------- */

void	HyperThrowCommand( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec )
{
	TalkValue*		vErrMsg;
		
	hi.mParam[0].GetValue( vErrMsg, s );
	
	s.SetExceptionValue( *vErrMsg );	// Store a copy of the error message where it can be retrieved after unwinding the stack.
	vCallRec.mException = true;			// Make sure main script loop knows there was an exception. The main loop is responsible for jumping

	// Maybe we should "goto" here ourselves?
}


#pragma mark -
#pragma mark [Utility Handlers]


/* --------------------------------------------------------------------------------
	CompareBeef:
		This function actually performs the compares for the = and <> operators.
		This tries to find the lowest common denominator for these two values
		and resorts to a string compare if they can't be made the same type.
	
	GIVES:
		bool	-	Are LHS and RHS the same or not.
	
	HI's FIELDS:
		0		-	LHS argument.
		1		-	RHS argument.
		Result	-	*undefined*
   ----------------------------------------------------------------------------- */

bool	CompareBeef( HyperInstruction& hi, ValueStack& s )
{
	ValueStorage	vLHS,
					vRHS;
	bool			vResult;
	unsigned long	vAvailableTypes;
	
	// Get what types we can get from our arguments:
	vAvailableTypes = hi.mParam[0].GetAvailableTypes( s );
	vAvailableTypes &= hi.mParam[1].GetAvailableTypes( s );	// Mask out those they don't have in common.
	
	if( (vAvailableTypes & VALUE_TYPE_LONG) == VALUE_TYPE_LONG )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_LONG );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_LONG );
		
		// Calculate and store result:
		vResult = (vLHS.longType == vRHS.longType);
	}
	else if( (vAvailableTypes & VALUE_TYPE_BOOL) == VALUE_TYPE_BOOL )
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_BOOL );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_BOOL );
		
		// Calculate and store result:
		vResult = (vLHS.boolType == vRHS.boolType);
	}
	else	// Else compare as strings, which every value can export to:
	{
		// Get 1st arg:
		hi.mParam[0].GetValue( vLHS, s, VALUE_TYPE_TEXT );
		
		// Get 2nd arg:
		hi.mParam[1].GetValue( vRHS, s, VALUE_TYPE_TEXT );
		
		// Calculate and store result:
		vResult = ((*vLHS.textType) == (*vRHS.textType));	// TextMunger overrides operator==.
		
		// The TextMungers returned are our responsibility:
		delete vLHS.textType;
		delete vRHS.textType;
	}
	
	return vResult;
}