/*	***************************************************************************

	PROJECT:	Joker
	
	FILE:		TalkURLValue.cpp
	
	PURPOSE:	A value of a variable.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-11-28	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkURLValue.h"
#include	<cstdlib>
#include	"strcase.h"


#pragma mark [Globals]

/* --------------------------------------------------------------------------------
	Globals/Static Variables:
   ----------------------------------------------------------------------------- */

// Put these li'l critters here.


#pragma mark -
#pragma mark [Class Methods]

/* --------------------------------------------------------------------------------
	* DESTRUCTOR:
	
	REVISIONS:
		2000-10-24	UK		Created.
   ----------------------------------------------------------------------------- */

TalkURLValue::~TalkURLValue()
{
}


/* --------------------------------------------------------------------------------
	GetValue:
		Retrieve the value we keep as a certain type.
	
	TAKES:
		desiredType	-	The data type to return.
	
	GIVES:
		returnValue	-	In this the value is returned. All of this data is
						allocated by TalkURLValue for the caller, and from then on
						belongs to the caller. TalkURLValue will not dispose of any
						returned data.
	
	REVISIONS:
		2000-10-24	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::GetValue( ValueStorage &returnValue, unsigned long desiredType,
								size_t startOffs, size_t endOffs ) const
{
	if( endOffs != 0 )	// Chunk expression!
	{
		GetChunk( returnValue, desiredType, startOffs, endOffs );
		return;
	}
	
	TextMunger*		vText;
	
	vText = DownloadURLData();
	
	switch( desiredType )
	{
		case VALUE_TYPE_LONG:
			returnValue.longType = TextToLong( *vText );
			delete vText;
			break;
		
		case VALUE_TYPE_DOUBLE:
			returnValue.doubleType = TextToDouble( *vText );
			delete vText;
			break;
		
		case VALUE_TYPE_TEXT:
			returnValue.textType = vText;
			// Caller takes over vText, no need to dispose.
			break;
		
		case VALUE_TYPE_BOOL:
			returnValue.boolType = TextToBool( *vText );
			delete vText;
			break;
		
		case VALUE_TYPE_POINT:
			TextToPoint( *vText, returnValue.pointType );
			delete vText;
			break;
		
		case VALUE_TYPE_RECT:
			TextToRect( *vText, returnValue.rectType );
			delete vText;
			break;
		
		case VALUE_TYPE_COLOR:
			TextToColor( *vText, returnValue.colorType );
			delete vText;
			break;
			
		case VALUE_TYPE_VALUE:
			throw runtime_error( "Can't get value type." );
			break;
		
		default:
			throw runtime_error( "Caught unknown kind of value request (TalkURLValue)." );
	}
}


/* --------------------------------------------------------------------------------
	GetChunk:
		Get a chunk out of this value.
	
	TAKES:
		desiredType	-	The data type to return.
	
	GIVES:
		returnValue	-	In this the value is returned. All of this data is
						allocated by TalkURLValue for the caller, and from then on
						belongs to the caller. TalkURLValue will not dispose of any
						returned data.
	
	REVISIONS:
		2000-10-24	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::GetChunk( ValueStorage &returnValue, unsigned long desiredType,
								size_t startOffs, size_t endOffs ) const
{
	TextMunger*		vText;
	
	if( startOffs >= endOffs )	// Empty chunk!
		vText = new TextMunger( "" );
	else
	{
		vText = DownloadURLData();	// Convert our value to text.
		if( (vText->GetLength() -endOffs) > 0 )
		{
			vText->SetOffset( endOffs );
			vText->DeleteData( vText->GetLength() -endOffs );	// Delete after chunk end.
		}
		if( startOffs > 0 )
		{
			vText->SetOffset( 0 );
			vText->DeleteData( startOffs );
		}
	}
	
	switch( desiredType )
	{
		case VALUE_TYPE_LONG:
			returnValue.longType = TextToLong( (*vText) );
			delete vText;
			break;
		
		case VALUE_TYPE_DOUBLE:
			returnValue.doubleType = TextToDouble( (*vText) );
			delete vText;
			break;
		
		case VALUE_TYPE_TEXT:
			returnValue.textType = vText;	// Caller now takes care of vText, no need to dispose.
			break;
		
		case VALUE_TYPE_BOOL:
			returnValue.boolType = TextToBool( (*vText) );
			delete vText;
			break;
		
		case VALUE_TYPE_RECT:
			TextToRect( (*vText), returnValue.rectType );
			delete vText;
			break;
		
		case VALUE_TYPE_COLOR:
			TextToColor( (*vText), returnValue.colorType );
			delete vText;
			break;
		
		case VALUE_TYPE_POINT:
			TextToPoint( (*vText), returnValue.pointType );
			delete vText;
			break;
		
		default:
			delete vText;
			throw runtime_error( "Can't convert chunk to this type (TalkURLValue)." );
	}
}


/* --------------------------------------------------------------------------------
	SetValue:
		Change our value to one of specified type and value.
	
	TAKES:
		providedType	-	The data type passed.
		inValue			-	The value to assign to us. We copy what's passed in,
							the caller retains ownership of the values passed in.
	
	GIVES:
		-
	
	REVISIONS:
		2000-10-24	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::SetValue( ValueStorage inValue, unsigned long providedType,
								bool followRefs, size_t startOffs, size_t endOffs )
{
	bool		disposeInValue = false;
	
	/* If chunk addressing was used to set this, we convert both the input
		value and this variable's value to strings and then perform the chunk
		replacement on that. Then we fake as if the user had passed this value
		in for assignment, by changing providedType to VALUE_TYPE_TEXT and
		making inValue point at the chunk-replaced TextMunger. */
	if( endOffs != 0
		&& followRefs )	// But if we're replacing a ref, don't do chunking!
	{
		TextMunger*		newValue;
		
		CopyToText( inValue, providedType, newValue );			// Get value to insert as a TextMunger.
		inValue.textType = DownloadURLData();					// Get our value as a TextMunger.
		
		// Actually insert input value into ours:
		inValue.textType->SetOffset( startOffs );
		inValue.textType->DeleteData( endOffs -startOffs );	// Kill previous data at that offset.
		inValue.textType->InsertMunger( (*newValue) );	// Write new munger into place of prevoius data.
		
		if( providedType != VALUE_TYPE_TEXT )	// We did conversion, Text Munger in newValue is our responsibility!
			delete newValue;	// Get rid of input value's text munger.
		
		providedType = VALUE_TYPE_TEXT;	// Re-label inValue to be of type text.
		disposeInValue = true;
	}
	else	// Convert inValue to text -- URLs can only contain text data:
	{
		TextMunger*		txValue;
		
		CopyToText( inValue, providedType, txValue );
		if( providedType != VALUE_TYPE_TEXT )
		{
			providedType = VALUE_TYPE_TEXT;
			inValue.textType = txValue;
			disposeInValue = true;
		}
	}
	
	// Set new values:
	switch( providedType )
	{
		case VALUE_TYPE_VALUE:
		case VALUE_TYPE_BOOL:
		case VALUE_TYPE_LONG:
		case VALUE_TYPE_DOUBLE:
		case VALUE_TYPE_POINT:
		case VALUE_TYPE_COLOR:
		case VALUE_TYPE_RECT:
			throw logic_error("URLs can only contain text data. Conversion failed.");
			break;
		
		case VALUE_TYPE_TEXT:
			UploadURLData( inValue.textType );
			break;
		
		default:
			throw runtime_error( "Can't accept this type (TalkURLValue)." );
	}
	
	
	if( disposeInValue )	// Kill chunk-value or converted value:
	{
		switch( providedType )
		{
			case VALUE_TYPE_BOOL:
			case VALUE_TYPE_LONG:
			case VALUE_TYPE_DOUBLE:
			case VALUE_TYPE_VALUE:
			case VALUE_TYPE_RECT:
			case VALUE_TYPE_COLOR:
			case VALUE_TYPE_POINT:
				break;
			
			case VALUE_TYPE_TEXT:
				delete inValue.textType;
				break;
			
			default:
				throw runtime_error( "Can't dispose of this type (TalkURLValue)." );
		}
	}
}


/* --------------------------------------------------------------------------------
	GetAvailableTypes:
		Return the types this value can provide. This is a pretty fast means of
		preflight-checking whether a value can be used and should be preferred to
		catching exceptions thrown when trying to GetValue().
	
	TAKES:
		-
	
	GIVES:
		unsigned long	-	A 32-bit flags value with the bit of each data type
							set which this can return.
	
	REVISIONS:
		2000-10-24	UK		Created.
   ----------------------------------------------------------------------------- */

unsigned long	TalkURLValue::GetAvailableTypes() const
{
	return VALUE_TYPE_TEXT;		// FIX ME! Should analyze data.
}


/* --------------------------------------------------------------------------------
	CopyValueTo:
		Set another value to our value, using our native type. If we are a
		reference of type VALUE_TYPE_VALUE, this copies the value of the value
		we point to.
	
	TAKES:
		vDestValue	-	The value to export our value to.
	
	GIVES:
		-
	
	REVISIONS:
		2000-10-24	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::CopyValueTo( TalkValue& vDestValue, bool followSrcRefs,
									bool followDstRefs ) const
{
	ValueStorage		tmpValue;
	
	tmpValue.textType = DownloadURLData();	// Munger returned is our responsibility!
	
	vDestValue.SetValue( tmpValue, VALUE_TYPE_TEXT, followDstRefs );
	delete tmpValue.textType;	// Get rid of it, SetValue() makes a copy.
	vDestValue.KillList();
}


/* --------------------------------------------------------------------------------
	GetListEntry:
		Get an array entry from this item.
	
	TAKES:
		returnValue	-	This will be set to a pointer to the array value.
		index		-	The array index of the item to look up.
	
	GIVES:
		-
	
	REVISIONS:
		2000-10-24	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::GetListEntry( TalkValue* &returnValue, const TextMunger& index )
{
	throw logic_error("URLs can't contain list items.");
}


/* --------------------------------------------------------------------------------
	SetList:
		Make this array a copy of the array passed in.
	
	TAKES:
		vList		-	The list to copy in.
	
	GIVES:
		-
	
	REVISIONS:
		2000-10-24	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::SetList( const TalkValueList& vList )
{
	throw runtime_error("Can't write a list to a URL.");
}


void	TalkURLValue::CopyListTo( TalkValueList& vList ) const
{
	throw runtime_error("Expected list, found URL.");
}


/* --------------------------------------------------------------------------------
	SetChunkRange:
		Restrict this value to a range.
	
	TAKES:
		offsStart	-	A start offset.
		offsEnd		-	An end offset.
	
	GIVES:
		-
	
	REVISIONS:
		2000-10-24	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::SetChunkRange( size_t offsStart, size_t offsEnd )
{
	throw logic_error("Can't set a file's chunk range.");
}


/* --------------------------------------------------------------------------------
	Delete:
		Delete a range of text or the entire file.
		
		This involves downloading the text, changing it and uploading it again
		using the platform-specific UploadURLData() and DownloadURLData() methods.
		For deleting a file, the platform-specific KillURLFile() method is called.
	
	TAKES:
		startOffs	-	A start offset.
		endOffs		-	An end offset.
	
	GIVES:
		-
	
	REVISIONS:
		2000-10-24	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::Delete( size_t startOffs, size_t endOffs )
{
	if( startOffs == 0 && endOffs == 0 )
		KillURLFile();
	else
	{
		TextMunger*		vText;
	
		vText = DownloadURLData();
		TexMunAPtr		theText(vText);
		vText->SetOffset( startOffs );
		vText->DeleteData( endOffs -startOffs );
		
		UploadURLData( vText );
	}
}



