/*	***************************************************************************

	PROJECT:	TalkValue
	
	FILE:		TalkValue.h
	
	PURPOSE:	Base class for any value like variables, expressions etc.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-11-28	UK		Created.
				
	************************************************************************ */

#ifndef TALKVALUE_H
#define TALKVALUE_H

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */
   
#include	"TextMunger.h"
#include	"TalkEntity.h"
#include	<map>

#ifdef COMPILING_SAL
#include	"XCanvas.h"
#endif

#pragma mark [Constants]

/* --------------------------------------------------------------------------------
	Constants:
   ----------------------------------------------------------------------------- */

// Bits for the different data types. All types we can retrieve or accept are set:
enum
{
	VALUE_TYPE_LONG			= (1 << 0),		// Integer.
	VALUE_TYPE_DOUBLE		= (1 << 1),		// Floating-point number.
	VALUE_TYPE_BOOL			= (1 << 2),		// boolean value.
	VALUE_TYPE_TEXT			= (1 << 3),		// Text string.
	VALUE_TYPE_VALUE		= (1 << 4),		// Another value.
	VALUE_TYPE_POINT		= (1 << 5),		// A coordinate on the screen.
	VALUE_TYPE_RECT			= (1 << 6),		// A rectangle on the screen.
	VALUE_TYPE_COLOR		= (1 << 7),		// An RGB color value.
	VALUE_TYPE_ENTITY		= (1 << 8),		// An object.
	VALUE_TYPE_LIST			= (1 << 9)		// An array.
};

#pragma mark [Data Types]

/* --------------------------------------------------------------------------------
	Data Types:
   ----------------------------------------------------------------------------- */

#ifndef COMPILING_SAL

struct	LongPoint
{
	long	x;	// horz.
	long	y;	// vert.
};

struct	LongColor
{
	unsigned short	red;
	unsigned short	green;
	unsigned short	blue;
};

struct	LongRect
{
	long	left;
	long	top;
	long	right;
	long	bottom;
};

#else /*COMPILING_SAL*/

typedef XCanvasRect LongRect;
typedef XCanvasPoint LongPoint;
typedef XCanvasColor LongColor;

#endif /*COMPILING_SAL*/

class	TalkValue;		// Forward.

union ValueStorage
{
	long			longType;	// Integer.
	double			doubleType;	// Floating-point number.
	bool			boolType;	// Boolean value.
	TextMunger*		textType;	// Text string.
	TalkValue*		valueType;	// Just reference to another value.
	LongPoint		pointType;	// Screen coordinate.
	LongColor		colorType;	// RGB Color value.
	LongRect		rectType;	// Screen area.
	TalkEntity*		entityType;	// An object.
};

typedef std::map<TextMunger,TalkValue*>	TalkValueList;

#pragma mark [Class Declaration]

/* --------------------------------------------------------------------------------
	Class declaration:
   ----------------------------------------------------------------------------- */

class	TalkValue : public TalkEntity
{
protected:
	// Put your member variables here.

public:
			TalkValue() : TalkEntity()		{};
	virtual	~TalkValue()					{};
	
	virtual unsigned long	GetAvailableTypes() const = 0;
	virtual unsigned long	GetNativeType() const = 0;
	
	virtual void	GetValue( ValueStorage &returnValue, unsigned long desiredType, size_t startOffs = 0, size_t endOffs = 0 ) const = 0;
	virtual void	SetValue( ValueStorage returnValue, unsigned long providedType, bool followRefs = false, size_t startOffs = 0, size_t endOffs = 0 ) = 0;
	virtual void	CopyValueTo( TalkValue& vDestValue, bool followSrcRefs = false, bool followDstRefs = false ) const	= 0;
	
	virtual void	GetListEntry( TalkValue* &returnValue, const TextMunger& index ) = 0;
	virtual void	SetList( const TalkValueList& vList ) = 0;
	virtual void	CopyListTo( TalkValueList& vList ) const = 0;
	virtual void	KillList() = 0;
	
	virtual void	SetChunkRange( size_t offsStart, size_t offsEnd ) = 0;	// For references only.
	virtual bool	GetChunkRange( size_t& offsStart, size_t& offsEnd ) const = 0;	// see above.
	virtual void	ClearChunkRange() = 0;
	
	virtual void	Delete( size_t startOffs, size_t endOffs = 0 ) = 0;
	virtual void	Delete()	{ Delete( 0, 0 ); };	// Make sure calls to this class's TalkEntity part also arrive at our Delete() method.
	
	virtual	TextMunger*		GetKeys() const = 0;
	virtual long			CountEntries() const = 0;
	
public:
	static long			TextToLong( const TextMunger& inText );
	static double		TextToDouble( const TextMunger& inText );
	static bool			TextToBool( const TextMunger& inText );
	static void			TextToPoint( const TextMunger& inText, LongPoint& outPos );
	static void			TextToColor( const TextMunger& inText, LongColor& outPos );
	static void			TextToRect( const TextMunger& inText, LongRect& outPos );
	static void			LongToText( long vLong, TextMunger* &outText );
	static void			DoubleToText( double vDouble, TextMunger* &outText );
	static void			BoolToText( bool vBoolean, TextMunger* &outText );
	static void			PointToText( const LongPoint& vLong, TextMunger* &outText );
	static void			ColorToText( const LongColor& pos, TextMunger* &outText );
	static void			RectToText( const LongRect& vLong, TextMunger* &outText );
	static long			DoubleToLong( double vDouble );
	static void			CopyToText( ValueStorage inValue, unsigned long providedType,
									TextMunger* &outMunger );
	static unsigned long	GetTextTypesAvailable( const TextMunger& vText );
};


#pragma mark [Prototypes]

/* --------------------------------------------------------------------------------
	Prototypes:
   ----------------------------------------------------------------------------- */



#endif /*TALKVALUE_H*/









