/*	***************************************************************************

	PROJECT:	Joker
	
	FILE:		JokerMain.cpp
	
	PURPOSE:	Main file for Joker
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-06-22	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"JokerMain.h"
#include	"TextMunger.h"
#include	"HyperTalk.h"
#include	"TalkTokenize.h"
#include	"TalkVarValue.h"
#include	"parse_error.h"
#include	"GetPartialPath.h"
#include	<MacTypes.h>
#include	<LowMem.h>
#include	<Resources.h>
#include	<Dialogs.h>
#include	<MacWindows.h>
#include	<TextEdit.h>
#include	<Quickdraw.h>
#include	<Sound.h>
#include	<TextUtils.h>
#include	<AppleEvents.h>
#include	<AEDataModel.h>
#include	<Files.h>
#include	<MacErrors.h>
#include	<iostream>
#include	<sioux.h>
#include	"MacUtils.h"

#pragma mark [Globals]

/* --------------------------------------------------------------------------------
	Globals:
   ----------------------------------------------------------------------------- */

bool		gQuit = false;

#pragma mark [Prototypes]

/* --------------------------------------------------------------------------------
	Prototypes:
   ----------------------------------------------------------------------------- */

pascal OSErr	MacJokerAppleEvtHandler( const AppleEvent *event, AppleEvent *reply, long refCon );
void			MacInitToolbox();
void			MacRunOneFile( char* vFileName );
void			MacProcessOneEvent( bool isBusy = false );
OSErr			SendOneAEvtMessage( AppleEvent *event, AppleEvent *reply,
									AEEventClass pClass, AEEventID pID );

#pragma mark -
#pragma mark [Implementation]

/* --------------------------------------------------------------------------------
	SendOneAEvtMessage:
		Dispatch an "apple event" message to a script named "cgi.txt" next to
		the Joker application.
	
	TAKES:
		event	-	The Apple Event to handle.
		reply	-	The reply where we can store our result.
		pClass	-	The class already extracted from the event.
		pID		-	The ID already extracted from the event.
	
	GIVES:
		-
	
	REVISIONS:
		2001-12-13	UK		Made const-correct.
		2000-11-07	UK		Created.
   ----------------------------------------------------------------------------- */

OSErr			SendOneAEvtMessage( const AppleEvent *event, AppleEvent *reply,
									AEEventClass pClass, AEEventID pID )
{
	OSErr			theErr;
	char*			vData;
	FILE*			vFile;
	size_t			vLength;
	
	// Open file:
	vFile = fopen( "cgi.txt", "r" );
	if( !vFile )
		return errAEEventNotHandled;
	
	// Get length:
	if( fseek( vFile, 0, SEEK_END ) != 0 )
		return errAEEventNotHandled;
	vLength = ftell( vFile );
	fseek( vFile, 0, SEEK_SET );	// Move mark back to start.
	
	// Allocate memory for file's contents & read them:
	vData = (char*) malloc( vLength );
	if( !vData )
		return errAEEventNotHandled;
	
	if( fread( vData, sizeof(char), vLength, vFile ) != vLength )
		return errAEEventNotHandled;
	
	TextMunger		script( vData, vLength );
	HyperTalk		interpreter;
	
	interpreter.Retain();	// Must call this or when Run() calls Release() it will try to call delete on our stack object.
	
	free( vData );
						
	// Now do the interesting stuff:
	try {
		std::cout << "\nWe are now going to execute the following script:\n";
		std::cout << script.String() << "\n";
		std::cout << "\nExecuting script.\n";
		
		TextMunger::FreeTempMemory();	// Free memory used by String();
		
		interpreter.Tokenize( script );
		interpreter.Parse();
		
		// Now send our AppleEvent message:
		TalkMemLocation		vResult;
		ValueStack			vStack;
		char				vAETypeStr[5];
		
		vAETypeStr[4] = 0;	// zero-terminate.
		
		// First push params on the stack:
		BlockMoveData( &pClass, vAETypeStr, sizeof(pClass) );	// Class.
		vStack.push_back( new TalkVarValue( TextMunger( vAETypeStr ) ) );
		BlockMoveData( &pID, vAETypeStr, sizeof(pID) );	// ID.
		vStack.push_back( new TalkVarValue( TextMunger( vAETypeStr ) ) );
		vStack.push_back( new TalkVarValue( TextMunger( "<Unknown Sender>" ) ) );	// Sender.
		vStack.push_back( new TalkVarValue( 3L ) );	// Param count.
		
		// Also keep a pointer to the Apple Event with the stack so commands like "reply" can access it:
		vStack.SetCurrMacAEvt( event );
		vStack.SetCurrMacAEReply( reply );
		
		// Actually run script:
		vResult.mType = MEM_LOCATION_TYPE_INVALID;	// We don't care about return value.
		try
		{
			interpreter.SendMessage( TextMunger( "appleEvent" ), &vStack, vResult, false );	// This gets rid of the params.
			
			// Now look whether somebody took care of it:
			if( vStack.GetCurrMacAEHandled() )
				theErr = noErr;
			else
				theErr = errAEEventNotHandled;
		}
		catch( exception& theExc )
		{
			std::cout << "Error: " << theExc.what() << endl;
			
			theErr = paramErr;
		}
		catch( ... )
		{
			std::cout << "Encountered unknown exception." << endl;
			
			theErr = paramErr;
		}
		
		std::cout << "Execution finished.";
		
		
	}
	catch( parse_error& pe )
	{
		char		errStr[256];
		
		std::cout << "\nError at characters " << pe.GetStartChar() << " to ";
		std::cout << pe.GetEndChar() << ": " << pe.what() << "\n";
		
		script.SetOffset( pe.GetStartChar() );
		script.ReadData( errStr, pe.GetEndChar() -pe.GetStartChar() );
		errStr[pe.GetEndChar() -pe.GetStartChar()] = 0;
		
		std::cout << "\t\"" << errStr << "\"\n\n";
		theErr = errAEEventNotHandled;
	}
	catch( bad_alloc& be )
	{
		std::cout << "\nOut of Memory: " << be.what() << "\n";
		
		theErr = memFullErr;
	}
	catch( exception& e )
	{
		std::cout << "\nError: " << e.what() << "\n";
		theErr = errAEEventNotHandled;
	}
	
	return theErr;
}


/* --------------------------------------------------------------------------------
	MacJokerAppleEvtHandler:
		Joker's Apple Event handler. This reacts on Apple Events and dispatches
		the appropriate system messages to the script. This also handles running
		scripts that are dropped onto Joker and such things.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2000-11-07	UK		Added support for arbitrary Apple Events.
   ----------------------------------------------------------------------------- */

pascal OSErr	MacJokerAppleEvtHandler( const AppleEvent *event, AppleEvent *reply, long refCon )
{
	AEEventClass	pClass;
	AEEventID		pID;
	DescType		returnedType;
	long			returnedSize;
	OSErr			theErr = noErr;
	
	theErr = AEGetAttributePtr( 	event, keyEventClassAttr,
							 		typeType, &returnedType,
							 		&pClass, sizeof( AEEventClass ), &returnedSize );
	if( theErr != noErr )
		return theErr;
	theErr = AEGetAttributePtr( 	event, keyEventIDAttr,
							 		typeType, &returnedType,
							 		&pID, sizeof( AEEventID ), &returnedSize );
	if( theErr != noErr )
		return theErr;
	
	switch( pClass )
	{
		case kCoreEventClass:
			switch( pID )
			{
				case kAEOpenApplication:
					MacRunOneFile(NULL);
					break;
				
				case kAEOpenDocuments:
					SendOneAEvtMessage( event, reply, pClass, pID );
					break;
				
				case kAEQuitApplication:
					gQuit = true;
					break;
				
				default:
					theErr = SendOneAEvtMessage( event, reply, pClass, pID );
			}
			break;
		
		default:
			theErr = SendOneAEvtMessage( event, reply, pClass, pID );
	}
	
	return( theErr );
}


/* --------------------------------------------------------------------------------
	MacInitToolbox:
		Initialize the Macintosh toolbox. If you don't do this, nothing works.
		Most of this can be removed if you're compiling for Carbon.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2000-10-27	UK		Created.
   ----------------------------------------------------------------------------- */

void	MacInitToolbox()
{
	SIOUXSettings.setupmenus = false;
	SIOUXSettings.standalone = false;
	SIOUXSettings.initializeTB = false;
	SIOUXSettings.asktosaveonclose = false;
	SIOUXSettings.showstatusline = true;
	SIOUXSettings.autocloseonquit = true;
	
	InitGraf( &qd.thePort );    // Start QuickDraw.
    InitFonts();                // Fonts, styles, text output.
    InitWindows();              // Window objects.
    InitMenus();                // Pull-Down and Pop-Up Menus.
    TEInit();                   // TextEdit - Text entry fields.
    InitDialogs( NULL );        // Dialog Windows.
    InitCursor();               // Mouse Cursor.
    
    // Install event handler for *any* Apple Event:
	OSErr err;
	err = AEInstallEventHandler( '****', '****', NewAEEventHandlerProc( MacJokerAppleEvtHandler ), 0, false );
	err = AEInstallEventHandler( kCoreEventClass, kAEOpenDocuments, NewAEEventHandlerProc( MacJokerAppleEvtHandler ), 0, false );
	err = AEInstallEventHandler( kCoreEventClass, kAEOpenApplication, NewAEEventHandlerProc( MacJokerAppleEvtHandler ), 0, false );
}


/* --------------------------------------------------------------------------------
	RunOneFile:
		Execute a script inside a file of specified name.
		
	REVISIONS:
		2000-10-27	UK		Created.
   ----------------------------------------------------------------------------- */

void	MacRunOneFile( char* vFileName )
{
	char		vTheFileName[256];
	char*		vArgv[16];
	char*		vStrPtr;
	short		x,
				vCount;
	Handle	vCompilerFlags;
	
	if( vFileName == NULL )
	{
		vFileName = vTheFileName;
		cout << "Enter name of script to run: ";
		cin >> vFileName;
	}
	
	/* Now get our command-line flags from a string list
		resource and put them in an array: */
	vCompilerFlags = Get1Resource( 'CST#', 128 );
	vCount = **((short**) vCompilerFlags);
	HLockHi( vCompilerFlags );
	
	vStrPtr = (char*) *vCompilerFlags;
	vStrPtr += sizeof(short);	// Move past counter.
	
	vArgv[0] = "Joker";		// Should be path, but we don't care.
	vArgv[1] = vFileName;	// File to open.
	for( x = 2; x < vCount +2; x++ )
	{
		vArgv[x] = vStrPtr;
		vStrPtr += strlen( vStrPtr ) +1;
	}
	
	JokerMain( vCount +2, vArgv );
	
	HUnlock( vCompilerFlags );
}


/* --------------------------------------------------------------------------------
	MacProcessOneEvent:
		This does the actual work during an event loop. It retrieves a new
		event using WaitNextEvent() and then dispatches it to the appropriate
		event handling functions.
		
	REVISIONS:
		2000-10-27	UK		Created.
   ----------------------------------------------------------------------------- */

void	MacProcessOneEvent( bool isBusy )
{
	EventRecord		vEvent;
	Boolean			isNull;
	
	isNull = WaitNextEvent( everyEvent, &vEvent, isBusy ? 1 : 10, NULL );
	if( (vEvent.what == keyDown) || !SIOUXHandleOneEvent( &vEvent ) )
	{
		switch( vEvent.what )
		{
			case keyDown:
			case autoKey:
				if( (vEvent.modifiers & cmdKey) == cmdKey )
				{
					switch( vEvent.message & charCodeMask )
					{
						case 'q':
						case 'Q':
							if( (vEvent.modifiers &cmdKey ) == cmdKey )
							{
								std::cout << "\n\nTerminating program ...\n";
								if( isBusy )
									std::cout << "Program will quit when all handlers have finished executing.\n";
								gQuit = true;
							}
							break;
						
						default:
							SIOUXHandleOneEvent( &vEvent );
							break;
					}
				}
				else
					SIOUXHandleOneEvent( &vEvent );
				break;
			
			case kHighLevelEvent:
				AEProcessAppleEvent( &vEvent );
				break;
			
			//default:
				//SIOUXHandleOneEvent( &vEvent );
		}
	}
}


/* --------------------------------------------------------------------------------
	HyperAppleEventInstruction:
		This handles the "appleEvent" message. When we arrive here the message
		wasn't trapped by any handlers. So we check whether it is an event Joker
		knows how to handle internally or otherwise we mark the event as unused
		so our AppleEvent handler can generate an error message.
	
	HI's FIELDS:
		0		-	event class
		1		-	event ID
		2		-	sender
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperAppleEventInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	AppleEventPtr	event;
	AppleEventPtr	reply;
	AEEventClass	pClass;
	AEEventID		pID;
	DescType		returnedType;
	long			returnedSize;
	OSErr			theErr = noErr;
	FSSpec			aFile;
	AEDescList		docList;
	long			i, n;
	AEKeyword		keyWrd;
	DescType		retType;
	Size			actualSize;
	
	// Get the AppleEvents to work with:
	event = s.GetCurrMacAEvt();
	reply = s.GetCurrMacAEReply();
	
	// Determine class and type:
	theErr = AEGetAttributePtr( 	event, keyEventClassAttr,
							 		typeType, &returnedType,
							 		&pClass, sizeof( AEEventClass ), &returnedSize );
	if( theErr != noErr )
		throw runtime_error( "Couldn't determine apple event class." );
	theErr = AEGetAttributePtr( 	event, keyEventIDAttr,
							 		typeType, &returnedType,
							 		&pID, sizeof( AEEventID ), &returnedSize );
	if( theErr != noErr )
		throw runtime_error( "Couldn't determine apple event ID." );
	
	// If it's an "open document" event, handle it:
	if( pClass == kCoreEventClass && pID == kAEOpenDocuments )
	{
		theErr =  AEGetParamDesc( event, keyDirectObject, typeAEList, &docList );	
		if( theErr != noErr )
			throw runtime_error( "Couldn't determine list of files to open." );
		theErr = AECountItems( &docList, &n );
		if( theErr != noErr )
			throw runtime_error( "Couldn't determine number of files to open." );
		
		// Get each document and open it:
		for (i = 1; i <= n; i++)
		{
			Str255		vFileName;
			char		vCFileName[256];
			
			theErr = AEGetNthPtr( &docList,
									i, 
									typeFSS, 
									&keyWrd,
									&retType, 
									&aFile, 
									sizeof( FSSpec ), 
									&actualSize );
			
			theErr = FSpGetFullPath( &aFile, vFileName );
			JokerPStringToC( vFileName, vCFileName );
		  #if COMPILING_SAL
			gApplication->OpenFile( vCFileName );
		  #else
			MacRunOneFile( vCFileName );
		  #endif
		}
		theErr = AEDisposeDesc( &docList );
	}
	else	// Else ...
		s.SetCurrMacAEHandled(false);	// ... make sure we know it wasn't handled.
}


/* --------------------------------------------------------------------------------
	JokerSpendTime:
		This is called after each line the compiler executes and at other
		regular intervals so you can process events while a script is running.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2000-10-27	UK		Created.
   ----------------------------------------------------------------------------- */

void	JokerSpendTime()
{
	MacProcessOneEvent(true);
}


/* --------------------------------------------------------------------------------
	main:
		Main entry point for Joker on MacOS. This does a small event loop and
		fetches 'odoc' Apple Events. In reponse to an 'oapp' Apple Event this
		asks the user to enter the name of a file to run.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2001-01-17	UK		Added comment on how to quit on startup.
		2000-10-26	UK		Created.
   ----------------------------------------------------------------------------- */

int		main( void )
{
	MacInitToolbox();
	
	std::cout << "Hit Command-Q to quit." << endl << endl;
	
	TalkTokenizer::SetNewline( '\n' );	// Make sure we use the right one for this platform.
	
	while( gQuit == false )
		MacProcessOneEvent(false);
	
	std::cout << "\nYou might have to hit Cmd-Q again to quit the console.\n\n";
	
	return	EXIT_SUCCESS;
}








