/*	***************************************************************************

	PROJECT:	Joker
	
	FILE:		mac/TalkURLValue.cpp
	
	PURPOSE:	Macintosh-specific implementation of a URL.
		
	COPYRIGHT:	(C) Copyright 2000 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		2000-10-26	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkURLValue.h"
#include	"JokerMain.h"
#include	"IterateDirectory.h"
#include	<cstdlib>
#include	<cstring>
#include	<URLAccess.h>
#include	<Script.h>
#include	<NumberFormatting.h>
#include	<TextUtils.h>
#include	<Resources.h>
#include	"MacUtils.h"


#pragma mark [Prototypes]

/* --------------------------------------------------------------------------------
	Prototypes:
   ----------------------------------------------------------------------------- */

pascal	void MyMacFolderIteratorProc( const CInfoPBRec *const cpbPtr,
										Boolean *quitFlag, void *yourDataPtr );
										
#pragma mark -
#pragma mark [Class Methods]


/* --------------------------------------------------------------------------------
	ReverseCString:
		Reverse the characters in a zero-terminated C string so the first letter
		is last and vice versa.
   ----------------------------------------------------------------------------- */

void	ReverseCString( char* vString )
{
	int		x, maxLen, halfLen;
	
	maxLen = strlen(vString);
	halfLen = maxLen /2;
	
	for( x = 0; x < halfLen; x++ )
	{
		char		tmp;
		
		tmp = vString[x];
		vString[x] = vString[maxLen -x -1];
		vString[maxLen -x -1] = tmp;
	}
}


void	JokerRsrcURLToMacPath( char* vURL, ResType* resType, short* resID, Str255 outPath )
{
	int			x;
	char		vURLStringData[512];
	char*		vURLString = vURLStringData;
	strcpy( vURLString, vURL );
	
	if( vURLString[7] == '/' )	// Absolute pathname.
		memmove( vURLString, vURLString +1, 512 );	// Remove leading slash.
	else
	{
		memmove( vURLString +1, vURLString, 511 );
		vURLString[7] = '/';	// On Mac, relative pathnames start with a separator character.
	}
	
	// Convert URL string to a Mac path.
	for( x = 0; vURLString[x] != 0; x++ )
	{
		if( vURLString[x] == '/' )
			vURLString[x] = ':';
		else if( vURLString[x] == ':' )
			vURLString[x] = '/';
	}
	
	vURLString += 7;	// Make sure we skip protocol designator.
	x -= 7;	// Make sure x stays string's length so we can use it to loop over the string backwards.
	
	char		vIdStr[256];
	short		currCharCount = 0;
	
	vIdStr[0] = 0;
	
	// Go backwards over string and extract ID/name and type:
	for( ; ((--x) > 0) && (vURLString[x] != ':') && (currCharCount < 256); )
	{
		vIdStr[currCharCount++] = vURLString[x];
		vIdStr[currCharCount] = 0;
	}
	
	ReverseCString( vIdStr );	// We collected the ID backwards, turn it around.
	
	if( x == 0 )
		throw runtime_error("Malformed \"rsrc:\" URL, couldn't find type before ID/name.");
	if( vURLString[x] != ':' )
		throw runtime_error("Resource name/ID in \"rsrc:\" URL too long.");
	
	char* theEnd;
	*resID = strtol( vIdStr, &theEnd, 10 );
	// FIX ME! check "theEnd" here to determine whether it was a valid number, otherwise do a per-name lookup.
	
	currCharCount = 0;
	
	// Now keep scanning backwards for res type:
	for( ; ((--x) > 0) && vURLString[x] != ':' && currCharCount < 256; )	// After the above, x is string's length!
	{
		vIdStr[currCharCount++] = vURLString[x];
		vIdStr[currCharCount] = 0;
	}
	
	ReverseCString( vIdStr );	// We collected the string backwards, turn it around.
	
	if( x == 0 )
		throw runtime_error("Malformed \"rsrc:\" URL, couldn't find file name before type.");
	if( vURLString[x] != ':' || currCharCount != 4 )
		throw runtime_error("Resource type in \"rsrc:\" URL must be four characters long.");
	
	vURLString[x] = 0;	// terminate string before start of type and ID so all that's left is file path.
	JokerCStringToP( vURLString, outPath );
	
	memmove( resType, vIdStr, 4 );
}


/* --------------------------------------------------------------------------------
	DownloadURLData:
		Fetch the data of the file pointed to by this URL and return it in a text
		munger.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2000-02-24	UK		Created.
   ----------------------------------------------------------------------------- */

TextMunger*	TalkURLValue::DownloadURLData() const
{
	TextMunger*		vResult;
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	size_t			vReadSize = 512;
	short			x;
	
	mTheURL.CopyToString( vURLString );
	if( strncmp( vURLString, "file://", 7 ) == 0 )	// String starts with file protocol?
	{
		if( vURLString[7] == '/' )	// Absolute pathname.
			memmove( vURLString, vURLString +1, 512 );	// Remove leading slash.
		else
		{
			memmove( vURLString +1, vURLString, 511 );
			vURLString[7] = '/';	// On Mac, relative pathnames start with a separator character.
		}
		
		// Convert URL string to a Mac path.
		for( x = 0; vURLString[x] != 0; x++ )
		{
			if( vURLString[x] == '/' )
				vURLString[x] = ':';
			else if( vURLString[x] == ':' )
				vURLString[x] = '/';
		}
		
		vURLString += 7;	// Make sure we skip protocol designator.
		
		FILE*		vFile;
		vResult = new TextMunger;
		
		vFile = fopen( vURLString, "r+" );
		if( vFile == NULL )
			throw runtime_error( "Couldn't open file of local URL." );
		
		// Read from that file in chunks of 512 bytes
		try
		{
			while( vReadSize == 512 )	// If fread returns less, it means we have reached end of file.
			{
				vReadSize = fread( vURLStringData, sizeof(char), vReadSize, vFile );
				vResult->InsertData( vURLStringData, vReadSize );
			}
		}
		catch( exception& err )
		{
			fclose( vFile );
			throw;
		}
		
		fclose( vFile );
	}
	else if( strncmp( vURLString, "rsrc://", 7 ) == 0 )	// String starts with Resource protocol?
	{
		short		refNum;
		Str255		vFName;
		Handle		resData = NULL;
		ResType		resType;
		short		resID;
		
		JokerRsrcURLToMacPath( vURLString, &resType, &resID, vFName );
		
		refNum = HOpenResFile( 0, 0L, vFName, fsRdPerm );
		if( refNum < 0 )
			throw runtime_error("Error opening resource file referenced by \"rsrc:\" URL.");
		try
		{
			UseResFile( refNum );	// In case it was already open.
			resData = Get1Resource( resType, resID );
			if( resData == NULL || ResError() != noErr )
				throw runtime_error( "Couldn't fetch resource referenced by \"rsrc:\" URL." );
			vResult = new TextMunger;
			HLock( resData );
			vResult->InsertData( *resData, GetHandleSize( resData ) );
			HUnlock( resData );
		}
		catch( exception& err )
		{
			if( resData )
				DisposeHandle( resData );
			CloseResFile( refNum );
			throw;
		}
		CloseResFile( refNum );
	}
	else
	{
	  #if TARGET_RT_MAC_CFM
		if( !URLAccessAvailable() )
			throw runtime_error( "URL Access is not available on this Macintosh." );
		
		OSStatus			err;
		URLOpenFlags		vFlags = kURLDisplayProgressFlag | kURLDisplayAuthFlag;
		Handle				vDestDataHandle;
		
		vDestDataHandle = NewHandle(0);
		if( MemError() != noErr || vDestDataHandle == NULL )
			throw bad_alloc();
		
		err = URLSimpleDownload( mTheURL.String(), (FSSpec*) NULL,
									vDestDataHandle,
									vFlags, NULL, (void*) 0 );
		if( err != noErr )
		{
			DisposeHandle( vDestDataHandle );
			throw runtime_error( "Download Failure." );
		}
		
		HLock( vDestDataHandle );
		try
		{
			vResult = new TextMunger( (*vDestDataHandle), GetHandleSize( vDestDataHandle ) );
			HUnlock( vDestDataHandle );
			DisposeHandle( vDestDataHandle );
		}
		catch( exception& err )
		{
			HUnlock( vDestDataHandle );
			DisposeHandle( vDestDataHandle );
			
			throw;
		}
		HUnlock( vDestDataHandle );
	  #else
		vResult = NULL;
		throw logic_error( "Downloading from remote URLs are not yet implemented on this platform." );
	  #endif
	}
  
	return vResult;
}


/* --------------------------------------------------------------------------------
	KillURLFile:
		Delete the file/folder referenced by this URL.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2001-07-15	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::KillURLFile()
{
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	size_t			vReadSize = 512;
	short			x;
	
	mTheURL.CopyToString( vURLString );
	if( strncmp( vURLString, "file://", 7 ) == 0 )	// String starts with file protocol?
	{
		if( vURLString[7] == '/' )	// Absolute pathname.
			memmove( vURLString, vURLString +1, 512 );	// Remove leading slash.
		else
		{
			memmove( vURLString +1, vURLString, 511 );
			vURLString[7] = '/';	// On Mac, relative pathnames start with a separator character.
		}
		
		// Convert URL string to a Mac path.
		for( x = 0; vURLString[x] != 0; x++ )
		{
			if( vURLString[x] == '/' )
				vURLString[x] = ':';
			else if( vURLString[x] == ':' )
				vURLString[x] = '/';
		}
		
		FSSpec		theFile;
		
		vURLString += 6;	// Make sure we skip protocol designator, but skip one less as we'll use first byte as length byte.
		vURLString[0] = strlen(vURLString) -1;
		
		if( FSMakeFSSpec( 0, 0L, (unsigned char*) vURLString, &theFile ) != noErr )
			throw runtime_error("Couldn't find file referenced by \"file:\" URL.");
		
		FSpDelete( &theFile );
		
	}
	else if( strncmp( vURLString, "rsrc://", 7 ) == 0 )	// String starts with Resource protocol? Then we just delete that resource.
	{
		short		refNum;
		Str255		vFName;
		Handle		resData = NULL;
		ResType		resType;
		short		resID;
		
		JokerRsrcURLToMacPath( vURLString, &resType, &resID, vFName );
		
		refNum = HOpenResFile( 0, 0L, vFName, fsRdWrPerm );
		if( refNum < 0 )
			throw runtime_error("Error opening resource file referenced by \"rsrc:\" URL.");
		try
		{
			UseResFile( refNum );	// In case it was already open.
			resData = Get1Resource( resType, resID );
			if( resData == NULL || ResError() != noErr )
				throw runtime_error( "Couldn't find resource referenced by \"rsrc:\" URL." );
			RemoveResource( resData );
		}
		catch( exception& err )
		{
			CloseResFile( refNum );
			throw;
		}
		CloseResFile( refNum );
	}
	else
		throw runtime_error("Deleting remote files is not supported.");
}

/* --------------------------------------------------------------------------------
	UploadURLData:
		Replace the data of the file pointed to by this URL through the data in
		the text munger passed.
	
	TAKES:
		vData	-	The text data to upload.
	
	GIVES:
		-
	
	REVISIONS:
		2000-02-24	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::UploadURLData( TextMunger* vData ) const
{
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	size_t			vReadSize = 512;
	
	mTheURL.CopyToString( vURLString );
	if( strncmp( vURLString, "file://", 7 ) == 0 )	// String starts with file protocol?
	{
		vURLString += 7;	// Make sure we skip protocol designator.
		
		if( vURLString[0] == '/' )	// Absolute pathname.
			memmove( vURLString, vURLString +1, 505 );	// Remove leading slash.
		else
		{
			memmove( vURLString +1, vURLString, 504 );
			vURLString[0] = '/';	// On Mac, relative pathnames start with a separator character.
		}
		
		// Convert URL string to a Mac path.
		short		x = 0;
		for( ; vURLString[x] != 0; x++ )
		{
			if( vURLString[x] == '/' )
				vURLString[x] = ':';
			else if( vURLString[x] == ':' )
				vURLString[x] = '/';
		}
		
		FILE*		vFile;
		
		vFile = fopen( vURLString, "w+" );
		if( vFile == NULL )
		{
			char		errStr[512];
			
			strcpy( errStr, "Couldn't open file \"" );
			strncat( errStr, vURLString, 512 );
			strncat( errStr, "\" of local URL.", 512 );
			
			throw runtime_error( errStr );
		}
		
		try
		{
			size_t		vStillToWrite = vData->GetLength();
			
			vData->SetOffset(0);	// Make sure we write all.
			
			// Now write in 512-byte blocks until we run out of data:
			while( vStillToWrite > 0 )
			{
				if( vStillToWrite > 512 )
					vReadSize = 512;
				else
					vReadSize = vStillToWrite;
				vStillToWrite -= vReadSize;
				vData->ReadData( vURLStringData, vReadSize );
				
				if( fwrite( vURLStringData, sizeof(char), vReadSize, vFile ) < vReadSize )
					throw runtime_error( "Couldn't write all data to file of local URL." );
			}
		}
		catch( exception& err )
		{
			fclose( vFile );
			throw;
		}
		
		fclose( vFile );
	}
	else if( strncmp( vURLString, "rsrc://", 7 ) == 0 )	// Starts with resource protocol?
	{
		short		refNum;
		Str255		vFName;
		Handle		resData = NULL;
		ResType		resType;
		short		resID;
		bool		didCreate = false;
		
		JokerRsrcURLToMacPath( vURLString, &resType, &resID, vFName );
		
		refNum = HOpenResFile( 0, 0L, vFName, fsRdWrPerm );
		if( ResError() == fnfErr )	// No such file?
		{
			HCreateResFile( 0, 0L, vFName );
			if( ResError() != noErr )
				throw runtime_error("Couldn't create file for \"rsrc:\" URL.");
			
			refNum = HOpenResFile( 0, 0L, vFName, fsRdWrPerm );
		}
		if( refNum < 0 )
		{
			char		errStr[512];
			
			strcpy( errStr, "Couldn't open url \"" );
			strncat( errStr, vURLString, 512 );
			strncat( errStr, "\".", 512 );
			
			throw runtime_error( errStr );
		}
		try
		{
			UseResFile( refNum );	// In case it was already open.
			resData = Get1Resource( resType, resID );
			if( resData == NULL || ResError() != noErr )
			{
				resData = NewHandle( vData->GetLength() );
				if( MemError() != noErr )
					throw bad_alloc();
				didCreate = true;
			}
			
			SetHandleSize( resData, vData->GetLength() );
			if( MemError() != noErr )
				throw bad_alloc();
			
			vData->SetOffset(0);	// Make sure we read the data from the start.
			
			HLock( resData );
			vData->ReadData( *resData, vData->GetLength() );
			HUnlock( resData );
			
			if( didCreate )
			{
				AddResource( resData, resType, resID, "\p" );
				if( ResError() != noErr )
					throw runtime_error( "Couldn't create new resource for non-existant \"rsrc:\" URL." );
			}
			ChangedResource( resData );
			WriteResource( resData );
		}
		catch( exception& err )
		{
			if( resData )
				DisposeHandle( resData );
			CloseResFile( refNum );
			throw;
		}
		CloseResFile( refNum );
	}
	else
	{
	  #if TARGET_RT_MAC_CFM
		if( !URLAccessAvailable() )
			throw runtime_error( "URL Access is not available on this Macintosh." );
		
		Str255				vFileName;
		FSSpec				vSpec;
		OSStatus			err = noErr;
		short				num, refNum;
		long				count;
		Handle				vDestDataHandle;
		
		vDestDataHandle = NewHandle(vData->GetLength());
		if( MemError() != noErr || vDestDataHandle == NULL )
			throw bad_alloc();
		try
		{
			vData->CopyToText( (*vDestDataHandle) );
		}
		catch( exception& err )
		{
			DisposeHandle( vDestDataHandle );
			throw;
		}
				
		// Create FSSpec to unique file name:
		while( err != fnfErr )
		{
			num = Random();
			NumToString( num, vFileName );
			err = FSMakeFSSpec( 0, 0, vFileName, &vSpec );
		}
		
		// Create & open temp file and write
		err = FSpCreate( &vSpec, '????', 'TEXT', smCurrentScript );
		if( err != noErr )
			throw runtime_error( "Couldn't create temp file for uploading." );
		
		try
		{
			try
			{
				err = FSpOpenDF( &vSpec, fsRdWrPerm, &refNum );
				if( err != noErr )
					throw runtime_error( "Couldn't open temp file for uploading." );
				
				count = GetHandleSize( vDestDataHandle );
				err = SetEOF( refNum, count );
				if( err != noErr )
					throw runtime_error( "Couldn't enlarge temp file for uploading." );
				
				err = SetFPos( refNum, fsFromStart, 0 );
				if( err != noErr )
					throw runtime_error( "Couldn't prepare writing to temp file for uploading." );
				
				err = FSWrite( refNum, &count, (*vDestDataHandle) );
				if( err != noErr )
					throw runtime_error( "Couldn't write to temp file for uploading." );
				
				FSClose( refNum );
				DisposeHandle( vDestDataHandle );
			}
			catch( exception& err )
			{
				DisposeHandle( vDestDataHandle );
				throw;
			}
				
			// Upload!
			err = URLSimpleUpload( mTheURL.String(), &vSpec,
									kURLDisplayProgressFlag | kURLDisplayAuthFlag,
									NULL, (void*) 0 );
			if( err != noErr )
				throw runtime_error( "Upload Failure." );
			
			FSpDelete( &vSpec );
		}
		catch( exception& err )
		{
			FSpDelete( &vSpec );
			throw;
		}
		
	  #else
		throw logic_error( "Uploads to remote URLs are not yet implemented on this platform." );
	  #endif
	}
}


/* --------------------------------------------------------------------------------
	GetPropertyValue:
		Determine the values of any properties of this URL. This implements the
		"files" property by listing the files in the specified directory using
		MoreFiles or calling upon URL access to list the files in a remote folder.
	
	TAKES:
		pName		-	Name of the property to get.
		outValue	-	A TalkValue* in which we store the property value.
	
	GIVES:
		-
	
	REVISIONS:
		2001-01-28	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::GetPropertyValue( const TextMunger& pName, TalkValue& outValue ) const
{
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	OSErr			err;
	ValueStorage	vOutData;
	
	if( pName == "files" )
	{
		mTheURL.CopyToString( vURLString );
		if( strncmp( vURLString, "file://", 7 ) == 0 )	// String starts with file protocol?
		{
			// Convert URL string to a Mac path.
			short		x = 0;
			for( ; vURLString[x] != 0; x++ )
			{
				if( vURLString[x] == '/' )
					vURLString[x] = ':';
				else if( vURLString[x] == ':' )
					vURLString[x] = '/';
			}
			
			if( x > 0 && vURLString[x-1] == ':' )	// Ends in colon?
				vURLString[x-1] = 0;	// Remove the colon.
			vURLString += 7;	// Make sure we skip protocol designator.
			
			try
			{
				Str255		vPasURLString;
				
				JokerCStringToP( vURLString, vPasURLString );
				
				vOutData.textType = new TextMunger();
				err = IterateDirectory( 0, 0, vPasURLString, 1,
										MyMacFolderIteratorProc,
										vOutData.textType );
				if( err != noErr )
					throw runtime_error( "Couldn't scan folder." );
				
				// Get rid of leading return:
				if( vOutData.textType->GetLength() > 1 )
				{
					vOutData.textType->SetOffset(0);
					vOutData.textType->DeleteData(1);
				}
				outValue.SetValue( vOutData, VALUE_TYPE_TEXT );
				
				delete vOutData.textType;
			}
			catch( exception& theError )
			{
				delete vOutData.textType;
				throw;
			}
		}
		else if( strncmp( vURLString, "rsrc://", 7 ) == 0 )	// String starts with resource protocol?
		{
			// Convert URL string to a Mac path.
			short		x = 0,
						refNum;
			
			for( ; vURLString[x] != 0; x++ )
			{
				if( vURLString[x] == '/' )
					vURLString[x] = ':';
				else if( vURLString[x] == ':' )
					vURLString[x] = '/';
			}
			
			if( x > 0 && vURLString[x-1] == ':' )	// Ends in colon?
				vURLString[x-1] = 0;	// Remove the colon.
			vURLString += 7;	// Make sure we skip protocol designator.
			
			// A leading path separator indicates relative pathnames on Mac, but absolute pathnames in URLs, so swap:
			if( vURLString[0] == ':' )
				vURLString++;
			else
				vURLString--;
			
			try
			{
				Str255		vPasURLString;
				FSSpec		theFile;
				short		x, y, maxx, maxy, theID;
				ResType		theType;
				Handle		theRes;
				Str255		theName;
				char		vNumStr[10];
				
				vOutData.textType = new TextMunger();
				
				JokerCStringToP( vURLString, vPasURLString );
				FSMakeFSSpec( 0, 0L, vPasURLString, &theFile );
				
				refNum = HOpenResFile( 0, 0L, vPasURLString, fsRdPerm );
				if( refNum < 0 )
					throw runtime_error("Error opening resource file referenced by \"rsrc:\" URL.");
				
				UseResFile( refNum );
				
				maxx = Count1Types();
				for( x = 1; x <= maxx; x++ )
				{
					Get1IndType( &theType, x );
					maxy = Count1Resources( theType );
					for( y = 1; y <= maxy; y++ )
					{
						SetResLoad(false);	// So we don't needlessly load all that resource data.
						theRes = Get1IndResource( theType, y );
						SetResLoad(true);
						
						if( x != 1 || y != 1 )	// Not first entry?
							vOutData.textType->Insert<char>('\n');	// Add newline between previous entry and this one.
						
						GetResInfo( theRes, &theID, &theType, theName );
						
						vOutData.textType->InsertData( &theType, 4 );
						sprintf( vNumStr, "/%d", theID );	// Slash to separate type and ID, so you can concat this with a slash and a URL to get a valid URL.
						vOutData.textType->InsertString( vNumStr );
					}
				}
				
				CloseResFile( refNum );
				
				outValue.SetValue( vOutData, VALUE_TYPE_TEXT );
				
				delete vOutData.textType;
			}
			catch( exception& theError )
			{
				delete vOutData.textType;
				CloseResFile( refNum );
				throw;
			}
		}
		else	// Remote URL?
		{
		  #if TARGET_RT_MAC_CFM
			if( !URLAccessAvailable() )
				throw runtime_error( "URL Access is not available on this Macintosh." );
			
			OSStatus			err;
			Handle				vDestDataHandle;
			
			vDestDataHandle = NewHandle(0);
			if( MemError() != noErr || vDestDataHandle == NULL )
				throw bad_alloc();
			
			err = URLSimpleDownload( mTheURL.String(), (FSSpec*) NULL,
										vDestDataHandle,
										kURLDisplayProgressFlag | kURLDisplayAuthFlag
										| kURLDirectoryListingFlag,
										NULL, (void*) 0 );
			if( err != noErr )
			{
				DisposeHandle( vDestDataHandle );
				throw runtime_error( "Directory listing failure." );
			}
			
			HLock( vDestDataHandle );
			try
			{
				vOutData.textType = NULL;
				vOutData.textType = new TextMunger( (*vDestDataHandle), GetHandleSize( vDestDataHandle ) );
				HUnlock( vDestDataHandle );
				DisposeHandle( vDestDataHandle );
				
				outValue.SetValue( vOutData, VALUE_TYPE_TEXT );
				delete vOutData.textType;
			}
			catch( exception& err )
			{
				HUnlock( vDestDataHandle );
				DisposeHandle( vDestDataHandle );
				if( vOutData.textType != NULL )
					delete vOutData.textType;
				
				throw;
			}
		  #else
			throw logic_error( "Listing contents of remote URLs is not yet implemented on this platform." );
		  #endif
		}
	}
	else
	{
		char	errstr[1024] = "URLs don't have a \"";	// FIX ME! Size limit on error message.
		
		strncat( errstr, pName.String(), 1024 );
		strncat( errstr, "\" property.", 1024 );
		throw runtime_error( errstr );
	}
}


/* --------------------------------------------------------------------------------
	MyMacFolderIteratorProc:
		Callback procedure called once for each file scanned when determining
		the value for the "files" property with local URLs.
	
	TAKES:
		cpbPtr		-	Information about the current file.
		quitFlag	-	Set this to abort the scan.
		yourDataPtr	-	A pointer to a TextMunger where the file list is to be
						stored.
		
	
	GIVES:
		-
	
	REVISIONS:
		2001-01-28	UK		Created.
   ----------------------------------------------------------------------------- */

pascal	void MyMacFolderIteratorProc( const CInfoPBRec *const cpbPtr,
										Boolean *quitFlag, void *yourDataPtr )
{
	try {
		TextMunger*		vOutData = (TextMunger*) yourDataPtr;
		Str255			vString;
		
		// Get file name:
		BlockMoveData( cpbPtr->hFileInfo.ioNamePtr, vString +1,	// +1 since we want to add a return before the name.
						cpbPtr->hFileInfo.ioNamePtr[0] +1 );	// +1 to account for length byte.
		
		// Move length byte to 1st byte and add a return:
		vString[0] = vString[1] +1;
		vString[1] = '\n';
		
		// Replace slashes through colons so URL looks right:
		short		x = 1;
		for( ; x <= vString[0]; x++ )
		{
			if( vString[x] == '/' )
				vString[x] = ':';
		}
		
		if( ( cpbPtr->hFileInfo.ioFlAttrib & ioDirMask ) != 0 )	// It's a folder!
			vString[ ++(vString[0]) ] = '/';	// Append a slash to indicate this.
		
		// Append file name to
		vOutData->SetOffset( vOutData->GetLength() );
		vOutData->InsertData( ((char*)vString) +1, vString[0] );
		
		// Give application time to handle events, other threads etc.:
		JokerSpendTime();
	}
	catch( exception& theErr )
	{
		*quitFlag = true;
	}
}