/*	

	PROJECT:	Joker
	
	FILE:		DrawingEnvironment.h
	
	PURPOSE:	Wrapper class for drawing routines.
	
	NOTE:		This draws a line including its first *and* last point. The
				platform-specific implementations of this might require
				you to nudge the coordinates a bit before drawing them, but
				will guarantee consistent results across platforms.
				
				Framed versions don't draw outside their object. I.e. if
				you frame and then fill, the framed version must be
				completely overdrawn, and you can rely on being able to
				erase a shape by simply filling with the background color.
				
				A framed rect draws the same as if you called DrawLine()
				four times, that is mathematically the bottom and right
				lie outside the area described by the rect.
	
	COPYRIGHT:	(C) Copyright 1998 by M. Uli Kusterer, all rights reserved.
	
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		99-03-07	UK		Created.
				
	 */

#pragma once

/* 
	Headers:
    */

#include	"JokerUtilities.h"

/* 
	Data structures:
    */

struct	DEPoint
{
	long		x;		// Horizontal coordinate.
	long		y;		// Vertical coordinate.
};

union	DERect
{
	// This is the official naming:
	struct {
		DEPoint		leftTop;
		DEPoint		rightBot;
	};
	// Mac user convenience:
	struct {
		DEPoint		topLeft;
		DEPoint		botRight;
	};
	// Second official naming:
	struct {
		long	left;
		long	top;
		long	right;
		long	bottom;
	};
};

/* Below, 65535 in all is white, 0 in all is black. */
struct	DEColor
{
	unsigned short		red;
	unsigned short		green;
	unsigned short		blue;
};

struct	DEPolygon
{
	long		count;		// # of points in polygon.
	DEPoint		point[1];	// Array of points in polygon. These are absolute coordinates, not offsets like on Mac!
};


/* 
	Class Implementation:
    */

class	DrawingEnvironment
{
protected:
	DEPoint		mPenPosition;
	DEPoint		mLineSize;
	DEColor		mForeColor;
	DEColor		mBackColor;

public:
	DrawingEnvironment()		{};
	virtual				~DrawingEnvironment();
	
// Global manipulation:
	virtual void		SetSize( long width, long height )					= 0;
	virtual DERect		GetSize()											= 0;
	virtual void		SetDepth( long theDepth )							= 0;
	virtual long		GetDepth()											= 0;
	
// Pen positioning:
	virtual void		MovePenTo( DEPoint pos )							{ mPenPosition = pos; };
	virtual DEPoint		GetPenPosition()									{ return mPenPosition; };

// Points:
	virtual void		DrawPoint( DEPoint thePt )							= 0;
	virtual DEColor		GetPointColor( DEPoint thePt )						= 0;

// Lines:
	virtual void		DrawLine( DEPoint fromPt, DEPoint endPt )			= 0;
	virtual void		SetLineSize( long width, long height )				{ mLineSize.x = width; mLineSize.y = height; };
	virtual DEPoint		GetLineSize() 										{ return mLineSize; };
	
// Colors:
	virtual	void		SetForeColor( DEColor theColor )					{ mForeColor = theColor; };
	virtual	DEColor		GetForeColor()										{ return mForeColor; };
	virtual	void		SetBackColor( DEColor theColor )					{ mBackColor = theColor; };
	virtual	DEColor		GetBackColor()										{ return mBackColor; };

// Shapes:
	virtual void		FrameRect( DERect theRect )							= 0;
	virtual void		FillRect( DERect theRect )							= 0;
	virtual void		FrameAndFillRect( DERect theRect )					{ FillRect( theRect ); FrameRect( theRect ); };
	
	virtual void		FrameEllipsis( DERect theRect )						= 0;
	virtual void		FillEllipsis( DERect theRect )						= 0;
	virtual void		FrameAndFillEllipsis( DERect theRect )				{ FillEllipsis( theRect ); FrameEllipsis( theRect ); };
	
	virtual void		FrameRoundrect( DERect theRect )					= 0;
	virtual void		FillRoundrect( DERect theRect )						= 0;
	virtual void		FrameAndFillRoundrect( DERect theRect )				{ FillRoundrect( theRect ); FrameRoundrect( theRect ); };

	virtual void		FramePolygon( DEPolygon thePoly, bool autoClose )	= 0;
	virtual void		FillPolygon( DEPolygon thePoly )					= 0;
	virtual void		FrameAndFillPolygon( DEPolygon thePoly, bool autoClose )		{ FillPolygon( thePoly ); FramePolygon( thePoly, autoClose ); };

// Save/Read (to/from PNG files):
	virtual void		DrawFile( FILE* theFile, DERect* theRect = NULL, bool scaleDontCrop = true )	= 0;
	virtual void		ExportToFile( FILE* theFile, DERect* theRect = NULL )	= 0;
};

/* 
	Prototypes:
    */

