/*	***************************************************************************

	PROJECT:	BasicValueStack
	
	FILE:		BasicValueStack.h
	
	PURPOSE:	The stack on which parameters and variables are kept.
				You have to subclass this for every platform if you need to
				pass along platform-specific data with system messages you
				send.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		2000-11-07	UK		Renamed from ValueStack.
		1999-12-01	UK		Created.
				
	************************************************************************ */

#ifndef	BASICVALUESTACK_H
#define BASICVALUESTACK_H	1

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkVarValue.h"
#include	<vector>
#include	<iostream>


#pragma mark [Class Declaration]

/* --------------------------------------------------------------------------------
	Class declaration:
   ----------------------------------------------------------------------------- */

class	HyperTalk;	// Forward.

class	BasicValueStack : public std::vector<TalkVarValue*>
{
protected:
	std::size_t				mBasePointer;		// This stack's basepointer. The stackpointer is this stack's size.
	long					mErrorLine;			// Current line. Used when error occurs to report where it was.
	HyperTalk*				mErrorScript;		// Current script. Used when error occurs to report where it was.
	TalkVarValue*			mExceptionValue;	// If an exception was thrown, this is the value it threw.
	TextMunger				mSpaces;			// Used for indenting debug output. This isn't a static variable in a function because ProjectBuilder doesn't allow those in framework targets like JokerLib.

public:
			BasicValueStack() : std::vector<TalkVarValue*>(), mBasePointer(0), mSpaces()	{};
	virtual ~BasicValueStack()												{};
	
	void			MoveBasePointer( std::size_t byVal )	{ mBasePointer += byVal; };
	std::size_t		GetBasePointer()						{ return mBasePointer; };
	void			SetBasePointer( std::size_t val )		{ mBasePointer = val; };
	
	void			SetExceptionValue( TalkValue& inVal )			{ mExceptionValue = new TalkVarValue(0L); mExceptionValue->Retain(); inVal.CopyValueTo( *mExceptionValue, true, false ); };
	TalkVarValue*	GetExceptionValue()								{ return mExceptionValue; };	// Caller must Retain() this to use it safely.
	void			ClearExceptionValue( bool doDispose = true )	{ if( doDispose && mExceptionValue ) mExceptionValue->Release(); mExceptionValue = NULL; };
	
	void	DumpOneValue( TalkVarValue* theValue )
	{
		TalkVarValue*			currVal;
		
		mSpaces.SetOffset(0);
		mSpaces.Insert<char>('\t');
		
		if( theValue->mList.size() != 0 && theValue->GetNativeType() != VALUE_TYPE_LIST )
			std::cout << "-array not cleared before changing value type-" << std::endl;
		
		if( theValue->GetNativeType() == VALUE_TYPE_LIST )	// Array.
		{
			TalkValueList::iterator		currElem;
			
			std::cout << mSpaces.String() << "array (" << (void*) theValue << ")" << std::endl;
			std::cout << mSpaces.String() << "{" << std::endl;
			for( currElem = theValue->mList.begin(); currElem != theValue->mList.end(); currElem++ )
			{
				std::cout << mSpaces.String() << "\t";
				std::cout << currElem->first.String() << ": ";	// output "key:"
				currVal = dynamic_cast<TalkVarValue*>(currElem->second);
				if( currVal )
					DumpOneValue( currVal );	// Output actual value.
				else
					std::cout << "-unknown value type-";
				std::cout << std::endl;
			}
			std::cout << mSpaces.String() << "}";
		}
		else
		{
			switch( theValue->GetNativeType() )
			{
				case VALUE_TYPE_LONG:
					std::cout << mSpaces.String() << "long (" << (void*) theValue << ") ";
					std::cout << theValue->mValue.longType;
					break;
				
				case VALUE_TYPE_DOUBLE:
					std::cout << mSpaces.String() << "double (" << (void*) theValue << ") ";
					std::cout << theValue->mValue.doubleType;
					break;
				
				case VALUE_TYPE_BOOL:
					std::cout << mSpaces.String() << "bool (" << (void*) theValue << ") ";
					if( theValue->mValue.boolType )
						std::cout << "true";
					else
						std::cout << "false";
					break;
				
				case VALUE_TYPE_TEXT:
					std::cout << mSpaces.String() << "text (" << (void*) theValue << ") \"";
					std::cout << theValue->mValue.textType->String();
					std::cout << "\"";
					break;
				
				case VALUE_TYPE_VALUE:
					std::cout << mSpaces.String() << "reference (" << (void*) theValue << ") " << (long) theValue->mValue.valueType;
					break;
				
				case VALUE_TYPE_ENTITY:
					std::cout << mSpaces.String() << "entity (" << (void*) theValue << ") " << (long) theValue->mValue.entityType;
					break;
				
				case VALUE_TYPE_POINT:
					std::cout << mSpaces.String() << "point (" << (void*) theValue << ") " << theValue->mValue.pointType.x << ", " << theValue->mValue.pointType.y;
					break;
				
				case VALUE_TYPE_COLOR:
					std::cout << mSpaces.String() << "color (" << (void*) theValue << ") " << theValue->mValue.colorType.red << ", " << theValue->mValue.colorType.green << ", " << theValue->mValue.colorType.blue;
					break;
				
				case VALUE_TYPE_RECT:
					std::cout << mSpaces.String() << "rect (" << (void*) theValue << ") " << theValue->mValue.rectType.left << ", " << theValue->mValue.rectType.top << ", " << theValue->mValue.rectType.right << ", " << theValue->mValue.rectType.bottom;
					break;
				
				default:
					std::cout << mSpaces.String() << "unknown (" << theValue->GetNativeType() << ")";
			}
		}
		
		mSpaces.SetOffset(1);
		mSpaces.DeleteData(1);
	};
	void	Dump()
	{
		BasicValueStack::iterator	itty;
		std::size_t					x = 0;
		
		std::cout << std::endl;
		std::cout << "Size: " << size() << " Base pointer: " << mBasePointer << std::endl;
		
		for( itty = begin(); itty != end(); itty++ )
		{
			++x;
			
			if( x == mBasePointer )
				std::cout << "> ";
			else
				std::cout << "  ";
			
			DumpOneValue( *itty );
			
			std::cout << std::endl;
		}
	};
	
	void	SetErrorSource( long errL, HyperTalk* errS )	{ mErrorLine = errL; mErrorScript = errS; };
	void	GetErrorSource( long& errL, HyperTalk* &errS )	{ errL = mErrorLine; errS = mErrorScript; };
};

typedef BasicValueStack::reverse_iterator	ValueStackRIterator;
typedef std::size_t ValStackLocation;


#pragma mark [Prototypes]

/* --------------------------------------------------------------------------------
	Prototypes:
   ----------------------------------------------------------------------------- */










#endif /*BASICVALUESTACK_H*/


