/* =============================================================================
	PROJECT:	Joker
	
	FILE:		TalkInstructions.h
	
	PURPOSE:	Instructions in a script.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	REVISIONS:
		1999-02-14	UK		Created.
   ========================================================================== */

#ifndef TALKINSTRUCTIONS_H
#define TALKINSTRUCTIONS_H

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkMemLocation.h"
#include	"TalkCallRecord.h"


#pragma mark [Data Types]

/* --------------------------------------------------------------------------------
	Data Types:
   ----------------------------------------------------------------------------- */

enum
{
	INSTRUCTION_PARAM_COUNT_ON_STACK	= 0xFF,		// Set mParamCount to this if the params are passed on the stack.
	TALK_INSTR_MAX_PARAMS				= 5			// Size of HyperInstruction's param array. I'm not sure this is used everywhere...
};

class	ValueStack;		// Forward.
class	HyperTalk;		// Forward.

// Pointer to a function to do the actual work of an instruction.
typedef void (*HyperInstrProcPtr)( class HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );

// One instruction of the lot that makes up a command:
class HyperInstruction
{
public:
	HyperInstrProcPtr	mInstructionProc;				// The instruction to perform.
	unsigned char		mParamCount;					// Number of parameters passed. Most instructions ignore this, but built-in functions or one-arg functions *must* check this!
	TalkMemLocation		mParam[TALK_INSTR_MAX_PARAMS];	// arguments to operate on.
	TalkMemLocation		mResult;						// Value where to store the result.
	HyperTalk*			mCaller;						// The owner of this instruction, the class containing the script.
	long				mRefCon;						// Custom data that is copied here from the command entry.
	
public:
	HyperInstruction();
	~HyperInstruction();
	
	void	SetRefCon( long n )	{ mRefCon = n; };
	long	GetRefCon()			{ return mRefCon; };
	
	void	ClearAllFields( bool doDispose );
	
	void	Dump();	// Debugging.
};



#pragma mark [Prototypes]

/* --------------------------------------------------------------------------------
	Prototypes:
   ----------------------------------------------------------------------------- */

void	HyperStoreInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperDoInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperBreakpointInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperPushParamInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperJSRInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperCharToNumInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperNumToCharInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperLongToNumInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperNumToLongInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperShortToNumInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperNumToShortInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperFindOffsetInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperBitAndInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperBitOrInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperBitNotInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperNumToHexInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperHexToNumInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperReplaceInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperGetParamInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperParamCountInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperIfInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperMakePackageInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperGetStdinInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperOutputInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperPutInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperGetGlobalPropInstruction( struct HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec );
void	HyperGetPropInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperSetPropInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperDefinePropInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperUndefinePropInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperReturnInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperExitInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperPassInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperAddInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperSubInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperMulInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperDivInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperLogicalAndInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperLogicalOrInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperAddCommand( struct HyperInstruction& hi, ValueStack& s,
							TalkCallRecord& vCallRec );
void	HyperSubCommand( struct HyperInstruction& hi, ValueStack& s,
							TalkCallRecord& vCallRec );
void	HyperMulCommand( struct HyperInstruction& hi, ValueStack& s,
							TalkCallRecord& vCallRec );
void	HyperDivCommand( struct HyperInstruction& hi, ValueStack& s,
							TalkCallRecord& vCallRec );
void	HyperCompareInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperCompareNotInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperLessThanInstruction( HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperGreaterThanInstruction( HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperLessThanEqualInstruction( HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec );
void	HyperGreaterThanEqualInstruction( HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec );
void	HyperConcatInstruction( HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperConcatSpaceInstruction( HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperUnaryMinusInstruction( HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperUnaryNotInstruction( HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperGetURLInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperGetEntryInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperGetKeysInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperCheckSumInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperDeclareGlobalInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperGetChunkOffsInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperGetChunkRefInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperCountChunkInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperDebuggerLineInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperDumpHdlStackInstruction( struct HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec );
void	HyperDeleteCommand( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperThrowCommand( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );


// Platform-specific instructions (implemented in GenTalkInstructions, MacTalkInstructions etc.):
void	HyperReplyInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperRequestInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperAppleEventInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperBeepInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );
void	HyperAskInstruction( struct HyperInstruction& hi, ValueStack& s,
								TalkCallRecord& vCallRec );

// Host-specific instructions (implement object architecture specific to hosts):
void	HyperGetStackEntityInstruction( struct HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec );
void	HyperGetMenuEntityInstruction( struct HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec );
void	HyperGetPartEntityInstruction( struct HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec );
void	HyperGetCurrStackEntityInstruction( struct HyperInstruction& hi, ValueStack& s,
											TalkCallRecord& vCallRec );

// Utility fcn:
bool	CompareBeef( HyperInstruction& hi, ValueStack& s );




#endif /*TALKINSTRUCTIONS_H*/