/*	***************************************************************************

	PROJECT:	TalkMemLocation
	
	FILE:		TalkMemLocation.h
	
	PURPOSE:	A memory address with relative addressing that refers to an
				entry on the stack, or contains an immediate value.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-02-14	UK		Created.
				
	************************************************************************ */

#ifndef TALKMEMLOCATION_H
#define TALKMEMLOCATION_H

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */
   
#include	"TalkValue.h"
#include	"ValueStack.h"
#include	<stdexcept>

// #if __MWERKS__
using namespace std;
// #endif


#pragma mark [Constants]

/* --------------------------------------------------------------------------------
	Constants:
   ----------------------------------------------------------------------------- */

enum
{
// Index of Accu in array of variables:
	MEM_INDEX_ACCU				= 0
};

// MemLocation types:
typedef enum MemLocationEnum
{
	MEM_LOCATION_TYPE_INVALID = 0,		// Invalid -- Used to identify cases where we don't want return values from functions.
	MEM_LOCATION_TYPE_IMMEDIATE,		// Constant.
	MEM_LOCATION_TYPE_STACK,			// Temporary or variable on the stack.
	MEM_LOCATION_TYPE_LONG				// A long -- used for pushing param count onto stack.
} MemLocationEnum;


#pragma mark [Class Declaration]

/* --------------------------------------------------------------------------------
	Class declaration:
   ----------------------------------------------------------------------------- */

// An immediate or stack variable:
class	TalkMemLocation
{
protected:
	union
	{
		TalkValue*			mValue;		// mType == MEM_LOCATION_TYPE_IMMEDIATE: Pointer to actual value.
		ValStackLocation	mLocation;	// mType == MEM_LOCATION_TYPE_STACK: Offset into stack.
		long				mLong;		// mType == MEM_LOCATION_TYPE_LONG: The number stored here.
	};
	
public:
	MemLocationEnum		mType;
	TalkMemLocation()								{ mType = MEM_LOCATION_TYPE_INVALID; mLong = 0; };
	TalkMemLocation( const TalkMemLocation& ml )
	{
		mType = MEM_LOCATION_TYPE_INVALID;
		AssignMemLocation(ml);
	};
	~TalkMemLocation()	{ if( mType == MEM_LOCATION_TYPE_IMMEDIATE ) mValue->Release(); };
	
// Used during parsing:
	TalkMemLocation& operator=( const TalkMemLocation& ml )
	{
		AssignMemLocation(ml);
		
		return *this;
	}
	void				AssignMemLocation( const TalkMemLocation& ml )
	{
		if( mType == MEM_LOCATION_TYPE_IMMEDIATE )
			mValue->Release();
		
		mType = ml.mType;
		switch( ml.mType )
		{
			case MEM_LOCATION_TYPE_IMMEDIATE:
				mValue = ml.mValue;
				mValue->Retain();
				break;
			
			case MEM_LOCATION_TYPE_STACK:
				mLocation = ml.mLocation;
				break;
			
			case MEM_LOCATION_TYPE_LONG:
			case MEM_LOCATION_TYPE_INVALID:
				mLong = ml.mLong;
				break;
		}
	}
	void				AssignImmediate( TalkValue* v )
	{
		if( mType == MEM_LOCATION_TYPE_IMMEDIATE )
			mValue->Release();
		v->Retain();
		mType = MEM_LOCATION_TYPE_IMMEDIATE;
		mValue = v;
	};
	void				AssignStackLocation( ValStackLocation l )
	{
		if( mType == MEM_LOCATION_TYPE_IMMEDIATE )
			mValue->Release();
		mType = MEM_LOCATION_TYPE_STACK;
		mLocation = l;
	};
	void				AssignLongImmediate( long l )
	{
		if( mType == MEM_LOCATION_TYPE_IMMEDIATE )
			mValue->Release();
		mType = MEM_LOCATION_TYPE_LONG;
		mLong = l;
	};
	void				AssignInvalid()
	{
		if( mType == MEM_LOCATION_TYPE_IMMEDIATE )
			mValue->Release();
		mType = MEM_LOCATION_TYPE_INVALID;
		mLong = 0;
	};
	TalkValue*			GetImmediate()
	{
		if( mType != MEM_LOCATION_TYPE_IMMEDIATE )
			throw runtime_error("Tried to retrieve MemLocation as Immediate when it isn't.");
		if( ((long)mValue) == 8 )
			throw runtime_error("Bug here!");
		return mValue;
	};
	ValStackLocation	GetStackLocation()	{ if( mType != MEM_LOCATION_TYPE_STACK ) throw runtime_error("Tried to retrieve MemLocation as StackLocation when it isn't."); return mLocation; };
	long				GetLongImmediate()	{ if( mType != MEM_LOCATION_TYPE_LONG ) throw runtime_error("Tried to retrieve MemLocation as LongImmediate when it isn't."); return mLong; };
	bool				IsInvalid()			{ return( mType == MEM_LOCATION_TYPE_INVALID ); };
	bool				IsStackLocation()	{ return( mType == MEM_LOCATION_TYPE_STACK ); };
	bool				IsImmediate()		{ return( mType == MEM_LOCATION_TYPE_IMMEDIATE ); };
	bool				IsLongImmediate()	{ return( mType == MEM_LOCATION_TYPE_LONG ); };
	
	MemLocationEnum		GetType()			{ return mType; };
	
// Used during execution:
	void			GetValue( ValueStorage& outValue, ValueStack& s, unsigned long desiredType );
	void			SetValue( ValueStorage& inValue, ValueStack& s, unsigned long providedType, bool followRefs = false );
	void			GetValue( TalkValue* &outValue, ValueStack& s );
	void			GetListEntry( TalkValue* &returnValue, ValueStack& s, TextMunger& index );
	void			CopyValueTo( TalkValue& vValue, ValueStack &s, bool followSrcRefs = false, bool followDstRefs = false );
	unsigned long	GetAvailableTypes( ValueStack &s );
};

#endif /* TALKMEMLOCATION_H */