/*	

	PROJECT:	TextMunger
	
	FILE:		TextMunger.h
	
	PURPOSE:	Text manipulation class for Joker.
	
	NOTE:		This uses malloc() and not new so we can use realloc() to
				resize its storage when needed.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-02-14	UK		Created.
				
	 */

#ifndef TEXTMUNGER_H
#define TEXTMUNGER_H

/* 
	DIRECTIONS:
	
	TextMunger is an advanced streaming class. It allows reading, writing, and
	inserting any kind of data. Note that the term 'text' is a bit misleading here.
	You actually only write bytes to this stream, you are not limited to their
	textual representation.
	
	To write to this file, either use textMungerInstance.Smash<char>( char inChar );
	which is a template, i.e. you can use any data type, not just char, or use
	textMungerInstance.SmashData( void* inData, size_t inLength );. Note that this
	doesn't enlarge the storage, it just overwrites any existing data.
	
	If you want to advance the read/write mark (aka offset) past the data written,
	you can use textMungerInstance.WriteData( void* inData, size_t inLength ); and
	textMungerInstance.Write<char>( char inChar );.
	
	You can use textMungerInstance.Insert<char>( char inChar ); or
	textMungerInstance.InsertData( char inChar ) to insert data at the given offset.
	If the current offset isn't at the end of the storage, the data beyond the
	current offset is automatically moved to make room for the new data. If the
	offset lies at the end of the storage, it will merely be enlarged.
	
	To read data, either use textMungerInstance.Peek<char>(); which returns the data
	read, or textMungerInstance.PeekData( void* ioData, size_t inSize ); which sets
	the pointer passed to the data read.
	
	To move the read/write mark (aka offset) after peeking, use textMungerInstance.Read<char>();
	or textMungerInstance.ReadData( void* ioData, size_t inSize );.
	
	There's also textMungerInstance.Delete<char>(); and
	textMungerInstance.DeleteData( size_t inSize ); which do the inverse of Insert...
	
	To manipulate the position of the read/write mark, use
	textMungerInstance.SetOffset( size_t inMark ); or
	textMungerInstance.AdvanceOffset( size_t inMark ); that position the offset either
	absolutely or relatively to the current offset. You can retrieve the current offset
	using textMungerInstance.GetOffset();.
	
	There are also more routines which copy the whole instance's data to a string, pointer
	(text) or another text munger as well as retrieving it from there, and you can get the
	length of the entire buffer's contents.
	
	Finally, there are some comparison functions that do case-insensitive text comparison.
*/

#pragma once
#pragma mark [Headers]

/* 
	Headers:
    */

//#include	"JokerUtilities.h"
#include	"XSmartPointer.h"
#include	<iostream>
#include	<memory>
#include	<stdlib.h>


#pragma mark [Class Declaration]

/* 
	Class declaration:
    */

class	RefCountedText
{
public:
	XSmartPointer	mText;		// Pointer to the text data we keep, created using malloc.
	std::size_t		mLength;	// Length of data allocated.
	
protected:
	std::size_t		mRefCount;
	
public:
	RefCountedText()	{ mText = NULL; mRefCount = 0; mLength = 0; };
	~RefCountedText()	{ if( mText != 0L ) free( mText ); mText = NULL; };
	
	void			RequestReadAccess()		{ mRefCount++; };
	RefCountedText*	RequestWriteAccess();
	void			GiveUpAccess()			{ if( --mRefCount == 0 ) delete this; };
};

// Actual text munger class:

class	TextMunger
{
protected:
	RefCountedText*			mText;			// Ref-counted unterminated string used for actual content.
	std::size_t				mLength;		// Length of part of string above we use.
	std::size_t				mOffset;		// Position of read/write mark.
	std::size_t				mFixedOffset;	// Some fixed offset into data for substrings.
	mutable unsigned long	mHash;			// This object's hash for compares. Even in const objects we should be allowed to keep this.
	bool					mDontOwnBuffer;	// TRUE if mText->mText is owned by the caller and not by this object.
	
public:
	TextMunger();												// Make empty string, default.
	TextMunger( const TextMunger& inMunger, std::size_t subStrOffs = 0, std::size_t subStrLen = 0 );	// Copy data from another munger.
	TextMunger( const char* inString );							// Copy data from a CString.
	TextMunger( const char* inText, const std::size_t inLength );	// Copy data of specified length from a pointer.
    TextMunger( char* inText, const std::size_t inLength, bool dontCopy = false );	// Take over a pointer.
	virtual ~TextMunger();
	
	std::size_t	GetLength() const	{ return mLength; };
	void		SetLength( std::size_t len );
	bool		EqualsString( const char* inStr, const std::size_t inLength ) const;
	bool		EqualsMunger( const TextMunger& inMun, const std::size_t inLength ) const;
	
// This --> other:
	void		CopyToString( char* outStr ) const;	// Zero-terminated.
	void		CopyToText( char* outText ) const;	// Not terminated.
	void		CopyToTextMunger( TextMunger& ioMunger ) const;
	
// Other --> this:
	void		CopyFromString( const char* inStr );	// Zero-terminated.
	void		CopyFromText( const char* inText, const std::size_t inLength );	// Not terminated.
	void		CopyFromTextMunger( const TextMunger& inMunger );
    void		TakeOverText( char* inText, const std::size_t inLength );	// Takes over the pointer, releasing our current one.
	
// Streaming access:
// Peeking:
	template<class T>
	T					Peek()
	{
		T		vTheType;
	
		//ASSERT_Range( mOffset +sizeof(T) <= mLength );
		memcpy( &vTheType, (char*) mText->mText +mOffset, sizeof(T) );
	
		return vTheType;
	};
	void		PeekData( void* ioData, std::size_t inLength );
	
// Offset (=read/write mark position):
	std::size_t	GetOffset()									{ return mOffset; };
	void		SetOffset( std::size_t inMark )				{ /*ASSERT_Range( inMark <= mLength);*/ mOffset = inMark; };
	void		AdvanceOffset( std::size_t inMark )			{ /*ASSERT_Range( mOffset+inMark <= mLength);*/ mOffset += inMark; };
	
// Reading: (= peek +offset advance)
	template<class T>
	T					Read()
	{
		T		vC;
		ReadData( &vC, sizeof(T) );
		return vC;
	};
	void		ReadData( void* ioData, std::size_t inLength )	{ PeekData( ioData, inLength ); AdvanceOffset( inLength ); };

// Writing (= smash +offset advance):
	template<class T>
	void				Write( const T inType )
	{
		WriteData( &inType, sizeof(T) );
	};
	void	WriteString( const char* inString )							{ WriteData( inString, strlen(inString) ); };
	void	WriteData( const void* inData, const std::size_t inLength )	{ SmashData( inData, inLength ); AdvanceOffset( inLength ); };
	void	WriteMunger( const TextMunger& inMunger )					{ SmashMunger( inMunger ); AdvanceOffset( inMunger.mLength ); };
	
// Storing:
	template<class T>
	void				Smash( const T inType )
	{
		SmashData( &inType, sizeof(T) );
	};
	void		SmashString( const char* inString )		{ SmashData( inString, strlen(inString) ); };
	void		SmashData( const void* inData, const std::size_t inLength );
	void		SmashMunger( const TextMunger& inMunger );

// Inserting (includes offset advance):
	template<class T>
	void				Insert( const T inType )
	{
		InsertData( &inType, sizeof(T) );
	};
	void		InsertString( const char* inString )		{ InsertData( inString, strlen(inString) ); };
	void		InsertData( const void* inData, const std::size_t inLength );
	void		InsertMunger( const TextMunger& inMunger );
	
// Deleting:
	template<class T>
	void				Delete()
	{
		DeleteData( sizeof(T) );
	};
	void			DeleteData( const std::size_t inLength );

// Convenience/for use by map:
	virtual				operator char*();
	virtual	TextMunger&	operator= ( const TextMunger& inMunger );
	virtual	bool		operator== ( const TextMunger& inMun ) const;
	virtual	bool		operator== ( const char* inStr ) const;
	virtual	bool		operator< ( const TextMunger& inMun ) const;
	virtual	bool		operator> ( const TextMunger& inMun ) const;
	char*				String() const;
	char*				GetTextPtr() const;
	static void			FreeTempMemory();
	void				RecalcHash() const;
};

typedef std::auto_ptr<TextMunger>	TexMunAPtr;


#pragma mark [Errors]

/* 
	Errors:
    */






#endif /*TEXTMUNGER_H*/





