/* ================================================================================
	PROJECT:	JokerLib
	FILE:		JokerLib.h
	PURPOSE:	Header file for JokerLib, the shared library that implements an
				xTalk interpreter.
	COPYRIGHT:	JokerLib and Joker are (c) Copyright 1999-2002 by M. Uli Kusterer,
				all rights reserved.
   ============================================================================= */

/*!	@header		JokerLib
	JokerLib is a library that lets any program linking to it execute scripts
	written in an xTalk-language. HyperCard's scripting language HyperTalk, for
	example. JokerLib provides the means to define custom commands, react to
	certain messages with default behaviour when they aren't intercepted, and
	sending messages to scripts.<br>
	For a description of the actual syntax of Joker's xTalk language, see the
	file &quot;Syntax.doc&quot;.
	@copyright	(c) Copyright 2001-2002 by M. Uli Kusterer, all rights reserved. */
/*! @link xTalk&nbsp;Syntax Syntax.doc*/
#ifndef ULI_JOKER_LIB_H
#define ULI_JOKER_LIB_H

#if PRAGMA_ONCE
#pragma once
#endif

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#if defined(__GNUC__) && (defined(__APPLE_CPP__) || defined(__APPLE_CC__))
#include	<Carbon/Carbon.h>
#else
#include	<MacTypes.h>
#endif
#include	"stdarg.h"


// ================================================================================
/*!	@section Introduction&nbsp;to&nbsp;Joker
	Joker is a nice little package of utility code that is intended to let a 
	client application execute programs written in one of the highest-level
	languages I have ever encountered, HyperTalk. HyperTalk is the
	&quot;scripting&quot; language used by Apple's HyperCard application. It also
	has been cloned by numerous other applications, including IncWell's SuperCard,
	MacroMedia's Director, MetaCard corp's MetaCard and Oracle's Media Objects.
	<p>The HyperTalk language provides the following benefits:
	<ul>
	<li>Very High Level Language that relieves you of tedious and error-prone
	operations such as memory management</li>
	<li>Virtual Compiler which combines the benefits of interpreted and
	compiled languages. This allows for very flexible runtime behaviour as
	well as decent speed</li>
	<li>Very readable and thus easily maintainable english-like syntax</li>
	<li>Automatic conversion between data types transparently and intuitively</li>
	<li>Support for overriding of methods and a messaging hierarchy of your
	choice.</li>
	<li>Most common text manipulation facilities already built-in.</li>
	<li>Object oriented</li>
	</ul>
	HyperTalk is not a Macro language like AppleScript. Though it is at least as
	easy to use as many Macro languages, it packs more power and doesn't suffer
	the primitive syntax many users have had to contend with up to now. Its lack
	of static data typing makes HyperTalk an ideal language for the casual
	programmer, while its text manipulation capabilities and powerful instructions
	still make it a valuable asset for professional programmers.
	<p>Joker has been designed to easily allow your application to make its
	objects (windows, documents, etc.) controllable with Joker and to add
	client-defined commands that work on these or other objects. This includes
	adding English-like syntax and thus prevents the user from even noticing that
	she is dealing with a library.
	<p>Furthermore Joker is a partially interpreted language, which does away with
	the long wait for the program to be compiled. You can simply code and execute. */

/*! @subsection	Scripts
	Just like you have the source code file as the basic storage element in
	languages like C, a Script is the basic storage element for code in Joker.<br>
	However, to allow for maximum flexibility, a Script is not tied to any file
	or resource type. A Script in Joker is simply an opaque reference to a block
	of data, which corresponds to the plain text form of your source code.
	<p>Whenever you want to execute HyperTalk code using Joker, you need a script.
	You create a new script object and give it the plain text form of your code.
	It will then perform translation of this script and determine the names
	of the <a href="#Handlers">handlers</a> contained in this script.
	<p>As soon as you have done this, you can begin using the code by simply
	invoking one of the handlers in the script. */
	
/*! @subsection	Handlers
	A Handler in Joker is conceptually the same as a subroutine in other
	programming languages. A Handler takes parameters, which let you
	pass data into a handler. In addition, each handler may have a return
	value.
	<p>There are two kinds of handlers:
	<ul><li>Command message handlers</li>
	<li>Function message handlers</li></ul>
	A command message handler is used like a command (&quot;procedure&quot; in Pascal
	parlance), and the return value is usually only employed to return error
	information, while a function message handler is usually used like any other
	built-in function or value, inside expressions if desired, where its return
	value will be inserted after it has been called.
	<p>A message handler is identified by its name and can be called both by
	the client application as well as from another handler. The process of calling
	a message handler is usually referred to as "sending a message". Such a message
	can then be intercepted by the message handler, the handler can decide to
	abort script execution entirely, or it can simply pass (i.e. hand on) the
	message up the hierarchy. How this hierarchy looks and whether it exists at all
	is left to the client. It could be a virtual function hierarchy where the
	message is passed to the superclass, or it could be a messaging hierarchy
	where the message is passed to another instance of an object. */
	
/*! @subsection	Values
	HyperTalk insulates the user from the fact that computers internally use
	different data types. While this means that you can not override functions
	based on their data types, it allows for e.g. simply requesting the fifth
	character of a number. Joker will automatically perform the conversion to
	a string and extract the appropriate character. It also makes file i/o
	a lot easier, because you can simply read the string "true" and use it
	as a boolean right away.
	<p>This flexibility however, has to end when the client application,
	written in a 3rd generation language like C or Pascal, needs to pass values
	into Joker, or retrieve handlers' return values from Joker. To account for
	this, Joker has the concept of a Value.
	<p>A Value is an object whose sole purpose is to contain data of various
	types and to convert between them. It generally works like a union in C.
	You just create an instance of a Value, and store a long, double, string,
	point, rectangle, RGB color or array in it. You can also extract values as
	any of these data types from a Value, as long as they are compatible.
	<p>Generally, all values can be converted to strings and back, and longs
	can be converted to doubles, and in some cases doubles to longs. */
	
/*! @subsection	Entities
	To be able to &quot;see&quot; an object in the client application, Joker
	uses Entities. An entity is an object on which you can install callback
	functions. Whenever a script tries to manipulate such an entity, which
	you can associate with your application's internal representation of
	its objects, Joker will call your callback and tell you about this.
	Then, the client is responsible for actually reacting on the messages.
	<p>All entities have a couple of common characteristics:
	<ul><li>Properties - A property is simply a bit of data identified by a
	name, like a data member in C++. Your scripts can use the &quot;set&quot;
	command to change properties and can read the values of properties</li>
	<li>Deletion - The user can use the &quot;delete&quot; command to attempt
	to delete an object represented by an entity. This doesn't make sense
	for all objects.</li>
	<li>Contents - Whenever an entity is used as a Value in a script,
	the client can decide what data should be used. This could be the text
	of an editable text field, or an XML representation of an object, or
	whatever else you choose.</li>
	<li>User properties - A script can define its own properties for an
	entity, which the client may choose to save along with the object, or
	it may just discard them and only allow their use as temporary
	runtime storage.</li></ul> */
	 
/*! @subsection	Object&nbsp;Descriptors
	Since Joker does not know anything about the client application's objects,
	there needs to be a way for the application to specify which entity goes
	with what object. To achieve this, special syntax (an
	&quot;object descriptor&quot;) can be registered with Joker that allows
	specifying an object from a script. The client simply registers a callback and
	whenever a particular object descriptor is encountered in a script during
	execution, the callback will be called upon to return the appropriate Entity
	for whatever the user entered. */

/*! @subsection	Tokenizing,&nbsp;parsing&nbsp;and&nbsp;execution
	Whenever a script is executed, Joker needs to perform two steps to translate
	it from its original text form into a form that can be executed by the computer.
	<ol>
	<li>Tokenization - During this step, Joker performs a first analysis of the
	stream of characters that is the text of the script. It divides it into tokens
	(&quot;words&quot;), which are simply small numbers telling apart all the
	identifiers in your script. Since a number takes about 2 bytes, while the
	text representation of a token takes 1 byte per character (e.g. 6 bytes for
	the "return" keyword), this speeds up the process of parsing the script, as
	Joker doesn't have to re-read each and every character when it wants to look
	ahead, and has to work with much less data.<br>
	Before executing or parsing a script, you must _always_ first tokenize it.</li>
	<li>Parsing - This is the actual process of translating the script into
	something useful. Joker will go over all tokens in the script and create
	"instructions" for each line. This is also the step where Joker splits
	the script into the separate message handlers. Once a script has been parsed
	successfully, you can execute (or &quot;run&quot;) any of its handlers.</li>
	</ol> */

// ================================================================================
/*!	@section Using&nbsp;Joker */
// ================================================================================

/*! @subsection	Configuring&nbsp;and&nbsp;initializing&nbsp;Joker
	Before you can use Joker, there is some setup work you have to do. First of
	all, before making your first call to any other Joker function, you must
	call the JokerInitialize() function, to allow Joker to set up everything.
	After that, you should call JokerShowSplash() to let your users know you
	are using Joker.
	<p>Apart from these two, you might want to call JokerSetNewline() to make
	sure Joker's output is formatted according to the needs of the slient
	application and and your console. You can also modify the behaviour of
	Joker using the JokerSetOption() call. Finally, you might want to reroute
	the "output" and '"put" without destination' commands to somewhere else
	than the console by specifying a JokerOuputProc. There are additional
	callback procedures that you may want to register at this time, as well.
	Finally, this is a good time to register any commands and object
	descriptors. */
/*! @subsection	Running&nbsp;a&nbsp;script&nbsp;from&nbsp;a&nbsp;text&nbsp;file
	!!! TO DO !!!
	@seealso JokerScriptRef */
/*! @subsection	Entities
	!!! TO DO !!!
	@seealso JokerEntityEventEnum */


// ================================================================================
/*!	@section Data&nbsp;Types */
// ================================================================================

/*! @typedef	JokerScriptRef
	Reference to a Joker script.
	The typical sequence of creating a script is:<br>
	<OL><LI>JokerNewScript()</LI>
	<LI>JokerTokenizeScript()</LI>
	<LI>JokerParseScript() or JokerParseScriptAsHandler</LI>
	<LI>JokerRunHandler()</LI>
	<LI>JokerReleaseScript</LI></OL>
	Where tokenizing and parsing must be repeated whenever the text of a
	script changes.<br>
	You can keep around a parsed and tokenized script and run several
	handlers in it until you release it.
	@seealso JokerNewScript JokerDisposeScript */
typedef		long*	JokerScriptRef;

/*!	@typedef 	JokerValueRef
	Value to pass to Joker.
	This is like an Apple Event Descriptor. Create one, assign it a
	value, and pass it to JokerRunHandler() as a parameter. Return
	values are also returned this way. A JokerValueRef may contain
	any of the following data types:
	<UL><LI>Long Integer</LI>
	<LI>ASCII Character String</LI>
	<LI>Quickdraw Point</LI>
	<LI>Quickdraw RGB Color</LI>
	<LI>Quickdraw Rectangle</LI>
	<LI>Double Precision Floating Point Number</LI>
	<LI>An Associative Array of other JokerValueRefs</LI></UL>
	Except for the last one, all of these can be converted into
	character strings, and under certain circumstances integers
	can be elevated to doubles or doubles reduced to integers.
	@seealso JokerNewValue JokerDisposeValue JokerVTypes */
typedef		long*	JokerValueRef;

/*!	@typedef	JokerEntityRef
	An entity is any kind of user-defined object. You can define
	your own kinds of entities and register them with JokerLib.<BR>
	Every entity may have properties, contents and the user may
	define and set user properties for it. Associated with every
	entity is a JokerEntityProc which is called to do the actual
	work whenever a (non-user) property or the contents are accessed.
	@seealso JokerNewEntity JokerDisposeEntity JokerRegisterObjectDescriptor
				JokerObjectDescriptorProcPtr JokerEntityProcPtr */
typedef		long*	JokerEntityRef;

/* MEMENTO for Uli:
	Keep these constants in sync with the ones in HyperTalk.h!!! */

/*! @typedef JokerCallResult
	Call results are returned by any handler call that you make in a script.
	@seealso JokerCallResultEnum JokerRunHandler JokerHandlerCallProcPtr */

typedef short		JokerCallResult;

/*!	@enum JokerCallResultEnum
	These are the constant values a JokerCallResult may have.
	@constant kJokerCallResultIntercepted	A handler caught this message, either
											by not passing it on or by calling
											exit <handlername>.
	@constant kJokerCallResultPassed		The message wasn't caught or the
											handler that caught it passed it on.
	@constant kJokerCallResultExited		A handler caught this message and
											called &quot;exit to top&quot; to
											exit from this message.
	@seealso JokerCallResult JokerRunHandler JokerHandlerCallProcPtr */
enum JokerCallResultEnum
{
	kJokerCallResultIntercepted	=	0,
	kJokerCallResultPassed,
	kJokerCallResultExited
};

/*! @typedef JokerOptionID
	Option IDs are what you use to activate/deactivare certain behaviour in the
	Interpreter.
	@seealso JokerOptionIDEnum JokerSetOption */
typedef unsigned short	JokerOptionID;
/*!	@enum	JokerOptionIDEnum
	These are the constant values a JokerOptionID may have.
	@constant  kJokerOptionAllowUnquotedLiterals	This entry is true or false to
													turn on/off support for unquoted
													string literals. Default is TRUE
													(HC).
	@constant  kJokerOptionDistinguishFcnCmd		This entry is true or false to
													turn on/off distinction of
													message and function handlers.
													(HC uses TRUE here)
	@constant  kJokerOptionUnqoutedLitsAsVars		True if you want unquoted literals
													to be created as a variable with
													its name in it, false if you really
													want them to be constants.
													(HC uses TRUE here)
	@constant  kJokerOptionEscapeCharsInStrings		True if you want to support C-style
													escaped characters in strings to
													insert returns and quotes in strings.
													If this is on, returns in multi-line
													string tokens will be turned into
													spaces.
	@seealso JokerOptionID JokerSetOption */
enum JokerOptionIDEnum
{
	kJokerOptionAllowUnquotedLiterals = 0,
	kJokerOptionDistinguishFcnCmd,
	kJokerOptionUnqoutedLitsAsVars,
	kJokerOptionEscapeCharsInStrings
};


/*! @typedef JokerVTypes
	This is a bit field with bit flags for each data type JokerValueRefs support.
	You can use this e.g. to determine the natural type of a JokerValueRef's contents
	or to determine to what types they can be converted.
	@seealso JokerVTypesEnum JokerGetValueAvailableTypes JokerValueRef */
typedef unsigned long	JokerVTypes;

/*! @enum JokerVTypesEnum
	These are the constants for the bits in JokerVTypes.
	@constant 	kJokerValueTypeLong			Integer.
	@constant 	kJokerValueTypeDouble		Floating-point number.
	@constant 	kJokerValueTypeBoolean		Boolean value.
	@constant 	kJokerValueTypeText			Text string.
	@constant 	kJokerValueTypeReference	For internal use only.
	@constant 	kJokerValueTypePoint		A coordinate on the screen.
	@constant 	kJokerValueTypeRectangle	A rectangle on the screen.
	@constant 	kJokerValueTypeColor		An RGB color value.
	@constant 	kJokerValueTypeEntity		A JokerEntityRef.
	@constant 	kJokerValueTypeArray		An array.
	@seealso JokerVTypes JokerGetValueAvailableTypes JokerValueRef */
enum JokerVTypesEnum
{
	kJokerValueTypeLong			= (1 << 0),
	kJokerValueTypeDouble		= (1 << 1),
	kJokerValueTypeBoolean		= (1 << 2),
	kJokerValueTypeText			= (1 << 3),
	kJokerValueTypeReference	= (1 << 4),
	kJokerValueTypePoint		= (1 << 5),
	kJokerValueTypeRectangle	= (1 << 6),
	kJokerValueTypeColor		= (1 << 7),
	kJokerValueTypeEntity		= (1 << 8),
	kJokerValueTypeArray		= (1 << 9)
};


/*! @typedef JokerEntityEvent
	This is an action selector passed to your JokerEntityProcs to
	tell them what action is expected from them.
	@seealso JokerEntityEventEnum JokerNewEntity JokerEntityProc */
typedef unsigned short JokerEntityEvent;

/*! @enum JokerEntityEventEnum
	These are the constants for JokerEntityEvents. For most of these, you are supposed to return
	1 on success and NULL if you didn't perform the requested action.
	@constant 	kJokerEntityGetPropEvent			Provide the value of a property. Name is in param.
													Set the ValueRef param2 to the value of the property.
	@constant 	kJokerEntitySetPropEvent			Change a property. Name is param, new value is in param2.
	@constant 	kJokerEntityGetContentsEvent		Return a JokerValueRef to your object's contents. You own
													this JokerValueRef.
	@constant 	kJokerEntityDeleteEvent				A script requested to delete this object. React accordingly.
	@seealso JokerEntityEvent JokerNewEntity JokerEntityProc */
enum JokerEntityEventEnum
{
	kJokerEntityGetPropEvent = 0,
	kJokerEntitySetPropEvent,
	kJokerEntityGetContentsEvent,
	kJokerEntityDeleteEvent
};


// --------------------------------------------------------------------------------
/*!	@typedef 			JokerOutputProcPtr
	@abstract			Whenever a script outputs something to the console, this
						function is called.
	@param data			A pointer to the text to write.
	@param amount		The number of bytes in the pointer to write.
	@param result		This function returns the amount of data actually written,
						or -1 when an error occurred.
	@seealso JokerSetOutputProc */
typedef		pascal long	(*JokerOutputProcPtr)( char* data, long amount );

// --------------------------------------------------------------------------------
/*!	@typedef 			JokerHandlerCallProcPtr
	@abstract			Whenever a handler can't be found in a script that you
						call using JokerRunHandler(), this callback is executed.
	@param handlerName	The name of the handler that was called.
	@param result		This is where you can store the result value you wish to
						return in response to this handler.
	@param isFcn		TRUE if this is a function handler, FALSE if it is a
						command handler. If Joker is set not to distinguish
						between function and command messages, the value of this
						is undefined.
	@param paramCount	The number of parameters passed to the handler.
	@param args			A va_list on which va_start() has already been called.
						This list contains paramCount JokerValueRefs that hold the
						parameters passed to the handler.
	@seealso JokerSetHandlerCallProc JokerCallResult JokerCallResultEnum */
typedef		pascal JokerCallResult	(*JokerHandlerCallProcPtr)( Str255 handlerName,
														JokerValueRef result,
														Boolean isFcn,
														long paramCount,
														va_list args );

// --------------------------------------------------------------------------------
/*!	@typedef 			JokerSpendTimeProcPtr
	@abstract			This is called between lines when running a handler.
	@discussion			You can use this to allow the user to switch to another
						application while a script is running.
	@seealso JokerSetSpendTimeProc */
typedef		pascal void	(*JokerSpendTimeProcPtr)();


// --------------------------------------------------------------------------------
/*!	@typedef 			JokerClientCmdProcPtr
	@abstract			Whenever a client command you registered using
						JokerRegisterClientCmd() is called by the user,
						this function is called to handle the command.
	@param paramCount	The number of parameters passed to the command.
	@param params		An array containing a JokerValueRef with the value for
						each parameter. You mustn't dispose of these values or
						assign a value to them.
	@seealso JokerRegisterClientCmd */
typedef		pascal void	(*JokerClientCmdProcPtr)( long paramCount,
													JokerValueRef params[] );


// --------------------------------------------------------------------------------
/*!	@typedef 			JokerEntityProcPtr
	@abstract			Whenever a user-defined object is manipulated, this
						proc is called.
	@param target		The object that was targeted.
	@param event		The action requested.
	@param param		A parameter, depending on the value of event.
	@param param2		A second parameter.
	@seealso JokerRegisterObjectDescriptor JokerEntityRef
				JokerEntityEvent JokerEntityEventEnum */
typedef		pascal JokerValueRef	(*JokerEntityProcPtr)( JokerEntityRef target,
															JokerEntityEvent event,
															JokerValueRef param,
															JokerValueRef param2 );


// --------------------------------------------------------------------------------
/*!	@typedef 			JokerObjectDescriptorProcPtr
	@abstract			Whenever a user-defined object descriptor is used,
						this function is called to return the appropriate object.
	@param paramCount	Number of parameters passed.
	@param params		Array with parameters.
	@result				Return a JokerEntityRef here, or NULL on error.
	@seealso JokerRegisterObjectDescriptor JokerEntityRef */
typedef		pascal JokerEntityRef	(*JokerObjectDescriptorProcPtr)( long paramCount,
														JokerValueRef params[] );


#ifdef GENERATING_JOKER_LIB		// Only used by Uli when compiling the library.
#pragma export on
#define JOKER_EXT		pascal
#else	// This is used by you when using the library:
#pragma import on
#define JOKER_EXT		extern pascal
#endif

#ifdef __cplusplus
extern "C" {
#endif


// ================================================================================
/*! @section Initialization&nbsp;and&nbsp;Global&nbsp;State
	Calls that manipulate Joker globally instead of per-instance. */
// ================================================================================

// --------------------------------------------------------------------------------
/*!	@function		JokerInitialize
	@abstract		Initialize Joker's internal global data structures.
	@discussion		Call this before making any other calls to Joker-related
					APIs.
	@seealso JokerShowSplash */
JOKER_EXT void	JokerInitialize();


// --------------------------------------------------------------------------------
/*!	@function		JokerShowSplash
	@abstract		Show Joker's splash screen.
	@discussion		If you haven't paid for Joker, you are required to call this
					function after initializing Joker using JokerInitialize(), or
					before your application quits.
					It will display a little splash screen telling the user that
					your product uses JokerLib. */
JOKER_EXT void	JokerShowSplash();

// --------------------------------------------------------------------------------
/*!	@function		JokerSetNewlineCharacter
	@abstract		Specify the default character to use for line breaks.
	@discussion		The constant &quot;newline&quot; represents the character to
					use for line breaks on the current platform. Depending on
					whether your application compiles under MPW or CodeWarrior,
					or whether you are deploying for the web or for the screen,
					you can specify the character to use for line breaks. This is
					also the character used when returning lists e.g. of the keys
					of an array.
	@param n		The character to use for line breaks by default. */
JOKER_EXT void	JokerSetNewlineCharacter( char n );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetOption
	@abstract		Turn on/off one of Joker's options.
	@discussion		Joker supports a number of options that control its behavior
					in certain situations, e.g. regarding treatment of unquoted
					string literals. These pertain to all scripts and must be set
					at startup, before parsing any scripts.
	@param x		The ID representing the option to set.
	@param state	TRUE to activate the behavior, FALSE to deactivate it. */
JOKER_EXT void	JokerSetOption( JokerOptionID x, Boolean state );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetOutputProc
	@abstract		Specify a callback function to handle console output.
	@discussion		Whenever the user requests data to be output to the console,
					either by using &quot;put&quot; without a destination or
					by assigning a value to standardOutput, the JokerOutputProc
					is called to display this value however you see fit.
	@param p		A pointer to a JokerOutputProc to call.
	@seealso JokerOutputProcPtr */
JOKER_EXT void	JokerSetOutputProc( JokerOutputProcPtr p );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetHandlerCallProc
	@abstract		Specify a callback function to be called when a message
					was not intercepted.
	@discussion		Whenever you send a message using JokerRunHandler() and the
					message is not intercepted by the user at all or the user
					passes the message after handling it, the HandlerCallProc is
					called to give you the opportunity of handling the message.
	@param p		A pointer to a JokerHandlerCallProc to call.
	@seealso JokerHandlerCallProcPtr */
JOKER_EXT void	JokerSetHandlerCallProc( JokerHandlerCallProcPtr p );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetSpendTimeProc
	@abstract		Specify a callback function to be called between lines.
	@discussion		After each line Joker executes during a call to
					JokerRunHandler(), the JokerSpendTimeProc is called. You can
					use this procedure to drive running animations or movies, give
					idle time to other programs etc.
	@param p		A pointer to a JokerSpendTimeProc to call.
	@seealso JokerSpendTimeProcPtr */
JOKER_EXT void	JokerSetSpendTimeProc( JokerSpendTimeProcPtr p );

// --------------------------------------------------------------------------------
/*!	@function				JokerRegisterClientCmd
	@abstract				Register a custom command.
	@discussion				Whenever a custom command is called by a script, the
							JokerClientCmdProc is called with the parameters in
							an array.<br>
							Currently, Joker has a limit of five parameters per
							client command. Also, you may only declare one command
							with the same name.
	@param paramCount		Number of parameters to the command, not including the
							direct argument.
	@param inIdentifiers	inIdentifiers[0] contains the name of the command.
							Any additional entries contain the labels for the
							corresponding indirect parameters. If any of those
							entries is NULL or an empty string, it is assumed
							that the parameter has no label. Labels and command
							names may only consist of one word. Joker makes a
							copy of the data in this array, you needn't keep it
							around after this call has returned.
	@param hasDirectArg		TRUE if you want to have a direct argument, FALSE if
							you want the command to start immediately with the
							first argument's label.
	@param p				The callback function to call whenever this command is
							invoked.
	@seealso JokerClientCmdProcPtr */
JOKER_EXT void			JokerRegisterClientCmd( long paramCount,
												StringPtr inIdentifiers[],
												Boolean hasDirectArg,
												JokerClientCmdProcPtr p );

// --------------------------------------------------------------------------------
/*!	@function				JokerRegisterObjectDescriptor
	@abstract				Register syntax for referring to your user-defined
							objects.
	@discussion				Whenever a script encounters an object descriptor in
							a script which you registered using this function, it
							calls the object descriptor proc you specified for
							this type. This proc is then responsible for returning
							the JokerEntityRef associated with that object.
							The paramCount and inIdentifiers and hasDirectArg
							parameters behave the same as for client commands.
	@param paramCount		Number of parameters to the command, not including the
							direct argument.
	@param inIdentifiers	inIdentifiers[0] contains the name of the object kind.
							Any additional entries contain the labels for the
							corresponding indirect parameters. If any of those
							entries is NULL or an empty string, it is assumed
							that the parameter has no label. Labels and command
							names may only consist of one word. Joker makes a
							copy of the data in this array, you needn't keep it
							around after this call has returned.
	@param hasDirectArg		TRUE if you want to have a direct argument, FALSE if
							you want the object descriptor to start immediately
							with the first argument's label.
	@seealso JokerObjectDescriptorProcPtr JokerEntityRef */
JOKER_EXT void			JokerRegisterObjectDescriptor( long paramCount,
													StringPtr inIdentifiers[],
													Boolean hasDirectArg,
													JokerObjectDescriptorProcPtr proc );


// --------------------------------------------------------------------------------
/*!	@function	JokerGetErrorString
	@abstract	Returns an error string, or an empty string if no error occurred.
	@param str	This Pascal string is set to the error message, or to an empty
				string if there was no error. This may be NULL if you're not
				interested in the error string but want to determine solely
				whether an error occurred.
	@result		TRUE if there was an error, FALSE if not. */
JOKER_EXT Boolean		JokerGetErrorString( StringPtr str );


// ================================================================================
/*! @section Manipulating&nbsp;Scripts */
// ================================================================================

// --------------------------------------------------------------------------------
/*!	@function		JokerNewScript
	@abstract		Create a new script reference.
	@discussion		This calls JokerRetainScript() on the new script once. Balance
					this call with one to JokerReleaseScript() when you are done
					using it.
	@result			A reference to a newly created script.
	@seealso JokerReleaseScript JokerRetainScript */
JOKER_EXT JokerScriptRef	JokerNewScript();

// --------------------------------------------------------------------------------
/*!	@function			JokerRetainScript
	@abstract			Increase the reference count of a script.
	@discussion			Scripts in Joker are reference-counted, this means that
						you can have several places of your code sharing a script
						without having to worry about one disposing of it and
						pulling the rug from under another place's feet. Every
						call to JokerRetainScript() must be balanced with a call
						to JokerReleaseScript(). As long as you are still
						retaining a script it can't be disposed of. When the last
						to retain releases a script, it will be disposed of.
						Since JokerRunHandler() also retains the scripts it runs,
						you can not dispose of a running script by accident.
	@param s			The script to retain.
	@seealso JokerReleaseScript */
JOKER_EXT void			JokerRetainScript( JokerScriptRef s );

// --------------------------------------------------------------------------------
/*!	@function			JokerReleaseScript
	@abstract			Decrease the reference count of a script, releasing it
						if the refcount hits zero.
	@param s			The script to release. The JokerScriptRef may not be
						valid anymore after calling this.
	@seealso JokerRetainScript */
JOKER_EXT void			JokerReleaseScript( JokerScriptRef s );

// --------------------------------------------------------------------------------
/*!	@function			JokerTokenizeScript
	@abstract			Assign the text of a script to a JokerScriptRef and
						prepare it for parsing.
	@discussion			You must tokenize a script using this call before you can
						parse it. You can call JokerTokenizeScript() a second time
						on a script to assign it new text, but then you must also
						re-parse it.<br>
						You do not need to keep around the script text
						after tokenizing it, Joker makes a copy for internal use.
	@param s			The script to assign the text to. You must have created
						this JokerScriptRef using JokerNewScript().
	@param data			A pointer to the script text.
	@param length		The size in bytes of the data in data. */
JOKER_EXT void			JokerTokenizeScript( JokerScriptRef s, const char* data,
												long length );

// --------------------------------------------------------------------------------
/*!	@function			JokerParseScript
	@abstract			Parse the list of tokens in a script.
	@discussion			Parsing a script converts the list of tokens into a list
						of instructions. This is not unlike compiling a script,
						just that Joker generates more high-level instructions,
						not machine code. You must parse a script using this call
						or using JokerParseScriptAsHandler() before you can run
						any handlers inside it.
	@param s			The script containing the tokens to parse. The script
						must have been tokenized already.
	@seealso JokerParseScriptAsHandler */
JOKER_EXT void			JokerParseScript( JokerScriptRef s );

// --------------------------------------------------------------------------------
/*!	@function			JokerParseScriptAsHandler
	@abstract			Parse the list of tokens in a script as if they were a
						handler with the specified name.
	@discussion			The text that was tokenized must be a raw list of
						commands, they mustn't contain any
						&quot;on&quot;/&quot;function&quot; lines or their
						corresponding &quot;end&quot; lines.<br>
						This call will currently only work if Joker is set up
						not to distinguish between function and command message
						handlers.
	@param s			The script containing the tokens to parse. The script
						must have been tokenized already.
	@param handlerName	The name of the handler to generate.
	@param isFcn		TRUE if you want to generate a function, FALSE
						if you want a command. This parameter is currently
						ignored.
	@seealso JokerParseScript */
JOKER_EXT void			JokerParseScriptAsHandler( JokerScriptRef s,
													Str255 handlerName,
													Boolean isFcn );

// --------------------------------------------------------------------------------
/*!	@function			JokerRunHandler
	@abstract			Run a function or command message handler in a
						JokerScriptRef.
	@discussion			A script may consist of a number of procedures and
						functions (command and function message handlers). This
						function looks up a function by name and runs it.
						While a script is running, this call increses its
						reference count so you can't accidentally dispose the
						running script.
	@param s			The script containing the handler to run. You must already
						have tokenized and parsed the script.
	@param handlerName	The name of the handler to call. Case is not important.
	@param isFcn		TRUE if the message handler to call is a function, FALSE
						if it is a command. This parameter is ignored if Joker is
						set up not to distinguish between commands and functions.
	@param returnValue	A JokerValueRef created using JokerNewValue(). This value
						will be set to the return value of the handler.
	@param paramCount	The number of parameters to follow.
	@param ...			paramCount parameters, each a JokerValueRef containing
						the values to pass as parameters to the handler.
	@result				A constantindicating whether the handler intercepted the
						message, exited to top or passed it.
	@seealso JokerCallResult JokerCallResultEnum */
JOKER_EXT JokerCallResult	JokerRunHandler( JokerScriptRef s, Str255 handlerName,
												Boolean isFcn, JokerValueRef returnValue,
												long paramCount, ... );


// ================================================================================
/*! @section Manipulating&nbsp;JokerEntityRefs */
// ================================================================================

// --------------------------------------------------------------------------------
/*!	@function		JokerNewEntity
	@abstract		Create a new JokerEntityRef.
	@discussion		Create a JokerEntityRef for each object which you want to be
					controllable from a script. Note that these objects will
					automatically keep track of any user properties that are
					defined for them.
	@result			A new JokerEntityRef. You are responsible for disposing of
					this. Whenever a script refers to this object and your
					JokerObjectDescriptorProc returns this JokerEntityRef, the
					associated JokerEntityProc is called.
	@seealso JokerDisposeEntity JokerEntityRef */
JOKER_EXT JokerEntityRef	JokerNewEntity( JokerEntityProcPtr p );

// --------------------------------------------------------------------------------
/*!	@function		JokerDisposeEntity
	@abstract		Get rid of a JokerEntityRef created using JokerNewEntity().
	@param jer		The JokerEntityRef you want to dispose of. This JokerEntityRef
					is no longer valid after this call.
	@seealso JokerNewEntity JokerEntityRef */
JOKER_EXT void				JokerDisposeEntity( JokerEntityRef jer );

// --------------------------------------------------------------------------------
/*!	@function		JokerGetEntityRefCon
	@abstract		Return the refCon of a JokerEntityRef.
	@discussion		The refCon is a long that you may freely use for whatever
					purpose you can think of. Usually a refCon is used to keep
					a pointer to the actual object that this JokerEntityRef is
					associated with.
	@param jer		The JokerEntityRef whose RefCon you want to get.
	@seealso JokerSetEntityRefCon JokerNewEntity JokerDisposeEntity JokerEntityRef */
JOKER_EXT long				JokerGetEntityRefCon( JokerEntityRef jer );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetEntityRefCon
	@abstract		Change the refCon of a JokerEntityRef.
	@discussion		The refCon is a long that you may freely use for whatever
					purpose you can think of. Usually a refCon is used to keep
					a pointer to the actual object that this JokerEntityRef is
					associated with.
	@param jer		The JokerEntityRef whose RefCon you want to change.
	@seealso JokerGetEntityRefCon JokerNewEntity JokerDisposeEntity JokerEntityRef */
JOKER_EXT void				JokerSetEntityRefCon( JokerEntityRef jer, long value );

// --------------------------------------------------------------------------------
/*!	@function		JokerGetEntityPropertyArray
	@abstract		Create an array with the property names as indexes and the
					property values as array element values.
	@param jer		The object whose properties you are interested in.
	@result			A value containing an array. You dispose of this.
	@seealso JokerGetEntityPropertyNameArray JokerSetEntityPropertyArray */
JOKER_EXT JokerValueRef		JokerGetEntityPropertyArray( JokerEntityRef jer );

// --------------------------------------------------------------------------------
/*!	@function		JokerGetEntityPropertyNameArray
	@abstract		Create an array containing the property names. The indexes are
					integers from 1 to the number of properties.
	@param jer		The object whose properties you are interested in.
	@result			A value containing an array. You need to dispose of this.
	@seealso JokerGetEntityPropertyArray JokerSetEntityPropertyArray */
JOKER_EXT JokerValueRef		JokerGetEntityPropertyNameArray( JokerEntityRef jer );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetEntityPropertyArray
	@abstract		Set the user properties of an entity to be the values of an
					array.
	@discussion		The array indexes are used as property names and the values
					of the array elements are used as property values.
	@param jer		The object whose user properties you want to change.
	@param v		An array with [property name] = <property value> elements,
					as it is returned by JokerGetEntityPropertyArray.
	@seealso JokerGetEntityPropertyArray */
JOKER_EXT void				JokerSetEntityPropertyArray( JokerEntityRef jer, JokerValueRef v );

// ================================================================================
/*! @section Manipulating&nbsp;JokerValueRefs */
// ================================================================================

// --------------------------------------------------------------------------------
/*!	@function		JokerNewValue
	@abstract		Create a new JokerValueRef.
	@result			A new JokerValueRef. You are responsible for disposing of
					this. You can assign a value to this JokerValueRef using
					any of the calls below.
	@seealso JokerDisposeValue JokerValueRef */
JOKER_EXT JokerValueRef	JokerNewValue();

// --------------------------------------------------------------------------------
/*!	@function		JokerCopyValue
	@abstract		Create a copy of a JokerValueRef.
	@param v		The JokerValueRef to copy.
	@result			A JokerValueRef to a copy of v. You are responsible for
					disposing of this.
	@seealso JokerCopyValue JokerValueRef */
JOKER_EXT JokerValueRef	JokerCopyValue( JokerValueRef v );

// --------------------------------------------------------------------------------
/*!	@function		JokerDisposeValue
	@abstract		Dispose of a JokerValueRef when you're done using it.
	@discussion		After you have called this, the JokerValueRef is no longer
					valid.<br>
					If the value passed in v contains an array, its array elements
					are automatically disposed of by this.<br>
					Do not call this function on any parameters you receive in
					any of the callback functions, or on any array elements you
					obtain using JokerFetchValueElement().
	@param v		The JokerValueRef to get rid of.
	@seealso JokerNewValue JokerFetchValueElement JokerValueRef */
JOKER_EXT void			JokerDisposeValue( JokerValueRef v );

// --------------------------------------------------------------------------------
/*!	@function		JokerGetValueAvailableTypes
	@abstract		Retrieve information on what data types are available in a value.
	@discussion		Use this to determine whether the value you are dealing with
					supports the data type you need.
	@param v		The JokerValueRef to get information on.
	@result			A bit field with constants from the JokerVTypesEnum indicating
					available types.
	@seealso JokerVTypes JokerVTypesEnum JokerValueRef */
JOKER_EXT JokerVTypes	JokerGetValueAvailableTypes( JokerValueRef v );

// ================================================================================
/*! @section Assigning&nbsp;Values&nbsp;to&nbsp;JokerValueRefs */
// ================================================================================
// --------------------------------------------------------------------------------
/*!	@function		JokerSetValueLong
	@abstract		Set the value of the specified JokerValueRef to a signed long
					integer.
	@param v		The JokerValueRef to which to assign the value.
	@param n		The value to assign to the JokerValueRef. */
JOKER_EXT void			JokerSetValueLong( JokerValueRef v, long n );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetValueDouble
	@abstract		Set the value of the specified JokerValueRef to a double.
	@param v		The JokerValueRef to which to assign the value.
	@param n		The value to assign to the JokerValueRef. */
JOKER_EXT void			JokerSetValueDouble( JokerValueRef v, double n );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetValueString
	@abstract		Set the value of the specified JokerValueRef to some text
					read from a Pascal string.
	@param v		The JokerValueRef to which to assign the value.
	@param str		A Pascal string with a length byte containing the text to copy
					to the JokerValueRef. */
JOKER_EXT void			JokerSetValueString( JokerValueRef v, Str255 str );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetValueChars
	@abstract		Set the value of the specified JokerValueRef to some text
					read from a char*.
	@param v		The JokerValueRef to which to assign the value.
	@param data		A pointer to the data to copy to the value.
	@param length	The number of bytes to copy from data to the JokerValueRef. */
JOKER_EXT void			JokerSetValueChars( JokerValueRef v, char* data, long length );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetValuePoint
	@abstract		Set the value of the specified JokerValueRef to a Point.
	@param v		The JokerValueRef to which to assign the value.
	@param pos		The value to assign. */
JOKER_EXT void			JokerSetValuePoint( JokerValueRef v, const Point pos );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetValueColor
	@abstract		Set the value of the specified JokerValueRef to an RGBColor.
	@param v		The JokerValueRef to which to assign the value.
	@param col		The value to assign. */
JOKER_EXT void			JokerSetValueColor( JokerValueRef v, const RGBColor* col );

// --------------------------------------------------------------------------------
/*!	@function		JokerSetValueRect
	@abstract		Set the value of the specified JokerValueRef to a rectangle.
	@param v		The JokerValueRef to which to assign the value.
	@param box		The value to assign. */
JOKER_EXT void			JokerSetValueRect( JokerValueRef v, const Rect* box );

// ================================================================================
/*! @section Arrays&nbsp;in&nbsp;JokerValueRefs */
// ================================================================================
// --------------------------------------------------------------------------------
/*!	@function		JokerFetchValueElement
	@abstract		This returns a reference to a value with the specified index
					in the specified value's array. If there is no such element yet,
					a new one is created. You get the actual element, modifying it
					modifies the array.
	@discussion		Every value can be an associative array, the indices are
					strings. However, a value can only <b>either</b> directly
					contain a value, <b>or</b> it can contain array elements.
					It may not contain both. If you assign one, the other is
					cleared.
	@param v		The JokerValueRef from which to get the element.
	@param index	The array index of the element you want to get/create.
	@result			The JokerValueRef to the element requested. Do not dispose of
					this JokerValueRef, or you'll damage the array it belongs to. */
JOKER_EXT JokerValueRef	JokerFetchValueElement( JokerValueRef v, Str255 index );


// --------------------------------------------------------------------------------
/*!	@function		JokerCountValueElements
	@abstract		Returns the number of elements in a JokerValueRef's array.
	@param v		The JokerValueRef of which to count the elements.
	@result			The number of elements in the array. */
JOKER_EXT long			JokerCountValueElements( JokerValueRef v );


// ================================================================================
/*! @section Retrieving&nbsp;Values&nbsp;from&nbsp;JokerValueRefs */
// ================================================================================
// --------------------------------------------------------------------------------
/*!	@function		JokerGetValueLong
	@abstract		Retrieve the value of the specified JokerValueRef as a
					signed long integer.
	@param v		The JokerValueRef from which to get the value.
	@result			The value retrieved. */
JOKER_EXT long			JokerGetValueLong( JokerValueRef v );

// --------------------------------------------------------------------------------
/*!	@function		JokerGetValueDouble
	@abstract		Retrieve the value of the specified JokerValueRef as a
					double.
	@param v		The JokerValueRef from which to get the value.
	@result			The value retrieved. */
JOKER_EXT double		JokerGetValueDouble( JokerValueRef v );

// --------------------------------------------------------------------------------
/*!	@function		JokerGetValueString
	@abstract		Retrieve the value of the specified JokerValueRef as a
					Pascal String with a length byte.
	@param v		The JokerValueRef from which to get the value.
	@param str		The value retrieved is assigned to the memory pointed to by
					this StringPtr. */
JOKER_EXT void			JokerGetValueString( JokerValueRef v, Str255 str );

// --------------------------------------------------------------------------------
/*!	@function		JokerGetValueChars
	@abstract		Retrieve the value of the specified JokerValueRef as a
					stream of characters.
	@param v		The JokerValueRef from which to get the value.
	@param data		The value retrieved is assigned to the memory pointed to by
					this char*.
	@param maxLen	The size of the buffer pointed to by data. This call will at
					most return maxLen bytes of data.
	@result			The number of characters still left after returning maxLen
					data. If this is a negative number, add it to maxLen to find
					out how much data was returned. */
JOKER_EXT long			JokerGetValueChars( JokerValueRef v, char* data, long maxLen );

// --------------------------------------------------------------------------------
/*!	@function		JokerGetValuePoint
	@abstract		Retrieve the value of the specified JokerValueRef as a
					Point.
	@param v		The JokerValueRef from which to get the value.
	@param pos		The value retrieved is assigned to the memory pointed to by
					this Point*. */
JOKER_EXT void			JokerGetValuePoint( JokerValueRef v, Point* pos );

// --------------------------------------------------------------------------------
/*!	@function		JokerGetValueColor
	@abstract		Retrieve the value of the specified JokerValueRef as a
					RGBColor.
	@param v		The JokerValueRef from which to get the value.
	@param col		The value retrieved is assigned to the memory pointed to by
					this RGBColor*. */
JOKER_EXT void			JokerGetValueColor( JokerValueRef v, RGBColor* col );

// --------------------------------------------------------------------------------
/*!	@function		JokerGetValueRect
	@abstract		Retrieve the value of the specified JokerValueRef as a
					rectangle.
	@param v		The JokerValueRef from which to get the value.
	@param box		The value retrieved is assigned to the memory pointed to by
					this Rect*. */
JOKER_EXT void			JokerGetValueRect( JokerValueRef v, Rect* box );
















#ifdef __cplusplus
}
#endif

#ifdef GENERATING_JOKER_LIB		// Only used by Uli when compiling the library.
#pragma export reset
#else
#pragma import reset
#endif




#endif /*ULI_JOKER_LIB_H*/