/*	***************************************************************************

	PROJECT:	Joker
	
	FILE:		carbon/CarbTalkURLValue.cpp
	
	PURPOSE:	MacOS X-specific implementation of a URL.
		
	COPYRIGHT:	(C) Copyright 2000 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
    NOTE:
        This is currently a half-breed between a Mac and a Unix port, mostly
        to prove to myself the portability of the code. I'll probably be
        changing this to use as many Unix/POSIX and ANSI APIs as possible,
        which should substantially cut down on later ports to other "real"
        Unixes.
	
	REVISIONS:
		2001-07-28	UK		Created based on Mac implementation.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkURLValue.h"
#include	"JokerMain.h"
#include	<cstdlib>
#include	<cstring>
#include	<dirent.h>
#if __MACH__
#include	<Carbon/Carbon.h>
#else
#include	<Carbon.h>
#endif

#pragma mark [Prototypes]

/* --------------------------------------------------------------------------------
	Prototypes:
   ----------------------------------------------------------------------------- */

// Go here.


#pragma mark -

/* --------------------------------------------------------------------------------
	JokerCStringToP:
		Copy a zero-terminated C-string to a P-String with a length byte.
		
	We could use c2pstr but that's only available pre-carbon. We could use
	CopyCStringToPascal, but that's only available in carbon. So we just roll our
	own.
   ----------------------------------------------------------------------------- */

void	JokerCStringToP( const char* cstr, unsigned char* outStr )
{
	outStr[0] = strlen(cstr);
	
	memmove( outStr +1, cstr, outStr[0] );
}

/* --------------------------------------------------------------------------------
	ReverseCString:
		Reverse the characters in a zero-terminated C string so the first letter
		is last and vice versa.
   ----------------------------------------------------------------------------- */

void	ReverseCString( char* vString )
{
	int		x, maxLen, halfLen;
	
	maxLen = strlen(vString);
	halfLen = maxLen /2;
	
	for( x = 0; x < halfLen; x++ )
	{
		char		tmp;
		
		tmp = vString[x];
		vString[x] = vString[maxLen -x -1];
		vString[maxLen -x -1] = tmp;
	}
}


void	JokerRsrcURLToMacPath( char* vURL, ResType* resType, short* resID, unsigned char* outPath )
{
	int			x = 0;
	char		vURLStringData[512];
	char*		vURLString = vURLStringData;
	strcpy( vURLString, vURL );
	
	/*if( vURLString[7] == '/' )	// Pathname relative to boot volume.
    {
		memmove( vURLString, vURLString +1, 512 );	// Remove leading slash.
    }
	else
	{
		memmove( vURLString +1, vURLString, 511 );
		vURLString[7] = '/';	// On Mac, relative pathnames start with a separator character.
	}
	
	// Convert URL string to a Mac path.
	for( x = 0; vURLString[x] != 0; x++ )
	{
		if( vURLString[x] == '/' )
			vURLString[x] = ':';
		else if( vURLString[x] == ':' )
			vURLString[x] = '/';
	}*/
	
	vURLString += 7;	// Make sure we skip protocol designator.
	x -= 7;	// Make sure x stays string's length so we can use it to loop over the string backwards.
	
	char		vIdStr[256];
	short		currCharCount = 0;
	
	vIdStr[0] = 0;
	
	// Go backwards over string and extract ID/name and type:
	for( ; ((--x) > 0) && (vURLString[x] != '/') && (currCharCount < 256); )
	{
		vIdStr[currCharCount++] = vURLString[x];
		vIdStr[currCharCount] = 0;
	}
	
	ReverseCString( vIdStr );	// We collected the ID backwards, turn it around.
	
	if( x == 0 )
		throw std::runtime_error("Malformed \"rsrc:\" URL, couldn't find type before ID/name.");
	if( vURLString[x] != '/' )
		throw std::runtime_error("Resource name/ID in \"rsrc:\" URL too long.");
	
	char* theEnd;
	*resID = strtol( vIdStr, &theEnd, 10 );
	// FIX ME! check "theEnd" here to determine whether it was a valid number, otherwise do a per-name lookup.
	
	currCharCount = 0;
	
	// Now keep scanning backwards for res type:
	for( ; ((--x) > 0) && vURLString[x] != '/' && currCharCount < 256; )	// After the above, x is string's length!
	{
		vIdStr[currCharCount++] = vURLString[x];
		vIdStr[currCharCount] = 0;
	}
	
	ReverseCString( vIdStr );	// We collected the string backwards, turn it around.
	
	if( x == 0 )
		throw std::runtime_error("Malformed \"rsrc:\" URL, couldn't find file name before type.");
	if( vURLString[x] != '/' || currCharCount != 4 )
		throw std::runtime_error("Resource type in \"rsrc:\" URL must be four characters long.");
	
	vURLString[x] = 0;	// terminate string before start of type and ID so all that's left is file path.
    //JokerCStringToP( vURLString, outPath );
	
	memmove( resType, vIdStr, 4 );
}

#pragma mark -

/* --------------------------------------------------------------------------------
	DownloadURLData:
		Fetch the data of the file pointed to by this URL and return it in a text
		munger.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2000-02-24	UK		Created.
   ----------------------------------------------------------------------------- */

TextMunger*	TalkURLValue::DownloadURLData() const
{
	TextMunger*		vResult;
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	size_t			vReadSize = 512;
	//short			x;
	
	mTheURL.CopyToString( vURLString );
	if( strncmp( vURLString, "file://", 7 ) == 0 )	// String starts with file protocol?
	{
		vURLString += 7;	// Make sure we skip protocol designator.
		
		FILE*		vFile;
		vResult = new TextMunger;
		
		vFile = fopen( vURLString, "r+" );
		if( vFile == NULL )
			throw std::runtime_error( "Couldn't open file of local URL." );
		
		// Read from that file in chunks of 512 bytes
		try
		{
			while( vReadSize == 512 )	// If fread returns less, it means we have reached end of file.
			{
				vReadSize = fread( vURLStringData, sizeof(char), vReadSize, vFile );
				vResult->InsertData( vURLStringData, vReadSize );
			}
		}
		catch( std::exception& err )
		{
			fclose( vFile );
			throw;
		}
		
		fclose( vFile );
	}
    else if( strncmp( vURLString, "rsrc://", 7 ) == 0 )	// String starts with Resource protocol?
	{
		short			refNum;
		unsigned char	vFName[512];
		Handle			resData = NULL;
		ResType			resType;
		short			resID;
        FSRef			fileRef;
        Boolean			isFolder;
        OSStatus		fileErr;
		
		JokerRsrcURLToMacPath( vURLString, &resType, &resID, vFName );
		
        fileErr = FSPathMakeRef( vFName, &fileRef, &isFolder );
        if( fileErr == fnfErr )
        {
            FILE* vFile = fopen( (char*) vFName, "w" );	// Just make sure a file exists so we can get an FSRef for it.
            if( !vFile )
                throw std::runtime_error( "Couldn't create file referenced by \"rsrc:\" URL (1)." );
            fclose( vFile );
            fileErr = FSPathMakeRef( vFName, &fileRef, &isFolder );	// Make an FSRef from the path (There's no PathToFSSpec call and we'd face a file name length limit anyway).
        }
        if( fileErr != noErr )
            throw std::runtime_error("Couldn't open resource file referenced by \"rsrc:\" URL.");
		// FIX ME! Make FSSpec from path and create the file for that if it's no res file yet and then open it.
        refNum = HOpenResFile( 0, 0L, vFName, fsRdPerm );
		if( refNum < 0 )
			throw std::runtime_error("Error opening resource file referenced by \"rsrc:\" URL.");
		try
		{
			UseResFile( refNum );	// In case it was already open.
			resData = Get1Resource( resType, resID );
			if( resData == NULL || ResError() != noErr )
				throw std::runtime_error( "Couldn't fetch resource referenced by \"rsrc:\" URL." );
			vResult = new TextMunger;
			HLock( resData );
			vResult->InsertData( *resData, GetHandleSize( resData ) );
			HUnlock( resData );
		}
		catch( std::exception& err )
		{
			if( resData )
				DisposeHandle( resData );
			CloseResFile( refNum );
			throw;
		}
		CloseResFile( refNum );
	}
    else
	{
		if( !URLAccessAvailable() )
			throw std::runtime_error( "URL Access is not available on this Macintosh." );
		
		OSStatus			err;
		URLOpenFlags		vFlags = kURLDisplayProgressFlag | kURLDisplayAuthFlag;
		Handle				vDestDataHandle;
		
		vDestDataHandle = NewHandle(0);
		if( MemError() != noErr || vDestDataHandle == NULL )
			throw std::bad_alloc();
		
		err = URLSimpleDownload( mTheURL.String(), (FSSpec*) NULL,
									vDestDataHandle,
									vFlags, NULL, (void*) 0 );
		if( err != noErr )
		{
			DisposeHandle( vDestDataHandle );
			throw std::runtime_error( "Download Failure." );
		}
		
		HLock( vDestDataHandle );
		try
		{
			vResult = new TextMunger( (*vDestDataHandle), GetHandleSize( vDestDataHandle ) );
			HUnlock( vDestDataHandle );
			DisposeHandle( vDestDataHandle );
		}
		catch( std::exception& err )
		{
			HUnlock( vDestDataHandle );
			DisposeHandle( vDestDataHandle );
			
			throw;
		}
		HUnlock( vDestDataHandle );
	}
  
	return vResult;
}


/* --------------------------------------------------------------------------------
	KillURLFile:
		Delete the file/folder referenced by this URL.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2001-07-15	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::KillURLFile()
{
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	//size_t			vReadSize = 512;
	short			x;
	
	mTheURL.CopyToString( vURLString );
	if( strncmp( vURLString, "file://", 7 ) == 0 )	// String starts with file protocol?
	{
		if( vURLString[7] == '/' )	// Absolute pathname.
			memmove( vURLString, vURLString +1, 512 );	// Remove leading slash.
		else
		{
			memmove( vURLString +1, vURLString, 511 );
			vURLString[7] = '/';	// On Mac, relative pathnames start with a separator character.
		}
		
		// Convert URL string to a Mac path.
		for( x = 0; vURLString[x] != 0; x++ )
		{
			if( vURLString[x] == '/' )
				vURLString[x] = ':';
			else if( vURLString[x] == ':' )
				vURLString[x] = '/';
		}
		
		FSSpec		theFile;
		
		vURLString += 6;	// Make sure we skip protocol designator, but skip one less as we'll use first byte as length byte.
		vURLString[0] = strlen(vURLString) -1;
		
		if( FSMakeFSSpec( 0, 0L, (unsigned char*) vURLString, &theFile ) != noErr )
			throw std::runtime_error("Couldn't find file referenced by \"file:\" URL.");
		
		FSpDelete( &theFile );
		
	}
    else if( strncmp( vURLString, "rsrc://", 7 ) == 0 )	// String starts with Resource protocol? Then we just delete that resource.
	{
		short		refNum;
		Str255		vFName;
		Handle		resData = NULL;
		ResType		resType;
		short		resID;
		
		JokerRsrcURLToMacPath( vURLString, &resType, &resID, vFName );
		
		refNum = HOpenResFile( 0, 0L, vFName, fsRdWrPerm );
		if( refNum < 0 )
			throw std::runtime_error("Error opening resource file referenced by \"rsrc:\" URL.");
		try
		{
			UseResFile( refNum );	// In case it was already open.
			resData = Get1Resource( resType, resID );
			if( resData == NULL || ResError() != noErr )
				throw std::runtime_error( "Couldn't find resource referenced by \"rsrc:\" URL." );
			RemoveResource( resData );
		}
		catch( std::exception& err )
		{
			CloseResFile( refNum );
			throw;
		}
		CloseResFile( refNum );
	}
	else
		throw std::runtime_error("Deleting remote files is not supported.");
}

/* --------------------------------------------------------------------------------
	UploadURLData:
		Replace the data of the file pointed to by this URL through the data in
		the text munger passed.
	
	TAKES:
		vData	-	The text data to upload.
	
	GIVES:
		-
	
	REVISIONS:
		2000-02-24	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::UploadURLData( TextMunger* vData ) const
{
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	size_t			vReadSize = 512;
	
	mTheURL.CopyToString( vURLString );
	if( strncmp( vURLString, "file://", 7 ) == 0 )	// String starts with file protocol?
	{
		vURLString += 7;	// Make sure we skip protocol designator.
				
		FILE*		vFile;
		
		vFile = fopen( vURLString, "w+" );
		if( vFile == NULL )
		{
			char		errStr[512];
			
			strcpy( errStr, "Couldn't open file \"" );
			strncat( errStr, vURLString, 512 );
			strncat( errStr, "\" of local URL.", 512 );
			
			throw std::runtime_error( errStr );
		}
		
		try
		{
			size_t		vStillToWrite = vData->GetLength();
			
			vData->SetOffset(0);	// Make sure we write all.
			
			// Now write in 512-byte blocks until we run out of data:
			while( vStillToWrite > 0 )
			{
				if( vStillToWrite > 512 )
					vReadSize = 512;
				else
					vReadSize = vStillToWrite;
				vStillToWrite -= vReadSize;
				vData->ReadData( vURLStringData, vReadSize );
				
				if( fwrite( vURLStringData, sizeof(char), vReadSize, vFile ) < vReadSize )
					throw std::runtime_error( "Couldn't write all data to file of local URL." );
			}
		}
		catch( std::exception& err )
		{
			fclose( vFile );
			throw;
		}
		
		fclose( vFile );
	}
    else if( strncmp( vURLString, "rsrc://", 7 ) == 0 )	// Starts with resource protocol?
	{
		short		refNum;
		Str255		vFName;
		Handle		resData = NULL;
		ResType		resType;
		short		resID;
		bool		didCreate = false;
		
		JokerRsrcURLToMacPath( vURLString, &resType, &resID, vFName );
		
		refNum = HOpenResFile( 0, 0L, vFName, fsRdWrPerm );
		if( ResError() == fnfErr )	// No such file?
		{
			HCreateResFile( 0, 0L, vFName );
			if( ResError() != noErr )
				throw std::runtime_error("Couldn't create file for \"rsrc:\" URL.");
			
			refNum = HOpenResFile( 0, 0L, vFName, fsRdWrPerm );
		}
		if( refNum < 0 )
		{
			char		errStr[512];
			
			strcpy( errStr, "Couldn't open url \"" );
			strncat( errStr, vURLString, 512 );
			strncat( errStr, "\".", 512 );
			
			throw std::runtime_error( errStr );
		}
		try
		{
			UseResFile( refNum );	// In case it was already open.
			resData = Get1Resource( resType, resID );
			if( resData == NULL || ResError() != noErr )
			{
				resData = NewHandle( vData->GetLength() );
				if( MemError() != noErr )
					throw std::bad_alloc();
				didCreate = true;
			}
			
			SetHandleSize( resData, vData->GetLength() );
			if( MemError() != noErr )
				throw std::bad_alloc();
			
			vData->SetOffset(0);	// Make sure we read the data from the start.
			
			HLock( resData );
			vData->ReadData( *resData, vData->GetLength() );
			HUnlock( resData );
			
			if( didCreate )
			{
				AddResource( resData, resType, resID, "\p" );
				if( ResError() != noErr )
					throw std::runtime_error( "Couldn't create new resource for non-existant \"rsrc:\" URL." );
			}
			ChangedResource( resData );
			WriteResource( resData );
		}
		catch( std::exception& err )
		{
			if( resData )
				DisposeHandle( resData );
			CloseResFile( refNum );
			throw;
		}
		CloseResFile( refNum );
	}
	else
	{
		if( !URLAccessAvailable() )
			throw std::runtime_error( "URL Access is not available on this Macintosh." );
		
		Str255				vFileName;
		FSSpec				vSpec;
		OSStatus			err = noErr;
		short				num, refNum;
		long				count;
		Handle				vDestDataHandle;
		
		vDestDataHandle = NewHandle(vData->GetLength());
		if( MemError() != noErr || vDestDataHandle == NULL )
			throw std::bad_alloc();
		try
		{
			vData->CopyToText( (*vDestDataHandle) );
		}
		catch( std::exception& err )
		{
			DisposeHandle( vDestDataHandle );
			throw;
		}
				
		// Create FSSpec to unique file name:
		while( err != fnfErr )
		{
			num = Random();
			NumToString( num, vFileName );
			err = FSMakeFSSpec( 0, 0, vFileName, &vSpec );
		}
		
		// Create & open temp file and write
		err = FSpCreate( &vSpec, '????', 'TEXT', smCurrentScript );
		if( err != noErr )
			throw std::runtime_error( "Couldn't create temp file for uploading." );
		
		try
		{
			try
			{
				err = FSpOpenDF( &vSpec, fsRdWrPerm, &refNum );
				if( err != noErr )
					throw std::runtime_error( "Couldn't open temp file for uploading." );
				
				count = GetHandleSize( vDestDataHandle );
				err = SetEOF( refNum, count );
				if( err != noErr )
					throw std::runtime_error( "Couldn't enlarge temp file for uploading." );
				
				err = SetFPos( refNum, fsFromStart, 0 );
				if( err != noErr )
					throw std::runtime_error( "Couldn't prepare writing to temp file for uploading." );
				
				err = FSWrite( refNum, &count, (*vDestDataHandle) );
				if( err != noErr )
					throw std::runtime_error( "Couldn't write to temp file for uploading." );
				
				FSClose( refNum );
				DisposeHandle( vDestDataHandle );
			}
			catch( std::exception& err )
			{
				DisposeHandle( vDestDataHandle );
				throw;
			}
				
			// Upload!
			err = URLSimpleUpload( mTheURL.String(), &vSpec,
									kURLDisplayProgressFlag | kURLDisplayAuthFlag,
									NULL, (void*) 0 );
			if( err != noErr )
				throw std::runtime_error( "Upload Failure." );
			
			FSpDelete( &vSpec );
		}
		catch( std::exception& err )
		{
			FSpDelete( &vSpec );
			throw;
		}
	}
}


/* --------------------------------------------------------------------------------
	GetPropertyValue:
		Determine the values of any properties of this URL. This implements the
		"files" property by listing the files in the specified directory using
		MoreFiles or calling upon URL access to list the files in a remote folder.
	
	TAKES:
		pName		-	Name of the property to get.
		outValue	-	A TalkValue* in which we store the property value.
	
	GIVES:
		-
	
	REVISIONS:
		2001-01-28	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::GetPropertyValue( const TextMunger& pName, TalkValue& outValue ) const
{
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	//OSErr			err;
	ValueStorage	vOutData;
	
	if( pName == "files" )
	{
		mTheURL.CopyToString( vURLString );
		if( strncmp( vURLString, "file://", 7 ) == 0 )	// String starts with file protocol?
		{
			vURLString += 7;	// Make sure we skip protocol designator.
			
            vOutData.textType = new TextMunger();
            TexMunAPtr		vKiller( vOutData.textType );
            
            DIR*				vDirectory = NULL;
            struct dirent*		vCurrFile = NULL;
            unsigned long		x = 0;
            
            // Scan folder:
            vDirectory = opendir( vURLString );
            
            while( (vCurrFile = readdir(vDirectory)) != NULL )
            {
                if( ++x == 1 || x == 2 )
                    continue;	// Skip the two first items, which are "." and "..".
                
                vOutData.textType->SetOffset( vOutData.textType->GetLength() );
                vOutData.textType->Insert<char>( '\n' );
                vOutData.textType->InsertData( vCurrFile->d_name, strlen(vCurrFile->d_name) );
                
                // Give application time to handle events, other threads etc.:
                JokerSpendTime();
            }
            
            closedir( vDirectory );
            
            // Get rid of leading return:
            if( vOutData.textType->GetLength() > 1 )
            {
                vOutData.textType->SetOffset(0);
                vOutData.textType->DeleteData(1);
            }
            outValue.SetValue( vOutData, VALUE_TYPE_TEXT );
		}
		else	// Remote URL?
		{
			if( !URLAccessAvailable() )
				throw std::runtime_error( "URL Access is not available on this Macintosh." );
			
			OSStatus			err;
			Handle				vDestDataHandle;
			
			vDestDataHandle = NewHandle(0);
			if( MemError() != noErr || vDestDataHandle == NULL )
				throw std::bad_alloc();
			
			err = URLSimpleDownload( mTheURL.String(), (FSSpec*) NULL,
										vDestDataHandle,
										kURLDisplayProgressFlag | kURLDisplayAuthFlag
										| kURLDirectoryListingFlag,
										NULL, (void*) 0 );
			if( err != noErr )
			{
				DisposeHandle( vDestDataHandle );
				throw std::runtime_error( "Directory listing failure." );
			}
			
			HLock( vDestDataHandle );
			try
			{
				vOutData.textType = NULL;
				vOutData.textType = new TextMunger( (*vDestDataHandle), GetHandleSize( vDestDataHandle ) );
				HUnlock( vDestDataHandle );
				DisposeHandle( vDestDataHandle );
				
				outValue.SetValue( vOutData, VALUE_TYPE_TEXT );
				delete vOutData.textType;
			}
			catch( std::exception& err )
			{
				HUnlock( vDestDataHandle );
				DisposeHandle( vDestDataHandle );
				if( vOutData.textType != NULL )
					delete vOutData.textType;
				
				throw;
			}
		}
	}
	else
	{
		char	errstr[1024] = "URLs don't have a \"";	// FIX ME! Size limit on error message.
		
		strncat( errstr, pName.String(), 1024 );
		strncat( errstr, "\" property.", 1024 );
		throw std::runtime_error( errstr );
	}
}
