/*	***************************************************************************

	PROJECT:	HyperToken
	
	FILE:		HyperToken.cpp
	
	PURPOSE:	One token in a HyperTalk script.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-02-14	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */
   
#include	"HyperToken.h"
#include	<iostream>


/* --------------------------------------------------------------------------------
	PrintToken:
		Command to aid in debugging. This outputs the characteristics of this
		token to the console. This happens in a way that you can print all tokens
		in a row to get something that looks almost like the script entered.
	
	REVISIONS:
		1999-12-12	UK	Moved from HyperTalk.cpp.
   ----------------------------------------------------------------------------- */

void	HyperToken::PrintToken()
{
	switch( mTokenType )
	{
		case TOKEN_TYPE_STRING:
			std::cout << "\"" << ((char*) mTokenValue) << "\" ";
			break;
		
		case TOKEN_TYPE_IDENTIFIER:
			std::cout << "'" << ((char*) mTokenValue) << "' ";
			break;
		
		case TOKEN_TYPE_OPERATOR:
			std::cout << ((char) mTokenValue) << " ";
			break;
		
		case TOKEN_TYPE_NEWLINE:
			std::cout << "\\n\n";
			break;
		
		case TOKEN_TYPE_START_HANDLER:
			std::cout << "ON ";
			break;
		
		case TOKEN_TYPE_START_FUNCTION:
			std::cout << "FUNCTION ";
			break;
		
		case TOKEN_TYPE_END:
			std::cout << "END ";
			break;
		
		case TOKEN_TYPE_PUT_COMMAND:
			std::cout << "PUT ";
			break;
		
		case TOKEN_TYPE_INTO_IDENTIFIER:
			std::cout << "INTO ";
			break;
		
		case TOKEN_TYPE_IF_IDENTIFIER:
			std::cout << "IF ";
			break;
		
		case TOKEN_TYPE_THEN_IDENTIFIER:
			std::cout << "THEN ";
			break;
		
		case TOKEN_TYPE_ELSE_IDENTIFIER:
			std::cout << "ELSE ";
			break;
		
		case TOKEN_TYPE_REPEAT_IDENTIFIER:
			std::cout << "REPEAT ";
			break;
		
		case TOKEN_TYPE_WHILE_IDENTIFIER:
			std::cout << "WHILE ";
			break;
		
		case TOKEN_TYPE_ENTRY_IDENTIFIER:
			std::cout << "ENTRY ";
			break;
		
		case TOKEN_TYPE_OF_IDENTIFIER:
			std::cout << "OF ";
			break;
		
		case TOKEN_TYPE_GLOBAL_IDENTIFIER:
			std::cout << "GLOBAL ";
			break;
		
		case TOKEN_TYPE_RETURN_IDENTIFIER:
			std::cout << "RETURN ";
			break;
		
		case TOKEN_TYPE_CHAR_IDENTIFIER:
			std::cout << "CHAR ";
			break;
		
		case TOKEN_TYPE_TO_IDENTIFIER:
			std::cout << "TO ";
			break;
		
		case TOKEN_TYPE_SET_COMMAND:
			std::cout << "SET ";
			break;
		
		case TOKEN_TYPE_INTEGER:
			std::cout << "#" << mTokenValue << " ";
			break;
	}
}


/* --------------------------------------------------------------------------------
	operator==:
		Compare two tokens with each other. This uses the token type as its
		primary comparison criterion and the value as the second, which means you
		can _not_ use this to compare TOKEN_TYPE_IDENTIFIER tokens.
	
	TAKES:
		inMunger	-	The text munger to compare to this one.
	
	GIVES:
		-
	
	REVISIONS:
		2001-03-10	UK	Created.
   ----------------------------------------------------------------------------- */

bool	HyperToken::operator== ( const HyperToken& t ) const
{
	if( mTokenType != t.mTokenType )
		return false;
	
	switch( mTokenType )	// Only look at value if it isn't a pointer for this type:
	{
		case TOKEN_TYPE_OPERATOR:
		case TOKEN_TYPE_INTEGER:
			if( mTokenValue != t.mTokenValue )
				return false;
			break;
	}
	
	return true;
}


/* --------------------------------------------------------------------------------
	operator<:
		Compare two tokens with each other. This uses the token type as its
		primary search criterion and the value as the second, which means you can
		_not_ use this to compare TOKEN_TYPE_IDENTIFIER tokens.
	
	TAKES:
		t	-	The RHS argument to the < operator.
	
	GIVES:
		-
	
	REVISIONS:
		2001-03-10	UK	Created.
   ----------------------------------------------------------------------------- */

bool	HyperToken::operator< ( const HyperToken& t ) const
{
	if( mTokenType < t.mTokenType )
		return true;
	
	if( mTokenValue < t.mTokenValue )
		return true;
	
	return false;
}


/* --------------------------------------------------------------------------------
	operator>:
		Compare two tokens with each other. This uses the token type as its
		primary search criterion and the value as the second, which means you can
		_not_ use this to compare TOKEN_TYPE_IDENTIFIER tokens.
	
	TAKES:
		t	-	The RHS argument to the > operator.
	
	GIVES:
		-
	
	REVISIONS:
		2001-03-10	UK	Created.
   ----------------------------------------------------------------------------- */

bool	HyperToken::operator> ( const HyperToken& t ) const
{
	if( mTokenType > t.mTokenType )
		return true;
	
	if( mTokenValue > t.mTokenValue )
		return true;
	
	return false;
}