/*	***************************************************************************

	PROJECT:	MaxJoker.mcp
	
	FILE:		TalkExtendableEntity.cpp
	
	PURPOSE:	Entity that knows how to handle user properties.
		
	COPYRIGHT:	(C) Copyright 2000 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		2000-11-21	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkExtendableEntity.h"
#include	"TalkVarValue.h"
//#include	"missingproperty_error.h"


#pragma mark [Globals]

/* --------------------------------------------------------------------------------
	Globals/Static Variables:
   ----------------------------------------------------------------------------- */

// Put these li'l critters here.


#pragma mark -
#pragma mark [Init/Kill]

/* --------------------------------------------------------------------------------
	* CONSTRUCTOR:
		Do some initialization.
	
	TAKES:
		-
		
	GIVES:
		-
	
	REVISIONS:
		2000-11-21	UK		Created.
   ----------------------------------------------------------------------------- */

TalkExtendableEntity::TalkExtendableEntity()
: TalkEntity()
{
	// Do your stuff here!
}


/* --------------------------------------------------------------------------------
	* DESTRUCTOR:
		Do some cleanup.
	
	REVISIONS:
		2000-11-21	UK		Created.
   ----------------------------------------------------------------------------- */

TalkExtendableEntity::~TalkExtendableEntity()
{
	KillUserProperties();
}


#pragma mark [Class Methods]

/* --------------------------------------------------------------------------------
	GetPropertyValue:
		Retrieve a user property from this object.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2000-11-21	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkExtendableEntity::GetPropertyValue( const TextMunger& pName,
												TalkValue& outValue ) const
{
	TalkValueList::const_iterator		vIterator;
	
	vIterator = mUserProperties.find( pName );
	if( vIterator != mUserProperties.end() )	// Found this property.
		(*vIterator).second->CopyValueTo( outValue );	// Don't follow any refs, outValue is a temp.
	else {}	// Didn't find it:
		//throw missingproperty_error( "No such property.", pName );
}


/* --------------------------------------------------------------------------------
	SetPropertyValue:
		Change a user property of this object.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2000-11-21	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkExtendableEntity::SetPropertyValue( const TextMunger& pName,
												const TalkValue& inValue )
{
	TalkValueList::iterator		vIterator;
	
	vIterator = mUserProperties.find( pName );
	if( vIterator != mUserProperties.end() )	// Found this property.
		inValue.CopyValueTo( *((*vIterator).second), true, true );	// Follow refs.
	else {}	// Didn't find it:
		//throw missingproperty_error( "No such property.", pName );
}


/* --------------------------------------------------------------------------------
	DefineProperty:
		Add a user property to this object.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2000-11-25	UK		Changed to not error on existing user properties.
		2000-11-21	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkExtendableEntity::DefineProperty( const TextMunger& pName )
{
	TalkValueList::iterator		vIterator;
	
	vIterator = mUserProperties.find( pName );
	if( vIterator == mUserProperties.end() )	// Doesn't exist yet?
	{
		 TalkVarValue*	mVal = new TalkVarValue( TextMunger("") );	// Make new one.
		 mVal->Retain();
		 mUserProperties[pName] = mVal;
	}
}


/* --------------------------------------------------------------------------------
	UndefineProperty:
		Remove a user property from this object.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2000-11-21	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkExtendableEntity::UndefineProperty( const TextMunger& pName )
{
	TalkValueList::iterator		vIterator;
	
	vIterator = mUserProperties.find( pName );
	if( vIterator != mUserProperties.end() )
	{
		(*vIterator).second->Release();			// Kill its value.
		mUserProperties.erase( vIterator );	// Remove entry in list.
	}
	else {}
		//throw missingproperty_error( "No such user property.", pName );
}


/* --------------------------------------------------------------------------------
	MergePropertyArrays:
		Merge two object's property arrays, by copying the properties of the
		input object to this one. This will either replace properties with the
		same name, or simply not copy properties this object already has.
	
	TAKES:
		ent				-	An object whose properties are to be copied to this
							one.
		skipDuplicates	-	TRUE not to copy properties that already exist in
							this object, FALSE to replace properties that
							already exist in this object with the ones in ent.
		
	REVISIONS:
		2002-04-02	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkExtendableEntity::MergePropertyArrays( TalkExtendableEntity* ent, bool skipDuplicates )

{
	TalkValueList::iterator		vDstIterator,
								vSrcIterator;
	
	for( vSrcIterator = ent->mUserProperties.begin(); vSrcIterator != ent->mUserProperties.end(); vSrcIterator++ )
	{
		vDstIterator = mUserProperties.find( vSrcIterator->first );
		if( vDstIterator == mUserProperties.end() )	// Doesn't exist yet?
		{
			TalkVarValue*	mVal = new TalkVarValue( 0L );	// Make new one.
			mVal->Retain();
			vSrcIterator->second->CopyValueTo( *mVal );
			mUserProperties[vSrcIterator->first] = mVal;
		}
		else if( !skipDuplicates )	// Exists and we're supposed to replace?
			vSrcIterator->second->CopyValueTo( *vDstIterator->second );	// Overwrite old value.
		// else just skip this source value.
	}
}


/* --------------------------------------------------------------------------------
	GetPropertyArray:
		Return a TalkValue that is an array of property values with the property
		names as the indexes.
	
	TAKES:
		-
	
	GIVES:
		TalkValue*	-	A pointer to a TalkVarValue object containing the list.
						You are responsible for disposing of this.
	
	REVISIONS:
		2001-07-18	UK		Created.
   ----------------------------------------------------------------------------- */

TalkValue*	TalkExtendableEntity::CopyPropertyArray()
{
	TalkValueList::iterator		vIterator;
	TalkVarValue*				vValue = new TalkVarValue(0L);
	TalkValue*					vElement;
	
	try {
		for( vIterator = mUserProperties.begin(); vIterator != mUserProperties.end(); vIterator++ )
		{
			vValue->GetListEntry( vElement, vIterator->first );	// Create entry with property name as index.
			vIterator->second->CopyValueTo( *vElement );		// Copy property value to that entry.
		}
		vValue->Retain();
	}
	catch( ... )
	{
		vValue->Release();
		throw;
	}
	
	return vValue;
}


/* --------------------------------------------------------------------------------
	GetPropertyNameArray:
		Return an array containing the names of all properties in this object.
		The keys are integers from 1 to n.
		
		The value is created for you. You _must_ get rid of it by calling
		Release() on it.
	
	TAKES:
		-
	
	GIVES:
		TalkValue*	-	A TalkVarValue object containing the array. You are
						responsible for disposing of this.
	
	REVISIONS:
		2001-07-18	UK		Created.
   ----------------------------------------------------------------------------- */

TalkValue*	TalkExtendableEntity::CopyPropertyNameArray()
{
	TalkValueList::iterator		vIterator;
	TalkVarValue*				vValue = new TalkVarValue(0L);
	TalkValue*					vElement;
	ValueStorage				vStor;
	char						vStr[20];
	long						x = 0;
	
	try {
		for( vIterator = mUserProperties.begin(); vIterator != mUserProperties.end(); vIterator++ )
		{
			sprintf( vStr, "%ld", ++x );
			TextMunger		vIndex( vStr );
			vValue->GetListEntry( vElement, vIndex );
			vStor.textType = new TextMunger( vIterator->first );
			TexMunAPtr	vKiller( vStor.textType );
			vElement->SetValue( vStor, VALUE_TYPE_TEXT );
		}
		vValue->Retain();
	}
	catch( ... )
	{
		vValue->Release();
		throw;
	}
	
	return vValue;
}


/* --------------------------------------------------------------------------------
	SetPropertyArray:
		Set the user properties of this object to have the names of the keys in
		an array and the values of the array elements.
	
	TAKES:
		vValue	-	An array of [property name] = <property value> elements
					like it's returned by GetPropertyArray(). The values in this
					array are COPIED to the user properties list.
	
	GIVES:
		-
	
	REVISIONS:
		2001-07-18	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkExtendableEntity::SetPropertyArray( const TalkValue& vValue )
{
	KillUserProperties();	// First get rid of any user properties this obj may have had.
	vValue.CopyListTo( mUserProperties );
}


/* --------------------------------------------------------------------------------
	KillUserProperties:
		Dispose of all user properties this object has and empty the list.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2001-07-18	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkExtendableEntity::KillUserProperties()
{
	TalkValueList::iterator		vIterator;
	
	for( vIterator = mUserProperties.begin(); vIterator != mUserProperties.end(); vIterator++ )
		vIterator->second->Release();
	
	mUserProperties.clear();
}




