/*	***************************************************************************

	PROJECT:	TalkMemLocation
	
	FILE:		TalkMemLocation.cpp
	
	PURPOSE:	A memory address with relative addressing that refers to an
				entry on the stack, or contains an immediate value.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-02-14	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */
   
#include	"TalkMemLocation.h"
#include	<stdexcept>


#pragma mark [Implementation]

/* --------------------------------------------------------------------------------
	GetValue:
		Get the value from a TalkMemLocation, fetching stack locations from
		the stack while returning immediates directly.
	
	REVISIONS:
		2002-01-01	UK	Now throws exception when it encounters
						MEM_LOCATION_TYPE_INVALID.
   ----------------------------------------------------------------------------- */

void	TalkMemLocation::GetValue( ValueStorage& outValue, ValueStack& s,
									unsigned long desiredType )
{
	switch( mType )
	{
		case MEM_LOCATION_TYPE_IMMEDIATE:
			mValue->GetValue( outValue, desiredType );
			break;
		
		case MEM_LOCATION_TYPE_STACK:
			s[ s.GetBasePointer() +mLocation ]->GetValue( outValue, desiredType );
			break;
		
		case MEM_LOCATION_TYPE_LONG:
			if( desiredType != VALUE_TYPE_LONG )
				throw std::runtime_error( "MEM_LOCATION_TYPE_LONG can't be returned as other types." );
			outValue.longType = mLong;
			break;
		
		case MEM_LOCATION_TYPE_INVALID:
			throw std::runtime_error( "TalkMemLocation::GetValue(1) - Internal error: MEM_LOCATION_TYPE_INVALID encountered." );
			break;
	}
}


/* --------------------------------------------------------------------------------
	SetValue:
		Set the value of a TalkMemLocation, fetching stack locations from
		the stack.
	
	REVISIONS:
		2002-01-01	UK	Now throws exception when it encounters
						MEM_LOCATION_TYPE_INVALID.
   ----------------------------------------------------------------------------- */

void	TalkMemLocation::SetValue( ValueStorage& inValue, ValueStack& s,
									unsigned long providedType, bool followRefs )
{
	switch( mType )
	{
		case MEM_LOCATION_TYPE_IMMEDIATE:
			throw std::runtime_error( "Tried to change immediate." );
			break;
		
		case MEM_LOCATION_TYPE_STACK:
			s[ s.GetBasePointer() +mLocation ]->SetValue( inValue, providedType, followRefs );
			break;
		
		case MEM_LOCATION_TYPE_LONG:
			throw std::runtime_error( "Tried to change MEM_LOCATION_TYPE_LONG." );
			break;
		
		case MEM_LOCATION_TYPE_INVALID:
			throw std::runtime_error( "TalkMemLocation::SetValue - Internal error: MEM_LOCATION_TYPE_INVALID encountered." );
			break;
	}
}


/* --------------------------------------------------------------------------------
	CopyValueTo:
		Call the referenced value's CopyValueTo() method or do the equivalent.
	
	REVISIONS:
		2002-01-01	UK	Now throws exception when it encounters
						MEM_LOCATION_TYPE_INVALID.
   ----------------------------------------------------------------------------- */

void	TalkMemLocation::CopyValueTo( TalkValue& vValue, ValueStack &s, bool followSrcRefs, bool followDstRefs )
{
	switch( mType )
	{
		case MEM_LOCATION_TYPE_IMMEDIATE:
			mValue->CopyValueTo( vValue, followSrcRefs, followDstRefs );
			break;
		
		case MEM_LOCATION_TYPE_STACK:
			s[ s.GetBasePointer() +mLocation ]->CopyValueTo( vValue, followSrcRefs, followDstRefs );
			break;
		
		case MEM_LOCATION_TYPE_LONG:	// This doesn't need to follow refs as the MemLocation contains the value.
			ValueStorage		vs;
			vs.longType = mLong;
			vValue.SetValue( vs, VALUE_TYPE_LONG );
			break;
		
		case MEM_LOCATION_TYPE_INVALID:
			throw std::runtime_error( "TalkMemLocation::CopyValueTo - Internal error: MEM_LOCATION_TYPE_INVALID encountered." );
			break;
	}
}


/* --------------------------------------------------------------------------------
	GetAvailableTypes:
		Returns as what types we can return this location's value.
	
	REVISIONS:
		2002-01-01	UK	Now throws exception when it encounters
						MEM_LOCATION_TYPE_INVALID.
		1999-12-12	UK	Created.
   ----------------------------------------------------------------------------- */

unsigned long	TalkMemLocation::GetAvailableTypes( ValueStack &s )
{
	switch( mType )
	{
		case MEM_LOCATION_TYPE_IMMEDIATE:
			return mValue->GetAvailableTypes();
			break;
		
		case MEM_LOCATION_TYPE_STACK:
			return s[ s.GetBasePointer() +mLocation ]->GetAvailableTypes();
			break;
		
		case MEM_LOCATION_TYPE_LONG:
			return VALUE_TYPE_LONG;
			break;
		
		case MEM_LOCATION_TYPE_INVALID:
			throw std::runtime_error( "TalkMemLocation::GetAvailableTypes - Internal error: MEM_LOCATION_TYPE_INVALID encountered." );
			break;
	}
	
	return 0;
}


/* --------------------------------------------------------------------------------
	GetListEntry:
		Returns a value at the specified position in a list.
	
	REVISIONS:
		2002-01-01	UK	Now throws exception when it encounters
						MEM_LOCATION_TYPE_INVALID.
		1999-12-15	UK	Created.
   ----------------------------------------------------------------------------- */

void	TalkMemLocation::GetListEntry( TalkValue* &returnValue, ValueStack& s,
										TextMunger& index )
{
	switch( mType )
	{
		case MEM_LOCATION_TYPE_IMMEDIATE:
			mValue->GetListEntry( returnValue, index );
			break;
		
		case MEM_LOCATION_TYPE_STACK:
			s[ s.GetBasePointer() +mLocation ]->GetListEntry( returnValue, index );
			break;
		
		case MEM_LOCATION_TYPE_LONG:
			throw std::runtime_error( "Can't return list item of MEM_LOCATION_TYPE_LONG." );
			break;
		
		case MEM_LOCATION_TYPE_INVALID:
			throw std::runtime_error( "TalkMemLocation::GetListEntry - Internal error: MEM_LOCATION_TYPE_INVALID encountered." );
			break;
	}
}


/* --------------------------------------------------------------------------------
	GetValue:
		Returns the value object we refer to. The value belongs to the stack etc.
		DO NOT DISPOSE!
	
	REVISIONS:
		2002-01-01	UK	Now throws exception when it encounters
						MEM_LOCATION_TYPE_INVALID.
		1999-12-15	UK	Created.
   ----------------------------------------------------------------------------- */

void	TalkMemLocation::GetValue( TalkValue* &returnValue, ValueStack& s )
{
	switch( mType )
	{
		case MEM_LOCATION_TYPE_IMMEDIATE:
			returnValue = mValue;
			break;
		
		case MEM_LOCATION_TYPE_STACK:
			returnValue = s[ s.GetBasePointer() +mLocation ];
			break;
		
		case MEM_LOCATION_TYPE_LONG:
			throw std::runtime_error( "Can't return value object of MEM_LOCATION_TYPE_LONG." );
			break;
		
		case MEM_LOCATION_TYPE_INVALID:
			throw std::runtime_error( "TalkMemLocation::GetValue(2) - Internal error: MEM_LOCATION_TYPE_INVALID encountered." );
			break;
	}
}