/*	***************************************************************************

	PROJECT:	TalkValue
	
	FILE:		TalkValue.cpp
	
	PURPOSE:	Base class for any value like variables, expressions etc.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-12-29	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkValue.h"
#include	<cstdlib>
#include	<stdexcept>
#include	"strcase.h"


#pragma mark [Globals]

/* --------------------------------------------------------------------------------
	Globals/Static Variables:
   ----------------------------------------------------------------------------- */

// Put these li'l critters here.


#pragma mark -
#pragma mark [Class Methods]

/* --------------------------------------------------------------------------------
	TextToLong:
		Convert a TextMunger to a long.
	
	TAKES:
		inText	-	the text data to convert.
	
	GIVES:
		long	-	The converted value.
	
	REVISIONS:
		1999-12-29	UK		Created.
   ----------------------------------------------------------------------------- */

long	TalkValue::TextToLong( const TextMunger& inText )
{
	long		vReturnValue;
	char		vTempString[256];
	char*		vStrEnd;
	
	inText.CopyToString( vTempString );
	vReturnValue = strtol( vTempString, &vStrEnd, 10 );	// 10 means decimal number, not hex.
	
	return vReturnValue;
}


/* --------------------------------------------------------------------------------
	TextToPoint:
		Convert a TextMunger to a point.
	
	TAKES:
		inText	-	the text data to convert.
	
	GIVES:
		outPos	-	The converted value.
	
	REVISIONS:
		2000-10-22	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkValue::TextToPoint( const TextMunger& inText, LongPoint& outPos )
{
	char		vTempString[256];
	char*		vStrEnd;
	
	// Convert first coordinate:
	inText.CopyToString( vTempString );
	outPos.x = strtol( vTempString, &vStrEnd, 10 );	// 10 means decimal number, not hex.
	
	// Now skip spaces and comma:
	while( vStrEnd[0] == ','
			|| vStrEnd[0] == ' '
			|| vStrEnd[0] == '\t' )
	{
		vStrEnd++;
	}
	
	// Convert second coordinate:
	outPos.y = strtol( vStrEnd, &vStrEnd, 10 );	// 10 means decimal number, not hex.
}


/* --------------------------------------------------------------------------------
	TextToRect:
		Convert a TextMunger to a rect.
	
	TAKES:
		inText	-	the text data to convert.
	
	GIVES:
		outPos	-	The converted value.
	
	REVISIONS:
		2000-10-22	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkValue::TextToRect( const TextMunger& inText, LongRect& outPos )
{
	char		vTempString[256];
	char*		vStrEnd;
	
	// Convert first coordinate:
	inText.CopyToString( vTempString );
	outPos.left = strtol( vTempString, &vStrEnd, 10 );	// 10 means decimal number, not hex.
	
	// Now skip spaces and comma:
	while( vStrEnd[0] == ','
			|| vStrEnd[0] == ' '
			|| vStrEnd[0] == '\t' )
	{
		vStrEnd++;
	}
	
	// Convert second coordinate:
	outPos.top = strtol( vStrEnd, &vStrEnd, 10 );	// 10 means decimal number, not hex.
	
	// Now skip spaces and comma:
	while( vStrEnd[0] == ','
			|| vStrEnd[0] == ' '
			|| vStrEnd[0] == '\t' )
	{
		vStrEnd++;
	}
	
	// Convert third coordinate:
	outPos.right = strtol( vStrEnd, &vStrEnd, 10 );	// 10 means decimal number, not hex.
	
	// Now skip spaces and comma:
	while( vStrEnd[0] == ','
			|| vStrEnd[0] == ' '
			|| vStrEnd[0] == '\t' )
	{
		vStrEnd++;
	}
	
	// Convert fourth coordinate:
	outPos.bottom = strtol( vStrEnd, &vStrEnd, 10 );	// 10 means decimal number, not hex.
}


/* --------------------------------------------------------------------------------
	TextToColor:
		Convert a TextMunger to a color.
	
	TAKES:
		inText	-	the text data to convert.
	
	GIVES:
		outPos	-	The converted value.
	
	REVISIONS:
		2000-10-23	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkValue::TextToColor( const TextMunger& inText, LongColor& outPos )
{
	char		vTempString[256];
	char*		vStrEnd;
	
	// Convert first coordinate:
	inText.CopyToString( vTempString );
	outPos.red = strtol( vTempString, &vStrEnd, 10 );	// 10 means decimal number, not hex.
	
	// Now skip spaces and comma:
	while( vStrEnd[0] == ','
			|| vStrEnd[0] == ' '
			|| vStrEnd[0] == '\t' )
	{
		vStrEnd++;
	}
	
	// Convert second coordinate:
	outPos.green = strtol( vStrEnd, &vStrEnd, 10 );	// 10 means decimal number, not hex.
	
	// Now skip spaces and comma:
	while( vStrEnd[0] == ','
			|| vStrEnd[0] == ' '
			|| vStrEnd[0] == '\t' )
	{
		vStrEnd++;
	}
	
	// Convert third coordinate:
	outPos.blue = strtol( vStrEnd, &vStrEnd, 10 );	// 10 means decimal number, not hex.
}


/* --------------------------------------------------------------------------------
	TextToDouble:
		Convert a TextMunger to a double.
	
	TAKES:
		inText	-	the text data to convert.
	
	GIVES:
		double	-	The converted value.
	
	REVISIONS:
		1999-12-29	UK		Created.
   ----------------------------------------------------------------------------- */

double	TalkValue::TextToDouble( const TextMunger& inText )
{
	double		vReturnValue;
	char		vTempString[256];
	char*		vStrEnd;
	
	inText.CopyToString( vTempString );
	vReturnValue = strtod( vTempString, &vStrEnd );
	
	return vReturnValue;
}


/* --------------------------------------------------------------------------------
	TextToBool:
		Convert a TextMunger to a double.
	
	TAKES:
		inText	-	the text data to convert.
	
	GIVES:
		bool	-	The converted value.
	
	REVISIONS:
		1999-12-29	UK		Created.
   ----------------------------------------------------------------------------- */

bool	TalkValue::TextToBool( const TextMunger& inText )
{
	bool		vResultValue;
	
	// Don't sweat it, TextMunger::EqualsString() is case-insensitive!
	if( inText.EqualsString( "true", inText.GetLength() ) )
		vResultValue = true;
	else if( inText.EqualsString( "false", inText.GetLength() ) )
		vResultValue = false;
	else
		throw std::runtime_error( "Expected boolean here (TalkValue)." );
	
	return vResultValue;
}


/* --------------------------------------------------------------------------------
	LongToText:
		Convert a long to a TextMunger with equivalent text representation of the
		number.
	
	TAKES:
		vLong	-	the number to convert.
		outText	-	A TextMunger* into which a newly allocated text munger will
					be put.
	
	GIVES:
		outText	-	A TextMunger with the equivalent. Caller is responsible for
					disposing of this.
	
	REVISIONS:
		1999-12-29	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkValue::LongToText( long vLong, TextMunger* &outText )
{
	char		vTempString[256];
	
	sprintf( vTempString, "%ld", vLong );	// Write decimal number vLong into vTempString.
	outText = new TextMunger( vTempString );	// Now make a TextMunger with that string.
}


/* --------------------------------------------------------------------------------
	PointToText:
		Convert a LongPoint to a TextMunger with equivalent text representation of the
		coordinate.
	
	TAKES:
		pos		-	the coordinate to convert.
		outText	-	A TextMunger* into which a newly allocated text munger will
					be put.
	
	GIVES:
		outText	-	A TextMunger with the equivalent. Caller is responsible for
					disposing of this.
	
	REVISIONS:
		2000-10-22	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkValue::PointToText( const LongPoint& pos, TextMunger* &outText )
{
	char		vTempString[256];
	
	sprintf( vTempString, "%ld", pos.x );	// Write decimal number vLong into vTempString.
	outText = new TextMunger( vTempString );	// Now make a TextMunger with that string.
	
	outText->SetOffset( outText->GetLength() );
	sprintf( vTempString, ",%ld", pos.y );
	outText->InsertString( vTempString );
}


/* --------------------------------------------------------------------------------
	RectToText:
		Convert a LongRect to a TextMunger with equivalent text representation of the
		coordinates.
	
	TAKES:
		pos		-	the coordinates to convert.
		outText	-	A TextMunger* into which a newly allocated text munger will
					be put.
	
	GIVES:
		outText	-	A TextMunger with the equivalent. Caller is responsible for
					disposing of this.
	
	REVISIONS:
		2000-10-22	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkValue::RectToText( const LongRect& pos, TextMunger* &outText )
{
	char		vTempString[256];
	
	sprintf( vTempString, "%ld", pos.left );		// Write decimal number vLong into vTempString.
	outText = new TextMunger( vTempString );			// Now make a TextMunger with that string.
	
	outText->SetOffset( outText->GetLength() );
	sprintf( vTempString, ",%ld", pos.top );
	outText->InsertString( vTempString );
	
	sprintf( vTempString, ",%ld", pos.right );
	outText->InsertString( vTempString );
	
	sprintf( vTempString, ",%ld", pos.bottom );
	outText->InsertString( vTempString );
}


/* --------------------------------------------------------------------------------
	ColorToText:
		Convert a LongColor to a TextMunger with equivalent text representation of
		the coordinates.
	
	TAKES:
		pos		-	the color values to convert.
		outText	-	A TextMunger* into which a newly allocated text munger will
					be put.
	
	GIVES:
		outText	-	A TextMunger with the equivalent. Caller is responsible for
					disposing of this.
	
	REVISIONS:
		2000-10-22	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkValue::ColorToText( const LongColor& pos, TextMunger* &outText )
{
	char		vTempString[256];
	
	sprintf( vTempString, "%d", pos.red );		// Write decimal number vLong into vTempString.
	outText = new TextMunger( vTempString );		// Now make a TextMunger with that string.
	
	outText->SetOffset( outText->GetLength() );
	sprintf( vTempString, ",%d", pos.green );
	outText->InsertString( vTempString );
	
	sprintf( vTempString, ",%d", pos.blue );
	outText->InsertString( vTempString );
}


/* --------------------------------------------------------------------------------
	DoubleToText:
		Convert a double to a TextMunger with equivalent text representation of the
		number.
	
	TAKES:
		vDouble	-	the number to convert.
		outText	-	A TextMunger* into which a newly allocated text munger will
					be put.
	
	GIVES:
		outText	-	A TextMunger with the equivalent. Caller is responsible for
					disposing of this.
	
	REVISIONS:
		2002-04-09	UK		Changed to remove trailing zeroes.
		1999-12-29	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkValue::DoubleToText( double vDouble, TextMunger* &outText )
{
	char		vTempString[256];
	int			x = 0;
	
	sprintf( vTempString, "%f", vDouble );		// Write double vDouble into vTempString.
	
	// Kill trailing zeroes:
	x = strlen(vTempString);
	while( x > 1 && vTempString[--x] == '0' )
		vTempString[x] = 0;
	if( x > 0 && vTempString[x] == '.' )
		vTempString[x] = 0;
	
	outText = new TextMunger( vTempString );		// Make TextMunger with that string.
}


/* --------------------------------------------------------------------------------
	LongToText:
		Convert a boolean to a TextMunger with equivalent text representation.
	
	TAKES:
		vBoolean -	the boolean to convert.
		outText	-	A TextMunger* into which a newly allocated text munger will
					be put.
	
	GIVES:
		outText	-	A TextMunger with the equivalent. Caller is responsible for
					disposing of this.
	
	REVISIONS:
		1999-12-29	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkValue::BoolToText( bool vBoolean, TextMunger* &outText )
{
	outText = new TextMunger( vBoolean ? "true" : "false" );
}


/* --------------------------------------------------------------------------------
	DoubleToLong:
		Convert a double to a long. This only works if the double is actually
		an integer. Else this'll throw an exception.
	
	TAKES:
		vDouble -	the double to convert.
	
	GIVES:
		long	-	the double as an integer.
	
	REVISIONS:
		2002-01-14	UK		Added check for loss of precision.
		2002-01-01	UK		Added typecast to shut up compiler warnings.
		1999-12-29	UK		Created.
   ----------------------------------------------------------------------------- */

long	TalkValue::DoubleToLong( double vDouble )
{
	long		vReturnValue;
	
	vReturnValue = (long) vDouble;
	
	if( vDouble != vReturnValue )
		throw std::runtime_error( "Expected integer here." );
	
	return vReturnValue;
}


/* --------------------------------------------------------------------------------
	CopyToText:
		Convert the value passed to a TextMunger and return that in outMunger.
		If the provided type is already a TextMunger, this just returns that.
	
	TAKES:
		inValue 		-	the value to convert.
		providedType	-	The type of the value to convert.
	
	GIVES:
		outMunger		-	This pointer is set to a TextMunger with the text
							data.
	
	REVISIONS:
		2000-10-22	UK		Added Point and Rect.
		1999-12-29	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkValue::CopyToText( ValueStorage inValue, unsigned long providedType,
										TextMunger* &outMunger )
{
	switch( providedType )
	{
		case VALUE_TYPE_BOOL:
			BoolToText( inValue.boolType, outMunger );	// Convert boolean to text.
			break;
		
		case VALUE_TYPE_LONG:
			LongToText( inValue.longType, outMunger );	// Convert long to text.
			break;
		
		case VALUE_TYPE_DOUBLE:
			DoubleToText( inValue.doubleType, outMunger );	// Convert double to text.
			break;
		
		case VALUE_TYPE_TEXT:
			outMunger = inValue.textType;	// Is already text.
			break;
		
		case VALUE_TYPE_POINT:
			PointToText( inValue.pointType, outMunger );	// Point --> Text.
			break;
			
		case VALUE_TYPE_RECT:
			RectToText( inValue.rectType, outMunger );		// Rect --> Text.
			break;
			
		default:
			throw std::runtime_error( "Unsupported type for string conversion (TalkValue)." );
	}
}


/* --------------------------------------------------------------------------------
	GetTextTypesAvailable:
		Examine some text and return what data types it may be converted to.
	
	TAKES:
		vText	-	A text munger with the text you want to convert.
	
	GIVES:
		unsigned long	-	Bits set for each data type the string can be
							converted to.
	
	REVISIONS:
		2001-05-03	UK		Moved here from TalkVarValue::GetAvailableTypes().
   ----------------------------------------------------------------------------- */

unsigned long		TalkValue::GetTextTypesAvailable( const TextMunger& vText )
{
	unsigned long	vFlags = VALUE_TYPE_TEXT;
			
	if( vText.GetLength() == 0 )
		return vFlags;	// Empty strings can *only* be strings.
	
	// If it's TRUE or FALSE, this string can be converted to a boolean:
	if( vText.EqualsString( "true", vText.GetLength() ) )
		vFlags |= VALUE_TYPE_BOOL;
	else if( vText.EqualsString( "false", vText.GetLength() ) )
		vFlags |= VALUE_TYPE_BOOL;
	else	// Check whether this would convert to a long or double:
	{
		bool		vIsALong = true;
		bool		vIsADouble = true;
		bool		vAlreadyHadPoint = false;
		short		vCommaCount = 0;
		TextMunger	vScannableText( vText );
        
		vScannableText.SetOffset(0);
		if( vScannableText.Peek<char>() == '-' )	// Have sign?
			vScannableText.AdvanceOffset(1);	// Skip, this doesn't need to be a number.
		
		while( (vIsALong || vIsADouble)
				&& vScannableText.GetOffset() < vScannableText.GetLength() )
		{
			switch( vScannableText.Read<char>() )
			{
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					break;
				
				case ',':
					vCommaCount++;
					break;
				
				case '.':
					if( !vAlreadyHadPoint )	// This is the only period in here.
					{
						vAlreadyHadPoint = true;	// Make sure we detect excess periods.
						vIsALong = false;			// Only floats can have a period.
						break;
					}
					else	// Two periods??? Not a number!
						;	// Drop through!!!
				
				default:	// Invalid character in numbers, exit.
					vIsALong = false;
					vIsADouble = false;
			}
		}
		
		if( vIsALong
			&& !vAlreadyHadPoint )
		{
			switch( vCommaCount )
			{
				case 1:
					vFlags |= VALUE_TYPE_POINT;
					break;
				
				case 2:
					vFlags |= VALUE_TYPE_COLOR;
					break;
				
				case 3:
					vFlags |= VALUE_TYPE_RECT;
					break;
			}
		}
		
		if( vIsALong && vCommaCount == 0 )
			vFlags |= VALUE_TYPE_LONG;
		
		if( vIsADouble && vCommaCount == 0 )
			vFlags |= VALUE_TYPE_DOUBLE;
	}
	
	return vFlags;
}







