/*	***************************************************************************

	PROJECT:	TalkVarValue
	
	FILE:		TalkVarValue.cpp
	
	PURPOSE:	A value of a variable.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-11-28	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkVarValue.h"
#include	<cstdlib>
//#include	"missingproperty_error.h"
#include	"strcase.h"


#pragma mark [Globals]

/* --------------------------------------------------------------------------------
	Globals/Static Variables:
   ----------------------------------------------------------------------------- */

// Put these li'l critters here.


#pragma mark -
#pragma mark [Class Methods]

/* --------------------------------------------------------------------------------
	* DESTRUCTOR:
	
	REVISIONS:
		1999-11-28	UK		Created.
   ----------------------------------------------------------------------------- */

TalkVarValue::~TalkVarValue()
{
	if( mNativeType == VALUE_TYPE_TEXT )
	{
		delete mValue.textType;
		/* We don't delete valueType if it's set, as this just means we're a
			reference to a value that belongs to someone else. Same goes for
			entityType. */
	}
	else
		KillList();
}


/* --------------------------------------------------------------------------------
	KillList:
		If this value is an array, kill all of its values.
		
	REVISIONS:
		2000-10-23	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::KillList()
{
	TalkValueList::iterator		itty;
	
	// Walk over the list and copy its entries to us:
	for( itty = mList.begin(); itty != mList.end(); itty++ )
		(*itty).second->Release();
	
	mList.clear();
}


/* --------------------------------------------------------------------------------
	GetValue:
		Retrieve the value we keep as a certain type.
		
		Note that this follows references and retrieves the value of an entity
		if, and only if, the kind of value requested is not an entity.
	
	TAKES:
		desiredType	-	The data type to return.
	
	GIVES:
		returnValue	-	In this the value is returned. All of this data is
						allocated by TalkVarValue for the caller, and from then on
						belongs to the caller. TalkVarValue will not dispose of any
						returned data.
	
	REVISIONS:
		2000-11-24	UK		Made this work properly for entities.
		1999-11-28	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::GetValue( ValueStorage &returnValue, unsigned long desiredType,
								std::size_t startOffs, std::size_t endOffs ) const
{
	std::size_t		vSubStartOffs,
                	vSubEndOffs;
    
    if( !mSubIsValid )
	{
		vSubStartOffs = startOffs;
		vSubEndOffs = endOffs;
	}
    else
    {
        vSubStartOffs = mSubStartOffs;
        vSubEndOffs = mSubEndOffs;
    }
	
	switch( mNativeType )
	{
		case VALUE_TYPE_VALUE:
			mValue.valueType->GetValue( returnValue, desiredType, vSubStartOffs, vSubEndOffs );
			return;
		
		case VALUE_TYPE_ENTITY:
			if( desiredType != VALUE_TYPE_ENTITY )
			{
				mValue.entityType->GetContents()->GetValue( returnValue, desiredType,
															vSubStartOffs, vSubEndOffs );
				return;
			}
	}
	
	if( vSubEndOffs != 0 )	// Chunk expression!
	{
		GetChunk( returnValue, desiredType, vSubStartOffs, vSubEndOffs );
		return;
	}
	
	switch( desiredType )
	{
		case VALUE_TYPE_LONG:
			switch( mNativeType )
			{
				case VALUE_TYPE_TEXT:
					returnValue.longType = TextToLong( (*mValue.textType) );
					break;
				
				case VALUE_TYPE_LONG:
					returnValue.longType = mValue.longType;
					break;
				
				case VALUE_TYPE_DOUBLE:
					returnValue.longType = DoubleToLong( mValue.doubleType );
					break;
				
				default:
					throw std::runtime_error( "Can't convert to integer (TalkVarValue)." );
					break;
			}
			break;
		
		case VALUE_TYPE_DOUBLE:
			switch( mNativeType )
			{
				case VALUE_TYPE_TEXT:
					returnValue.doubleType = TextToDouble( (*mValue.textType) );
					break;
				
				case VALUE_TYPE_LONG:
					returnValue.doubleType = mValue.longType;
					break;
				
				case VALUE_TYPE_DOUBLE:
					returnValue.doubleType = mValue.doubleType;
					break;
				
				default:
					throw std::runtime_error( "Can't convert to double (TalkVarValue)." );
					break;
			}
			break;
		
		case VALUE_TYPE_TEXT:
			switch( mNativeType )
			{
				case VALUE_TYPE_DOUBLE:
					DoubleToText( mValue.doubleType, returnValue.textType );
					break;
				
				case VALUE_TYPE_LONG:
					LongToText( mValue.longType, returnValue.textType );
					break;
				
				case VALUE_TYPE_BOOL:
					BoolToText( mValue.boolType, returnValue.textType );
					break;
				
				case VALUE_TYPE_TEXT:
					returnValue.textType = new TextMunger( (*mValue.textType) );
					break;
				
				case VALUE_TYPE_POINT:
					PointToText( mValue.pointType, returnValue.textType );
					break;
				
				case VALUE_TYPE_COLOR:
					ColorToText( mValue.colorType, returnValue.textType );
					break;
				
				case VALUE_TYPE_RECT:
					RectToText( mValue.rectType, returnValue.textType );
					break;
				
				case VALUE_TYPE_LIST:
					if( mList.size() == 0 )	// Empty list?
					{
						returnValue.textType = new TextMunger("");
						break;
					}
					// Else drop through!
				
				default:
					throw std::runtime_error( "Can't convert to text (TalkVarValue)." );
					break;
			}
			break;
		
		case VALUE_TYPE_BOOL:
			switch( mNativeType )
			{
				case VALUE_TYPE_BOOL:
					returnValue.boolType = mValue.boolType;
					break;
				
				case VALUE_TYPE_TEXT:
					returnValue.boolType = TextToBool( (*mValue.textType) );
					break;
				
				default:
					throw std::runtime_error( "Can't convert to boolean (TalkVarValue)." );
					break;
			}
			break;
		
		case VALUE_TYPE_POINT:
			switch( mNativeType )
			{
				case VALUE_TYPE_POINT:
					returnValue.pointType = mValue.pointType;
					break;
				
				case VALUE_TYPE_TEXT:
					TextToPoint( (*mValue.textType), returnValue.pointType );
					break;
				
				default:
					throw std::runtime_error( "Can't convert to point (TalkVarValue)." );
					break;
			}
			break;
		
		case VALUE_TYPE_RECT:
			switch( mNativeType )
			{
				case VALUE_TYPE_RECT:
					returnValue.rectType = mValue.rectType;
					break;
				
				case VALUE_TYPE_TEXT:
					TextToRect( (*mValue.textType), returnValue.rectType );
					break;
				
				default:
					throw std::runtime_error( "Can't convert to rect (TalkVarValue)." );
					break;
			}
			break;
		
		case VALUE_TYPE_COLOR:
			switch( mNativeType )
			{
				case VALUE_TYPE_COLOR:
					returnValue.colorType = mValue.colorType;
					break;
				
				case VALUE_TYPE_TEXT:
					TextToColor( (*mValue.textType), returnValue.colorType );
					break;
				
				default:
					throw std::runtime_error( "Can't convert to color (TalkVarValue)." );
					break;
			}
			break;
			
		case VALUE_TYPE_VALUE:
			throw std::runtime_error( "Can't get value type." );
			break;
		
		case VALUE_TYPE_ENTITY:
			switch( mNativeType )
			{
				case VALUE_TYPE_ENTITY:
					returnValue.entityType = mValue.entityType;
					break;
				
				default:
					throw std::runtime_error( "Can't convert to entity (TalkVarValue)." );
					break;
			}
			break;
				
		default:
			throw std::runtime_error( "Caught unknown kind of value request (TalkVarValue)." );
	}
}


/* --------------------------------------------------------------------------------
	GetChunk:
		Get a chunk out of this value.
	
	TAKES:
		desiredType	-	The data type to return.
	
	GIVES:
		returnValue	-	In this the value is returned. All of this data is
						allocated by TalkVarValue for the caller, and from then on
						belongs to the caller. TalkVarValue will not dispose of any
						returned data.
	
	REVISIONS:
		1999-12-29	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::GetChunk( ValueStorage &returnValue, unsigned long desiredType,
								std::size_t startOffs, std::size_t endOffs ) const
{
	TextMunger*		vText;
	
	if( startOffs >= endOffs )	// Empty chunk!
		vText = new TextMunger( "" );
	else
	{
		CopyToText( mValue, mNativeType, vText );	// Convert our value to text.
		if( mNativeType == VALUE_TYPE_TEXT )
			vText = new TextMunger( (*vText), startOffs, endOffs -startOffs );	// We need a copy since we change it.
		else
		{
			if( (vText->GetLength() -endOffs) > 0 )
			{
				vText->SetOffset( endOffs );
				vText->DeleteData( vText->GetLength() -endOffs );	// Delete after chunk end.
			}
			if( startOffs > 0 )
			{
				vText->SetOffset( 0 );
				vText->DeleteData( startOffs );
			}
		}
	}
	
	switch( desiredType )
	{
		case VALUE_TYPE_LONG:
			returnValue.longType = TextToLong( (*vText) );
			delete vText;
			break;
		
		case VALUE_TYPE_DOUBLE:
			returnValue.doubleType = TextToDouble( (*vText) );
			delete vText;
			break;
		
		case VALUE_TYPE_TEXT:
			returnValue.textType = vText;	// Caller now takes care of vText, no need to dispose.
			break;
		
		case VALUE_TYPE_BOOL:
			returnValue.boolType = TextToBool( (*vText) );
			delete vText;
			break;
		
		case VALUE_TYPE_RECT:
			TextToRect( (*vText), returnValue.rectType );
			delete vText;
			break;
		
		case VALUE_TYPE_COLOR:
			TextToColor( (*vText), returnValue.colorType );
			delete vText;
			break;
		
		case VALUE_TYPE_POINT:
			TextToPoint( (*vText), returnValue.pointType );
			delete vText;
			break;
		
		default:
			delete vText;
			throw std::runtime_error( "Can't convert chunk to this type (TalkVarValue)." );
	}
}


/* --------------------------------------------------------------------------------
	SetValue:
		Change our value to one of specified type and value.
	
	TAKES:
		providedType	-	The data type passed.
		inValue			-	The value to assign to us. We copy what's passed in,
							the caller retains ownership of the values passed in.
	
	GIVES:
		-
	
	REVISIONS:
		2000-20-23	UK		Made this clear list value after assignment to prevent
							it from overriding the assigned value.
		1999-12-29	UK		Added chunk support.
		1999-11-28	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::SetValue( ValueStorage inValue, unsigned long providedType,
								bool followRefs, std::size_t startOffs, std::size_t endOffs )
{
	ValueStorage		vOldValues = mValue;	// So we can dispose of this after setting.
	unsigned long		vOldType = mNativeType;
	
	// Make sure our range is applied even if we're a ref.
	if( !mSubIsValid )	// This doesn't have a range, try to pass on what was passed in:
	{
		mSubStartOffs = startOffs;
		mSubEndOffs = endOffs;
	}
	
	if( followRefs )
	{
		switch( mNativeType )	// We're just a ref.
		{
			case VALUE_TYPE_VALUE:
				mValue.valueType->SetValue( inValue, providedType, followRefs, mSubStartOffs, mSubEndOffs );	// Set value we point to.
				return;
			
			case VALUE_TYPE_ENTITY:
				mValue.entityType->GetContents()->SetValue( inValue, providedType, followRefs, mSubStartOffs, mSubEndOffs );	// Set value of object.
				return;
		}
	}
	
	/* If chunk addressing was used to set this, we convert both the input
		value and this variable's value to strings and then perform the chunk
		replacement on that. Then we fake as if the user had passed this value
		in for assignment, by changing providedType to VALUE_TYPE_TEXT and
		making inValue point at the chunk-replaced TextMunger. */
	if( mSubEndOffs != 0
		&& followRefs )	// But if we're replacing a ref, don't do chunking!
	{
		TextMunger*		newValue;
		
		CopyToText( inValue, providedType, newValue );			// Get value to insert as a TextMunger.
		CopyToText( mValue, mNativeType, inValue.textType );	// Get our value as a TextMunger.
		
		// Actually insert input value into ours:
		inValue.textType->SetOffset( mSubStartOffs );
		inValue.textType->DeleteData( mSubEndOffs -mSubStartOffs );	// Kill previous data at that offset.
		inValue.textType->InsertMunger( (*newValue) );	// Write new munger into place of prevoius data.
		
		if( providedType != VALUE_TYPE_TEXT )	// We did conversion, Text Munger in newValue is our responsibility!
			delete newValue;	// Get rid of input value's text munger.
		
		providedType = VALUE_TYPE_TEXT;	// Re-label inValue to be of type text.
		vOldType = VALUE_TYPE_TEXT;		// Make sure it is disposed when we're done.
		vOldValues = inValue;			//  same as above.
	}
	else if( !followRefs )
		mSubIsValid = false;
	
	// Set new values:
	switch( providedType )
	{
		case VALUE_TYPE_VALUE:
		case VALUE_TYPE_BOOL:
		case VALUE_TYPE_LONG:
		case VALUE_TYPE_DOUBLE:
		case VALUE_TYPE_POINT:
		case VALUE_TYPE_COLOR:
		case VALUE_TYPE_RECT:
		case VALUE_TYPE_ENTITY:
			mValue = inValue;
			mNativeType = providedType;
			break;
		
		case VALUE_TYPE_TEXT:
			try {
				mValue.textType = new TextMunger( (*inValue.textType) );
				if( !mValue.textType )
					throw std::runtime_error( "Couldn't allocate text storage (TalkVarValue)." );
				mNativeType = providedType;
			}
			catch( std::exception& error )
			{
				mValue = vOldValues;
				
				throw;
			}
			break;
		
		default:
			throw std::runtime_error( "Can't accept this type (TalkVarValue)." );
	}
	
	// Clean up:
	KillList();	/* Make sure our list is gone in case we had one.
					This needn't care about killing the newly assigned value, since
					this function can't get a value of type list. */
	
	switch( vOldType )
	{
		case VALUE_TYPE_BOOL:
		case VALUE_TYPE_LONG:
		case VALUE_TYPE_DOUBLE:
		case VALUE_TYPE_VALUE:
		case VALUE_TYPE_RECT:
		case VALUE_TYPE_COLOR:
		case VALUE_TYPE_POINT:
		case VALUE_TYPE_ENTITY:
		case VALUE_TYPE_LIST:	// Already killed above.
			break;
		
		case VALUE_TYPE_TEXT:
			delete vOldValues.textType;	// This also kills the chunk-value (see above).
			break;
		
		default:
			throw std::runtime_error( "Can't dispose of this type (TalkVarValue)." );
	}
}


/* --------------------------------------------------------------------------------
	GetAvailableTypes:
		Return the types this value can provide. This is a pretty fast means of
		preflight-checking whether a value can be used and should be preferred to
		catching exceptions thrown when trying to GetValue().
	
	TAKES:
		-
	
	GIVES:
		unsigned long	-	A 32-bit flags value with the bit of each data type
							set which this can return.
	
	REVISIONS:
		1999-11-28	UK		Created.
   ----------------------------------------------------------------------------- */

unsigned long	TalkVarValue::GetAvailableTypes() const
{
	switch( mNativeType )
	{
		case VALUE_TYPE_VALUE:
			return mValue.valueType->GetAvailableTypes();
		
		case VALUE_TYPE_ENTITY:
			return mValue.entityType->GetContents()->GetAvailableTypes();
	}
	
	unsigned long		vFlags = 0;
	
	switch( mNativeType )
	{
		case VALUE_TYPE_TEXT:
			/*vFlags |= VALUE_TYPE_TEXT;
			
			if( mValue.textType->GetLength() == 0 )
				break;	// Empty strings can *only* be strings.
			
			// If it's TRUE or FALSE, this string can be converted to a boolean:
			if( mValue.textType->EqualsString( "true", mValue.textType->GetLength() ) )
				vFlags |= VALUE_TYPE_BOOL;
			else if( mValue.textType->EqualsString( "false", mValue.textType->GetLength() ) )
				vFlags |= VALUE_TYPE_BOOL;
			else	// Check whether this would convert to a long or double:
			{
				bool		vIsALong = true;
				bool		vIsADouble = true;
				bool		vAlreadyHadPoint = false;
				short		vCommaCount = 0;
				
				mValue.textType->SetOffset(0);
				if( mValue.textType->Peek<char>() == '-' )	// Have sign?
					mValue.textType->AdvanceOffset(1);	// Skip, this doesn't have to be a number.
				
				while( (vIsALong || vIsADouble)
						&& mValue.textType->GetOffset() < mValue.textType->GetLength() )
				{
					switch( mValue.textType->Read<char>() )
					{
						case '0':
						case '1':
						case '2':
						case '3':
						case '4':
						case '5':
						case '6':
						case '7':
						case '8':
						case '9':
							break;
						
						case ',':
							vCommaCount++;
							break;
						
						case '.':
							if( !vAlreadyHadPoint )	// This is the only period in here.
							{
								vAlreadyHadPoint = true;	// Make sure we detect excess periods.
								vIsALong = false;			// Only floats can have a period.
								break;
							}
							else	// Two periods??? Not a number!
								;	// Drop through!!!
						
						default:	// Invalid character in numbers, exit.
							vIsALong = false;
							vIsADouble = false;
					}
				}
				
				if( vIsALong
					&& !vAlreadyHadPoint )
				{
					switch( vCommaCount )
					{
						case 1:
							vFlags |= VALUE_TYPE_POINT;
							break;
						
						case 2:
							vFlags |= VALUE_TYPE_COLOR;
							break;
						
						case 3:
							vFlags |= VALUE_TYPE_RECT;
							break;
					}
				}
				
				if( vIsALong && vCommaCount == 0 )
					vFlags |= VALUE_TYPE_LONG;
				
				if( vIsADouble && vCommaCount == 0 )
					vFlags |= VALUE_TYPE_DOUBLE;
			}*/
			vFlags |= GetTextTypesAvailable( *mValue.textType );
			break;
		
		case VALUE_TYPE_LONG:
			vFlags |= VALUE_TYPE_TEXT | VALUE_TYPE_LONG | VALUE_TYPE_DOUBLE;
			break;
		
		case VALUE_TYPE_DOUBLE:
			vFlags |= VALUE_TYPE_TEXT | VALUE_TYPE_DOUBLE;
			break;
		
		case VALUE_TYPE_BOOL:
			vFlags |= VALUE_TYPE_TEXT | VALUE_TYPE_BOOL;
			break;
		
		case VALUE_TYPE_POINT:
			vFlags |= VALUE_TYPE_TEXT | VALUE_TYPE_POINT;
			break;
		
		case VALUE_TYPE_COLOR:
			vFlags |= VALUE_TYPE_TEXT | VALUE_TYPE_COLOR;
			break;
		
		case VALUE_TYPE_RECT:
			vFlags |= VALUE_TYPE_TEXT | VALUE_TYPE_RECT;
			break;
		
		case VALUE_TYPE_LIST:
			vFlags |= VALUE_TYPE_LIST;
			break;
		
		default:
			vFlags = 0;
	}
	
	return vFlags;
}


/* --------------------------------------------------------------------------------
	CopyValueTo:
		Set another value to our value, using our native type. If we are a
		reference of type VALUE_TYPE_VALUE, this copies the value of the value
		we point to.
	
	TAKES:
		vDestValue	-	The value to export our value to.
	
	GIVES:
		-
	
	REVISIONS:
		2000-20-23	UK		Made this clear list value after assignment to prevent
							it from overriding the assigned value.
		1999-11-28	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::CopyValueTo( TalkValue& vDestValue, bool followSrcRefs,
									bool followDstRefs ) const
{
	if( mNativeType != VALUE_TYPE_LIST )
	{
		/* If we're a reference but not a chunk and supposed to follow source
			references (source is this) just call our value's CopyValueTo() method: */
		if( !mSubIsValid && followSrcRefs )
		{
			switch( mNativeType )
			{
				case VALUE_TYPE_VALUE:
					mValue.valueType->CopyValueTo( vDestValue, followSrcRefs, followDstRefs );
					return;
				
				case VALUE_TYPE_ENTITY:
					mValue.entityType->GetContents()->CopyValueTo( vDestValue, followSrcRefs, followDstRefs );
					return;
			}
		}
		
		ValueStorage		tmpValue = mValue;
		unsigned long		vType = mNativeType;
		
		// If we're supposed to follow source refs:
		if( mSubIsValid && followSrcRefs )
		{
			GetValue( tmpValue, VALUE_TYPE_TEXT );
			vType = VALUE_TYPE_TEXT;
		}
		
		vDestValue.SetValue( tmpValue, vType, followDstRefs );
		
		if( mSubIsValid && followSrcRefs )
			delete tmpValue.textType;	// Get rid of our text copy.
		else if( mSubIsValid && !followDstRefs )
			vDestValue.SetChunkRange( mSubStartOffs, mSubEndOffs );
		
		vDestValue.KillList();
	}
	else 
		vDestValue.SetList( mList );
}


/* --------------------------------------------------------------------------------
	GetListEntry:
		Get an array entry from this item.
	
	TAKES:
		returnValue	-	This will be set to a pointer to the array value.
		index		-	The array index of the item to look up.
	
	GIVES:
		-
	
	REVISIONS:
		1999-12-13	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::GetListEntry( TalkValue* &returnValue, const TextMunger& index )
{
	switch( mNativeType )
	{
		case VALUE_TYPE_VALUE:
			mValue.valueType->GetListEntry( returnValue, index );
			return;
		
		case VALUE_TYPE_ENTITY:
			mValue.entityType->GetContents()->GetListEntry( returnValue, index );
			return;
	}
	
	TalkValueList::iterator		vIterator;
	
	vIterator = mList.find( index );
	if( vIterator == mList.end() )
	{
		returnValue = new TalkVarValue( (long) 0 );
		returnValue->Retain();
		mList[index] = returnValue;
		
		if( mNativeType == VALUE_TYPE_TEXT )
			delete mValue.textType;
		mNativeType = VALUE_TYPE_LIST;
	}
	else
		returnValue = (*vIterator).second;
}


/* --------------------------------------------------------------------------------
	SetList:
		Make this array a copy of the array passed in. Called by CopyValueTo().
	
	TAKES:
		vList		-	The list to copy in.
	
	GIVES:
		-
	
	REVISIONS:
        2001-07-28	UK		Changed to use const_iterator to satisfy PB.
		2000-10-21	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::SetList( const TalkValueList& vList )
{
	TalkValueList::const_iterator		itty;
	#if JOKER_DEBUG
	  static TextMunger					spaces;
	  
	  spaces.SetOffset(0);
	  spaces.Insert<char>('\t');
	#endif
	
	// First, get rid of any data besides the list we might contain:
	if( mNativeType == VALUE_TYPE_TEXT )
		delete mValue.textType;
	
	/* WARNING: We can't kill mList here since we might be copying sub-entries of
		mList list over it. Instead, we build the list in vTempList and only after
		it has been recreated in there, we kill mList and overwrite it with
		the contents of vTempList. */
	
	mNativeType = VALUE_TYPE_LIST;
	TalkValueList		vTempList;
	
	// Now walk over the list and copy its entries to us:
	for( itty = vList.begin(); itty != vList.end(); itty++ )
	{
		TalkVarValue*		vNewVal = NULL;
		
		vNewVal = new TalkVarValue(0L);
		vNewVal->Retain();
		
		try {
			(*itty).second->CopyValueTo( *vNewVal, true, true );	// Calls SetList if it's also a list.
			vTempList[(*itty).first] = vNewVal;
			
			#if JOKER_DEBUG
			  std::cout << spaces.String() << "Copying element: ";
			  std::cout << (*itty).first.String() << std::endl;
			#endif
		}
		catch( std::exception& err )
		{
			vNewVal->Release();
			throw;
		}
	}
	
	KillList();	// Get rid of old list entries.
	mList = vTempList;
	
	#if JOKER_DEBUG
	  spaces.SetOffset(0);
	  spaces.DeleteData(1);
	#endif
}


/* --------------------------------------------------------------------------------
	CopyListTo:
		Make an array a copy of this array.
	
	TAKES:
		vList		-	The list to copy to. This list must be empty, or you must
						have made sure that both lists do not share any entries
						with the same index, or this will leak.
	
	GIVES:
		-
	
	REVISIONS:
        2001-07-28	UK		Changed to use const_iterator to satisfy PB.
		2000-10-21	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::CopyListTo( TalkValueList& vList ) const
{
	TalkValueList::const_iterator		itty;
	
	// First, check whether this is a list at all:
	if( mNativeType != VALUE_TYPE_LIST )
		throw std::runtime_error("Expected array here.");
	
	// Now walk over the list and copy its entries to us:
	for( itty = mList.begin(); itty != mList.end(); itty++ )
	{
		TalkVarValue*		vNewVal = NULL;
		
		vNewVal = new TalkVarValue(0L);
		vNewVal->Retain();
		
		try {
			(*itty).second->CopyValueTo( *vNewVal, true, true );	// Calls SetList if it's also a list.
			vList[(*itty).first] = vNewVal;
		}
		catch( std::exception& err )
		{
			vNewVal->Release();
			throw;
		}
	}
}


/* --------------------------------------------------------------------------------
	GetKeys:
		Return all keys of a value in a text munger, one per line.
	
	TAKES:
		vList		-	The list to copy in.
	
	GIVES:
		-
	
	REVISIONS:
        2001-07-28	UK		Changed to use const_iterator to satisfy PB.
		2000-10-21	UK		Created.
   ----------------------------------------------------------------------------- */

TextMunger*	TalkVarValue::GetKeys() const
{
	TalkValueList::const_iterator		itty;
	TextMunger*							vKeysList;
	
	vKeysList = new TextMunger;
	
	try
	{
		// Now walk over the list and append entry keys to our text list:
		for( itty = mList.begin(); itty != mList.end(); )
		{
			vKeysList->InsertMunger( (*itty).first );
			itty++;	// Move on.
			if( itty != mList.end() )	// If it's last, don't append return.
				vKeysList->Insert<char>( '\n' );
		}
	}
	catch( std::exception& err )
	{
		delete vKeysList;
		throw;
	}
	
	return vKeysList;
}


/* --------------------------------------------------------------------------------
	SetChunkRange:
		Restrict this value to a range.
	
	TAKES:
		offsStart	-	A start offset.
		offsEnd		-	An end offset.
	
	GIVES:
		-
	
	REVISIONS:
		1999-12-30	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::SetChunkRange( std::size_t offsStart, std::size_t offsEnd )
{
	mSubStartOffs = offsStart;
	mSubEndOffs = offsEnd;
	mSubIsValid = true;
}


/* --------------------------------------------------------------------------------
	GetPropertyValue:
		Retrieve a property of this value resp. the value or entity this value
		points to.
	
	TAKES:
		pName		-	Name of the property to get.
		outValue	-	The value of this value is set to the property value
						without following any references.
	
	GIVES:
		-
	
	REVISIONS:
		2000-11-14	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::GetPropertyValue( const TextMunger& pName, TalkValue& outValue ) const
{
	ValueStorage	vs;
	
	switch( mNativeType )
	{
		case VALUE_TYPE_VALUE:
			mValue.valueType->GetPropertyValue( pName, outValue );
			return;
		
		case VALUE_TYPE_ENTITY:
			mValue.entityType->GetPropertyValue( pName, outValue );
			return;
	}
	
	if( pName == "properties" )
	{
		TextMunger	props("properties");
		vs.textType = &props;
		outValue.SetValue( vs, VALUE_TYPE_TEXT );
	}
	else {}
		// throw missingproperty_error( "Variables don't have this property.", pName );
}


/* --------------------------------------------------------------------------------
	SetPropertyValue:
		Change a property of this value resp. of the value or entity this value
		points to.
	
	TAKES:
		pName		-	Name of the property to set.
		outValue	-	The value to assign to the property.
	
	GIVES:
		-
	
	REVISIONS:
		2000-11-14	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::SetPropertyValue( const TextMunger& pName, const TalkValue& inValue )
{
	switch( mNativeType )
	{
		case VALUE_TYPE_VALUE:
			mValue.valueType->SetPropertyValue( pName, inValue );
			return;
		
		case VALUE_TYPE_ENTITY:
			mValue.entityType->SetPropertyValue( pName, inValue );
			return;
	}
	
	if( pName == "properties" )
		throw std::runtime_error( "Can't set the \"properties\" property." );
	else {}
		// throw std::missingproperty_error( "No such property.", pName );
}


/* --------------------------------------------------------------------------------
	GetContents:
		Get the contents of this entity. Should never be called for a TalkVarValue
		but we need to define this since this method in TalkEntity is pure
		virtual.
	
	TAKES:
		-
	
	GIVES:
		TalkValue*	-	The value is owned by this value. DO NOT DISPOSE!
	
	REVISIONS:
		2001-07-28	UK		Changed to return entity's contents instead of this
                            object to satisfy PB's constness requirements.
		2000-11-14	UK		Created.
   ----------------------------------------------------------------------------- */

TalkValue*	TalkVarValue::GetContents() const
{
	if( mNativeType == VALUE_TYPE_ENTITY )
		return( mValue.entityType->GetContents() );
	else
		throw std::logic_error( "Man are we screwed up! A value was just addressed as an entity!" );
	
	return NULL;
}


/* --------------------------------------------------------------------------------
	Delete:
		Delete a chunk of this variable.
	
	TAKES:
		startOffs	-	The start offset or zero if it's entire value.
		endOffs		-	The end offset or zero if it's entire value.
	
	GIVES:
		-
	
	REVISIONS:
		2001-07-15	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkVarValue::Delete( std::size_t startOffs, std::size_t endOffs )
{
	// Make sure our range is applied even if we're a ref.
	if( !mSubIsValid )	// This doesn't have a range, try to pass on what was passed in:
	{
		mSubStartOffs = startOffs;
		mSubEndOffs = endOffs;
	}
	
	if( mSubStartOffs == 0 && mSubEndOffs == 0 )
	{
		switch( mNativeType )
		{
			case VALUE_TYPE_TEXT:
				mValue.textType->SetLength(0);
				break;
			
			case VALUE_TYPE_LONG:
			case VALUE_TYPE_DOUBLE:
			case VALUE_TYPE_RECT:
			case VALUE_TYPE_POINT:
			case VALUE_TYPE_COLOR:
				mValue.textType = new TextMunger("");
				mNativeType = VALUE_TYPE_TEXT;
				break;
			
			case VALUE_TYPE_VALUE:
				mValue.valueType->Delete();
				break;
			
			case VALUE_TYPE_ENTITY:
				mValue.entityType->Delete();
				break;
		}
	}
	else
	{
		switch( mNativeType )
		{
			case VALUE_TYPE_TEXT:
				mValue.textType->SetOffset(mSubStartOffs);
				mValue.textType->DeleteData(mSubEndOffs -mSubStartOffs);
				break;
			
			case VALUE_TYPE_LONG:
			case VALUE_TYPE_DOUBLE:
			case VALUE_TYPE_RECT:
			case VALUE_TYPE_POINT:
			case VALUE_TYPE_COLOR:
			{
				ValueStorage	newValue;
				
				CopyToText( mValue, mNativeType, newValue.textType );
				newValue.textType->SetOffset(mSubStartOffs);
				newValue.textType->DeleteData(mSubEndOffs -mSubStartOffs);
				mValue.textType = newValue.textType;
				mNativeType = VALUE_TYPE_TEXT;
				break;
			}
			
			case VALUE_TYPE_VALUE:
				mValue.valueType->Delete(mSubStartOffs,mSubEndOffs);
				break;
			
			case VALUE_TYPE_ENTITY:
				mValue.entityType->GetContents()->Delete(mSubStartOffs,mSubEndOffs);
				break;
		}
	}
}



