/*	

	PROJECT:	TextMunger
	
	FILE:		TextMunger.cpp
	
	PURPOSE:	Text manipulation class for Joker.
	
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		2001-05-03	UK		Improved code by inserting some calls to assert().
		1999-02-14	UK		Created.
				
	 */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TextMunger.h"
#include	<cctype>
#include	<stdexcept>


#pragma mark [Globals]

/* --------------------------------------------------------------------------------
	Globals/Static Variables:
   ----------------------------------------------------------------------------- */

static XSmartPointer			gTextAsString( NULL );	// Temp storage used for returning C strings.
// This is a global since else const functions would complain.


#pragma mark -
#pragma mark [Init/Kill]

/* --------------------------------------------------------------------------------
	 CONSTRUCTOR:
	
	REVISIONS:
		1999-12-05	UK	Added ref counting.
		1999-02-14	UK	Created.
   ----------------------------------------------------------------------------- */

TextMunger::TextMunger()
{
	mText = new RefCountedText();
	mText->RequestReadAccess();
	mLength = 0;	// String is empty.
	mOffset = 0;
	mFixedOffset = 0;
	mHash = 0;
	mDontOwnBuffer = false;
}


/* --------------------------------------------------------------------------------
	 COPY CONSTRUCTOR:
		This just registers with the RefCountedText object of the other text
		munger. They share the data until one of them requests write access. Only
		then data is actually copied.
		
		Optionally, you can request this munger to only share a substring with
		the munger it's modeled after. This is good for read access, but for
		write access you will get a copy of the whole string, not just the
		substring.
	
	TAKES:
		inMunger -		Text munger instance to copy data from.
	
	REVISIONS:
		1999-12-05	UK	Added ref counting and support for substrings.
		1999-02-14	UK	Created.
   ----------------------------------------------------------------------------- */

TextMunger::TextMunger( const TextMunger& inMunger, std::size_t subStrOffs, std::size_t subStrLen )
{
	mText = inMunger.mText;
	mText->RequestReadAccess();
	if( subStrLen == 0
		|| (inMunger.mLength -subStrOffs) < subStrLen )	// What about too big sub strings???
		mLength = inMunger.mLength -subStrOffs;
	else
		mLength = subStrLen;
	mOffset = inMunger.mOffset -subStrOffs;
	mFixedOffset = inMunger.mFixedOffset;
	mFixedOffset += subStrOffs;
	mDontOwnBuffer = false;
	
	// If we're en exact duplicate, we can take over the hash:
	if( subStrLen == 0
		&& subStrOffs == 0 )
		mHash = inMunger.mHash;
	else	// Else mark it for recalculation.
		mHash = 0;
}


/* --------------------------------------------------------------------------------
	 CONSTRUCTOR( char* ):
		Create this object's data based on a C string.
		
		Use of strings is discouraged as it makes it impossible to use NULL chars.
		Use texts instead, or best, another TextMunger object.
	
	TAKES:
		inString -		C string to copy data from.
	
	REVISIONS:
		1999-12-05	UK	Added ref counting.
		1999-02-14	UK	Created.
   ----------------------------------------------------------------------------- */

TextMunger::TextMunger( const char* inString )
{
	mText = new RefCountedText();
	mText->RequestReadAccess();
	mLength = 0;	// String is empty.
	mOffset = 0;
	mFixedOffset = 0;
	mHash = 0;
	mDontOwnBuffer = false;
	
	CopyFromString( inString );
}


/* --------------------------------------------------------------------------------
	 CONSTRUCTOR( char*, size_t ):
		Create this object's data based on a text.
	
	TAKES:
		inText -		Pointer to data to copy into this munger.
		inLength -		Length of data to copy from inText.
		dontCopy -		TRUE to have the munger use the buffer passed in instead
						of copying it. But be careful, you mustn't change the size
						of such a buffer.
	
	REVISIONS:
        2001-07-28	UK	Split into two constructors to satisfy PB's constness
                        requirements.
		2001-05-03	UK	Added support for not copying and directly using a pointer
						owned by the caller.
		1999-12-05	UK	Added ref counting.
		1999-02-14	UK	Created.
   ----------------------------------------------------------------------------- */

TextMunger::TextMunger( const char* inText, const std::size_t inLength )
{
	mText = new RefCountedText();	// Gives us a NULL pointer as mText->mText.
	mText->RequestReadAccess();
	mLength = 0;	// String is empty.
	mOffset = 0;
	mFixedOffset = 0;
	mHash = 0;
	mDontOwnBuffer = false;
	
    CopyFromText( inText, inLength );
}

TextMunger::TextMunger( char* inText, const std::size_t inLength, bool dontCopy )
{
    if( dontCopy )
    {
        mText = NULL;
        mDontOwnBuffer = false;
        TakeOverText( inText, inLength );
    }
    else
    {
        mText = new RefCountedText();	// Gives us a NULL pointer as mText->mText.
        mText->RequestReadAccess();
        mLength = 0;	// String is empty.
        mOffset = 0;
        mFixedOffset = 0;
        mHash = 0;
        mDontOwnBuffer = dontCopy;
        
        CopyFromText( inText, inLength );
    }
}


void	TextMunger::TakeOverText( char* inText, const std::size_t inLength )
{
    if( mDontOwnBuffer )
	{
		mText->mText = NULL;
		mText->mLength = mLength = 0;
	}
	if( mText )
	{
		mText->GiveUpAccess();
		mText = NULL;
	}
    
    mText = new RefCountedText();	// Gives us a NULL pointer as mText->mText.
	mText->RequestReadAccess();
	mLength = 0;	// String is empty.
	mOffset = 0;
	mFixedOffset = 0;
	mHash = 0;
	mDontOwnBuffer = true;
	
    mText->mText = inText;	// No need to dispose as the pointer isn't allocated for empty strings.
    mText->mLength = mLength = inLength;
}


/* --------------------------------------------------------------------------------
	 DESTRUCTOR:
		Free the memory we allocated.
	
	REVISIONS:
		1999-12-05	UK	Added ref counting.
		1999-02-14	UK	Created.
   ----------------------------------------------------------------------------- */

TextMunger::~TextMunger()
{
	if( mDontOwnBuffer )
	{
		mText->mText = NULL;
		mText->mLength = mLength = 0;
	}
	if( mText )
	{
		mText->GiveUpAccess();
		mText = NULL;
	}
}


#pragma mark [Class Methods]

/* --------------------------------------------------------------------------------
	CopyToString:
		Copy this object's text to the string passed. You must ensure that outStr
		is large enough to hold this object's text. If inTerminate is TRUE, we
		also append a NULL character and replace all NULL characters inside this
		with a space.
		
		Use of strings is discouraged as it makes it impossible to use NULL chars.
		Use texts instead, or best, another TextMunger object.
	
	TAKES:
		outStr -	C string pointer that is large enough to hold this munger's
					data.
	
	GIVES:
		outStr -	The C string this points to is set to our data + zero byte
					terminator.
	
	REVISIONS:
		1999-12-05	UK	Added ref counting.
		1999-02-14	UK	Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::CopyToString( char* outStr ) const
{
	unsigned long		x = 0;
	
	while( x < mLength )
	{
		if( mText->mText[x +mFixedOffset] == 0 )
			outStr[x] = ' ';
		else
			outStr[x] = mText->mText[x +mFixedOffset];
		
		x++;
	}
	
	outStr[x] = 0;	// Terminate the string.
}


/* --------------------------------------------------------------------------------
	CopyToText:
		Copy this object's text to a text.
	
	TAKES:
		outText -		Pointer to storage large enough to hold our data.
	
	GIVES:
		outText -		The data of this pointer is set to our data.
	
	REVISIONS:
        2001-07-28	UK	Added cast to satisfy PB's stricter promotion rules.
		1999-12-05	UK	Added ref counting.
		1999-02-14	UK	Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::CopyToText( char* outText ) const
{
	/* Note: The syntax is BACKWARDS than with BlockMove(), it is
	 			memmove( dest, source, length )! */
	if( mLength > 0 )
		memmove( outText, (char*) mText->mText +mFixedOffset, mLength );
}


/* --------------------------------------------------------------------------------
	CopyToTextMunger:
		Copy our text to another text munger. We do this by calling its
		CopyFromText member function with our data members.

		This resets the read/write offset of ioMunger to 0.
	
	TAKES:
		ioMunger - The munger to copy to.
	
	GIVES:
		ioMunger - This munger's data is set to a copy of our data.
	
	REVISIONS:
		2002-01-14	UK	Added resetting of offset.
		1999-12-05	UK	Added ref counting.
		1999-02-14	UK	Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::CopyToTextMunger( TextMunger& ioMunger ) const
{
	ioMunger.CopyFromText( GetTextPtr(), mLength );
	
	ioMunger.mOffset = 0;	// Reset offset so we don't get any errors if offset would lie outside the new text.
}


/* --------------------------------------------------------------------------------
	GetTextPtr:
		Return the pointer to the text in this text munger.
	
	TAKES:
		-
	
	GIVES:
		char*	-	Pointer to the actual data used by this. Do not modify!!!
	
	REVISIONS:
        2001-07-28	UK	Added cast to satisfy PB's stricter promotion rules.
		2000-11-07	UK	Created.
   ----------------------------------------------------------------------------- */

char*	TextMunger::GetTextPtr() const
{
	return( (char*) mText->mText +mFixedOffset );
}


/* --------------------------------------------------------------------------------
	CopyFromText:
		Copy a text's contents into this object. If an error occurs during this
		this object remains unchanged and propagates the exception. This uses
		memmove, it's safe to pass this object's own data to this routine to get
		a subsection of the data.
		This call resets this object's read/write offset to 0.
	
	TAKES:
		inText -	Pointer to the data to set this munger's data to.
		inLength -	Amount of data to copy from inText.
	
	GIVES:
		-
	
	REVISIONS:
		2002-01-14	UK	Added resetting of offset.
		1999-12-05	UK	Added ref counting.
		1999-02-14	UK	Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::CopyFromText( const char* inText, const std::size_t inLength )
{
	//unsigned long		x = 0;
	char*				vTempText = NULL;
	
	if( mDontOwnBuffer )	throw std::runtime_error("Can't change data of a shared buffer.");
	
	mText = mText->RequestWriteAccess();	// Make sure we get our own copy if it's shared.
	
	if( mText->mText != 0L )				// Did we have storage before?
		vTempText = mText->mText;			// Remember it in case we fail.
	
	mText->mText = NULL;					// We need this to see if new worked but memmove failed.
	
	try {
		if( inLength > 0 )
		{
			mText->mText = (char*) malloc( inLength );	// Attempt to create new storage.
			if( mText->mText == 0L )
				throw std::bad_alloc();
		}
		mText->mLength = inLength;
		
		if( inLength > 0 )
			memmove( (char*) mText->mText +mFixedOffset, inText, inLength );	// Copy text to our storage.
		
		mLength = inLength;					// Update our data member.
	
		if( vTempText )						// We had storage before.
			free( vTempText );				// Get rid of it.
		
		mOffset = 0;	// Reset offset so we don't get any errors if offset would lie outside the new text.
	}
	catch( std::exception& inException )			// On error.
	{
		if( mText->mText != 0L )			// new succeeded, but memmove failed.
			free( mText->mText );
		
		mText->mText = vTempText;			// Reclaim old storage.
		
		throw inException;					// Propagate exception and jump out of this fcn.
	}
}


/* --------------------------------------------------------------------------------
	CopyFromString:
		Copy a string's contents into this object. If an error occurs during this
		this object remains unchanged and propagates the exception.
		This call resets this object's read/write mark to 0.
		
		Use of strings is discouraged as it makes it impossible to use NULL chars.
		Use texts instead, or best, another TextMunger object.
	
	TAKES:
		inText -	A C string with the text to set our data to.
	
	GIVES:
		-
	
	REVISIONS:
		2002-01-14	UK	Added resetting of offset.
		1999-02-14	UK	Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::CopyFromString( const char* inText )
{
	std::size_t			vLength = strlen( inText );
	//unsigned long		x = 0;
	char*				vTempText = NULL;
	
	if( mDontOwnBuffer )	throw std::runtime_error("Can't change data of a shared buffer.");
	
	mText = mText->RequestWriteAccess();	// Make sure we get a copy if it's shared.
	
	if( mText->mText != 0L )					// Did we have storage before?
		vTempText = mText->mText;				// Remember it in case we fail.
	
	mText->mText = NULL;					// We need this to see if new worked but memcpy failed.
	
	try {					
		if( vLength > 0 )
		{
			mText->mText = (char*) malloc( vLength );	// Attempt to create new storage.
			if( mText->mText == 0L )
				throw std::bad_alloc();
		}
		else
			mText->mText = NULL;	// Don't try to allocate for empty strings.
		
		mText->mLength = vLength;
		
		if( vLength > 0 )
			memcpy( mText->mText, inText, vLength );	// Copy text to our storage.
		
		mLength = vLength;					// Update our data member.
		mFixedOffset = 0;					// Now we don't have fixed offset anymore.
		
		if( vTempText )						// We had storage before.
			free( vTempText );				// Get rid of it.
		
		mOffset = 0;	// Reset offset so we don't get any errors if offset would lie outside the new text.
	}
	catch( std::exception& inException )			// On error.
	{
		if( mText->mText != 0L )			// malloc succeeded, but memcpy failed.
			free( mText->mText );
		
		mText->mText = vTempText;			// Reclaim old storage.
		
		throw inException;					// Propagate exception and jump out of this fcn.
	}
}


/* --------------------------------------------------------------------------------
	CopyFromTextMunger:
		Get text from another TextMunger. We do this by passing us to the other
		munger's CopyToTextMunger() member function.
		This call resets this object's read/write mark to 0.
	
	TAKES:
		inMunger -	The text munger we are to copy data from.
	
	GIVES:
		-
	
	REVISIONS:
		1999-02-14	UK		Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::CopyFromTextMunger( const TextMunger& inMunger )
{
	if( mDontOwnBuffer )	throw std::runtime_error("Can't change data of a shared buffer.");
	
	inMunger.CopyToTextMunger( (*this) );
}


/* --------------------------------------------------------------------------------
	EqualsString:
		Loop over a C-string and compare it to the data starting at the current
		offset for the specified length.
	
	TAKES:
		inStr		-	Pointer to string to compare to.
		inLength	-	Length of data to compare. If inStr isn't same length this
						returns false.
	
	GIVES:
		bool		-	true if equal, false if not.
	
	REVISIONS:
		1999-11-27	UK		Created.
   ----------------------------------------------------------------------------- */

bool	TextMunger::EqualsString( const char* inStr, const std::size_t inLength ) const
{
	//ASSERT_Range( mOffset +inLength <= mLength );	// Are we big enough?
	//ASSERT_Argument( inStr );
	
	std::size_t		x;
	
	for( x = 0; x < inLength; x++ )
	{
		if( tolower(inStr[x]) != tolower( (*(char*)(((long)mText->mText) +mOffset +mFixedOffset +x)) ) )
			return false;
	}
	
	return( inStr[x] == 0 );	// If string doesn't end here, it's longer than our data and thus not equal.
}


/* --------------------------------------------------------------------------------
	EqualsMunger:
		Compare a certain length of this text to a certain length of another
		munger. The substrings start at the respective offsets.
	
	TAKES:
		inStr		-	Munger to compare to.
		inLength	-	Length of data to compare.
	
	GIVES:
		bool		-	true if equal, false if not.
	
	REVISIONS:
		1999-11-27	UK		Created.
   ----------------------------------------------------------------------------- */

bool	TextMunger::EqualsMunger( const TextMunger& inMun, const std::size_t inLength ) const
{
	//ASSERT_Range( mOffset +inLength <= mLength );	// Are we big enough?
	
	std::size_t		x;
	
	// If both have hashes, exit if not even hashes match:
	if( mHash != 0
		&& inMun.mHash != 0
		&& mHash != inMun.mHash )
		return false;
	
	for( x = 0; x < inLength; x++ )
	{
		if( tolower( (*(char*)(((long)inMun.mText->mText) +inMun.mOffset +inMun.mFixedOffset +x)) )
			!= tolower( (*(char*)(((long)mText->mText) +mOffset +mFixedOffset +x)) ) )
			return false;
	}
	
	return true;
}


/* --------------------------------------------------------------------------------
	operator char*:
		This allows passing TextMungers as strings to C functions. All TextMungers
		use the same static variable as temporary memory, so be careful with this.
		Also, be sure to call TextMunger::FreeTempMemory() after using this so
		the temporary memory won't stay around too long. It's also a good idea to
		call this from your main event loop.
	
	TAKES:
		-
	
	GIVES:
		char* -		A pointer to a C string with a copy of our data.
	
	REVISIONS:
		1999-02-14	UK		Created.
   ----------------------------------------------------------------------------- */

TextMunger::operator char*()
{
	return( String() );
}


/* --------------------------------------------------------------------------------
	String:
		The actual beef behind operator char*. Note that the storage is
		deallocated and newly allocated (and re-used) on every call to this. That
		is, your pointer becomes invalid if you call this again after retrieving
		it.
	
	TAKES:
		-
	
	GIVES:
		char* -		A pointer to a C string containing a copy of this string's
					data.
	
	REVISIONS:
		1999-02-14	UK		Created.
   ----------------------------------------------------------------------------- */

char*	TextMunger::String() const
{
	// gTextAsString is a static variable shared by all objects of this type.
	
	if( gTextAsString != NULL )
	{
		free( gTextAsString );				// Get rid of any previous temp memory.
		gTextAsString = NULL;
	}
	
	gTextAsString = (char*) malloc( mLength +1 );	// Get new temp memory of proper size.
	CopyToString( gTextAsString );					// Copy our contents to the string.
	
	return gTextAsString;
}


/* --------------------------------------------------------------------------------
	PeekData:
		Read the specified amount of data at the current offset. This *doesn't*
		change the offset!
		
		This uses memcpy as a guard against CPUs which don't like odd offsets, e.g.
		the original 68000 CPU. This should make x-platform easier.
	
	TAKES:
		ioData -	Pointer to storage large enough to hold the desired amount of
					data.
		inLength -	Amount of data to copy (in bytes).
	
	GIVES:
		ioData -	This pointer's data is set to the data starting at the current
					mark and ending after inLength bytes have been copied.
	
	REVISIONS:
        2001-07-28	UK	Added cast to satisfy PB's stricter promotion rules.
		1999-02-16	UK	Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::PeekData( void* ioData, std::size_t inLength )
{
	//ASSERT_Range( mOffset +inLength <= mLength );	// Are we still within our bounds?
	//ASSERT_Argument( ioData );
	memcpy( ioData, (char*) mText->mText +mOffset +mFixedOffset, inLength );
}


/* --------------------------------------------------------------------------------
	SmashData:
		Write the specified amount of data at the current offset. This *doesn't*
		change the offset!
		
		This uses memmove as a guard against CPUs which don't like odd offsets,
		e.g. the original 68000 CPU. This should make x-platform easier.
	
	TAKES:
		inData -	Pointer to data to write.
		inLength -	Length of data to write from inData.
	
	GIVES:
		-
	
	REVISIONS:
        2001-07-28	UK	Added cast to satisfy PB's stricter promotion rules.
		1999-02-16	UK	Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::SmashData( const void* inData, const std::size_t inLength )
{
	//ASSERT_Range( mOffset +inLength <= mLength );	// Are we big enough?
	//ASSERT_Argument( inData );
	
	mText = mText->RequestWriteAccess();
	mHash = 0;
	memmove( (char*) mText->mText +mOffset +mFixedOffset, inData, inLength );
}


/* --------------------------------------------------------------------------------
	SmashMunger:
		Write the data from another munger to this munger.
	
	TAKES:
		inMunger	-	The munger whose data we are to insert.
	
	GIVES:
		-
	
	REVISIONS:
        2001-07-28	UK	Added cast to satisfy PB's stricter promotion rules.
		1999-02-16	UK	Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::SmashMunger( const TextMunger& inMunger )
{
	SmashData( (char*) inMunger.mText->mText +inMunger.mFixedOffset, inMunger.mLength );
}


/* --------------------------------------------------------------------------------
	operator=:
		Copy one TextMunger to the other. This just registers this munger with
		the other munger's RefCountedText object after unregistering from ours.
		This way, if none ever changes the data, both share one string. Copying
		isn't done until one of them requests write access.
	
	TAKES:
		inMunger	-	The text munger whose data we are to copy.
	
	GIVES:
		-
	
	REVISIONS:
		1999-12-05	UK	Created.
   ----------------------------------------------------------------------------- */

TextMunger&	TextMunger::operator= ( const TextMunger& inMunger )
{
	if( mDontOwnBuffer )	throw std::runtime_error("Can't change data of a shared buffer.");
	
	mText->GiveUpAccess();
	inMunger.mText->RequestReadAccess();
	mText = inMunger.mText;
	mLength = inMunger.mLength;
	mFixedOffset = inMunger.mFixedOffset;
	
	return (*this);
}


/* --------------------------------------------------------------------------------
	operator==:
		Compare two text mungers to each other. This ignores the offsets and is
		only satisfied if both mungers are the same length. It ignores differences
		in upper- and lowercase, though.
	
	TAKES:
		inMunger	-	The text munger to compare to this one.
	
	GIVES:
		-
	
	REVISIONS:
		1999-12-05	UK	Created.
   ----------------------------------------------------------------------------- */

bool	TextMunger::operator== ( const TextMunger& inMun ) const
{
	std::size_t		x;
	
	if( inMun.mLength != mLength )
		return false;
	
	if( mHash != 0
		&& inMun.mHash != 0
		&& mHash != inMun.mHash )
		return false;
	
	for( x = 0; x < mLength; x++ )
	{
		if( tolower( (*(char*)(((long)inMun.mText->mText) +inMun.mFixedOffset +x)) )
			!= tolower( (*(char*)(((long)mText->mText) +mFixedOffset +x)) ) )
			return false;
	}
	
	return true;
}


/* --------------------------------------------------------------------------------
	operator==:
		Compare a text munger and a C string to each other. This ignores the
		offsets and is only satisfied if both are the same length. It ignores
		differences in upper- and lowercase, though.
	
	TAKES:
		inStr	-	The C string to compare this munger to.
	
	GIVES:
		-
	
	REVISIONS:
		2001-12-13	UK	Created.
   ----------------------------------------------------------------------------- */

bool	TextMunger::operator== ( const char* inStr ) const
{
	return EqualsString( inStr, strlen(inStr) );
}


/* --------------------------------------------------------------------------------
	operator<:
		Compare two text mungers to each other. This ignores the offsets and is
		only satisfied if both mungers are the same length. It ignores differences
		in upper- and lowercase, though.
	
	TAKES:
		inMunger	-	The text munger to compare to this one.
	
	GIVES:
		-
	
	REVISIONS:
		1999-12-09	UK	Created.
   ----------------------------------------------------------------------------- */

bool	TextMunger::operator< ( const TextMunger& inMun ) const
{
	std::size_t		x,
					vLength;
	
	if( inMun.mLength > mLength )
		vLength = mLength;
	else
		vLength = inMun.mLength;
	
	for( x = 0; x < vLength; x++ )
	{
		char		inCh = tolower(*(char*)(((long)inMun.mText->mText) +inMun.mFixedOffset +x)),
					myCh = tolower(*(char*)(((long)mText->mText) +mFixedOffset +x));
		
		if( inCh > myCh )
			return true;
		else if( inCh < myCh )
			return false;
		// If they are the same, we proceed to next char.
	}
	
	return mLength < inMun.mLength;
}


/* --------------------------------------------------------------------------------
	operator>:
		Compare two text mungers to each other. This ignores the offsets and is
		only satisfied if both mungers are the same length. It ignores differences
		in upper- and lowercase, though.
	
	TAKES:
		inMunger	-	The text munger to compare to this one.
	
	GIVES:
		-
	
	REVISIONS:
		1999-12-09	UK	Created.
   ----------------------------------------------------------------------------- */

bool	TextMunger::operator> ( const TextMunger& inMun ) const
{
	std::size_t		x,
					vLength;
	
	if( inMun.mLength > mLength )
		vLength = mLength;
	else
		vLength = inMun.mLength;
	
	for( x = 0; x < vLength; x++ )
	{
		char		inCh = tolower(*(char*)(((long)inMun.mText->mText) +inMun.mFixedOffset +x)),
					myCh = tolower(*(char*)(((long)mText->mText) +mFixedOffset +x));
		
		if( inCh < myCh )
			return true;
		else if( inCh > myCh )
			return false;
		// If they are the same, we proceed to next char.
	}
	
	return mLength > inMun.mLength;
}


/* --------------------------------------------------------------------------------
	InsertData:
		Write the specified amount of data at the current offset, enlarging the
		storage if necessary and moving up data following the point before which
		we want to insert. This also advances the offset past the data written.
	
	TAKES:
		inData -	Pointer to data to insert.
		inLength -	Length of data to insert from inData.
	
	GIVES:
		-
	
	REVISIONS:
        2001-07-28	UK	Added cast to satisfy PB's stricter promotion rules.
		1999-02-16	UK	Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::InsertData( const void* inData, const std::size_t inLength )
{
	std::size_t		vRemainderLength = mLength -mOffset;
	char*			vNewData;
	
	if( mDontOwnBuffer )	throw std::runtime_error("Can't change data of a shared buffer.");
	
	if( inLength == 0 )
		return;
	
	if( mLength < mOffset )
		throw std::out_of_range("TextMunger::InsertData offset exceeds text size.");
	
	mText = mText->RequestWriteAccess();
	
	mHash = 0;
	
	if( mText->mText != NULL && mText->mLength > 0 )
	{
		vNewData = (char*) realloc( mText->mText, mLength +inLength +mFixedOffset );
		if( !vNewData )
			throw	std::bad_alloc();
		mText->mLength = mLength +inLength +mFixedOffset;
		mText->mText = vNewData;
	}
	else
	{
		mText->mText = (char*) malloc( inLength +mFixedOffset );
		if( mText->mText == 0L )
			throw	std::bad_alloc();
		mText->mLength = inLength +mFixedOffset;
	}
	
	//ASSERT_Argument( inData );
	
	if( vRemainderLength > 0 )
		memmove( (char*) mText->mText +mOffset +mFixedOffset +inLength,
                    (char*) mText->mText +mOffset +mFixedOffset, vRemainderLength );
	
	mLength += inLength;
	
	WriteData( inData, inLength );
}


/* --------------------------------------------------------------------------------
	SetLength( size_t len ):
		Resize this text munger.
	
	TAKES:
		inData -	Pointer to data to insert.
		inLength -	Length of data to insert from inData.
	
	GIVES:
		-
	
	REVISIONS:
		2000-11-10	UK	Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::SetLength( std::size_t len )
{
	char*		vNewData;
	
	if( mDontOwnBuffer )	throw std::runtime_error("Can't change data of a shared buffer.");
	
	mText = mText->RequestWriteAccess();
	
	mHash = 0;
	
	if( len == 0 )		// We want to get rid of all data?
	{
		if( mText->mText != NULL )
			free( mText->mText );
		mText->mText = NULL;
		mText->mLength = 0;
		mText->mText = NULL;
	}
	else if( mText->mText != NULL && mText->mLength > 0 )
	{
		vNewData = (char*) realloc( mText->mText, len +mFixedOffset );
		if( !vNewData )
			throw	std::bad_alloc();
		mText->mLength = len +mFixedOffset;
		mText->mText = vNewData;
	}
	else
	{
		mText->mText = (char*) malloc( len );
		if( mText->mText == 0L )
			throw	std::bad_alloc();
		mText->mLength = len;
	}
	
	mLength = len;
}


/* --------------------------------------------------------------------------------
	DeleteData:
		Delete the specified amount of data starting at the offset. This moves
		down any data behind the deleted range.
	
	TAKES:
		inLength -	Length of data to delete.
	
	GIVES:
		-
	
	REVISIONS:
        2001-07-28	UK	Added cast to satisfy PB's stricter promotion rules.
		1999-12-30	UK	Moved from InsertData() to avoid bug w/ unsigned values.
   ----------------------------------------------------------------------------- */

void	TextMunger::DeleteData( const std::size_t inLength )
{
	std::size_t		vRemainderLength = mLength -mOffset;
	char*			vNewData;
	
	if( mDontOwnBuffer )	throw std::runtime_error("Can't change data of a shared buffer.");
	
	if( inLength == 0 || mLength == 0 || mText->mText == NULL )
		return;
	
	mText = mText->RequestWriteAccess();
	
	mHash = 0;
	
	// We need to move data down before reallocating or it'll be gone.
	vRemainderLength -= inLength;
	
	if( vRemainderLength > 0 )
		memmove( (char*) mText->mText +mOffset +mFixedOffset,
                    (char*)mText->mText +mOffset +mFixedOffset +inLength,
                    vRemainderLength );
	
	if( mText->mText != 0L )
	{
		if( mLength -inLength == 0 )
		{
			free( mText->mText );
			mText->mText = NULL;
			mText->mLength = 0;
		}
		else
		{
			vNewData = (char*) realloc( mText->mText, mLength -inLength +mFixedOffset );
			if( !vNewData )
				throw	std::bad_alloc();
			mText->mLength = mLength -inLength +mFixedOffset;
			mText->mText = vNewData;
		}
	}
	
	mLength -= inLength;
}


/* --------------------------------------------------------------------------------
	InsertMunger:
		Write the data from another munger to this munger.
	
	TAKES:
		inMunger	-	The munger whose data we are to insert.
	
	GIVES:
		-
	
	REVISIONS:
		1999-02-16	UK	Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::InsertMunger( const TextMunger& inMunger )
{
	RefCountedText*		vTheText;
	
	if( mDontOwnBuffer )	throw std::runtime_error("Can't change data of a shared buffer.");
	
	if( inMunger.mLength == 0 || inMunger.mText->mText == NULL )
		return;
	
	// Make sure nobody changes the text behind our back:
	vTheText = inMunger.mText;
	vTheText->RequestReadAccess();
	
	// Actually insert the data:
	InsertData( (char*) (((long)inMunger.mText->mText) +inMunger.mFixedOffset),
				inMunger.mLength );
	
	vTheText->GiveUpAccess();	// Now we don't need it anymore.
}



/* --------------------------------------------------------------------------------
	FreeTempMemory:
		Free the memory this uses when converting this object to a C String. Call
		this when you need more RAM, TextMunger will automatically re-allocate it
		when needed.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		1999-02-15	UK		Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::FreeTempMemory()
{
	if( gTextAsString != 0L )
	{
		free( gTextAsString );
		gTextAsString = NULL;
	}
}


/* --------------------------------------------------------------------------------
	RecalcHash:
		Recalculate the hash for this munger's text.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		1999-12-12	UK		Created.
   ----------------------------------------------------------------------------- */

void	TextMunger::RecalcHash() const
{
	if( mHash == 0 )	// Hash invalid.
	{
		if( mText->mText == NULL || mLength == 0 )
			return;
		
		std::size_t		x;
		
		for( x = 0; x < mLength; x++ )
		{
			mHash += tolower( (*(char*)(((long)mText->mText) +mFixedOffset +x)) );
			if( mHash > 1000000 )
				mHash %= x;
		}
	}
}


/* --------------------------------------------------------------------------------
	RequestWriteAccess:
		Returns a pointer to a RefCountedText object. If you are the only user of
		this object, it's a pointer to this object, else it'll return a new object
		with its own copy of the data.
		
	WARNING:
		You must have requested read access before calling this.
	
	TAKES:
		-
	
	GIVES:
		RefCountedText*	-	a pointer to an object whose text you may modify.
	
	REVISIONS:
		2000-10-20	UK		Fixed bug that failed to set the length when copying.
		1999-12-15	UK		Created.
   ----------------------------------------------------------------------------- */

RefCountedText*	RefCountedText::RequestWriteAccess()
{
	if( mRefCount == 1 )
		return this;
	else
	{
		RefCountedText*		vRefCText;
		
		vRefCText = new RefCountedText();
		if( mLength > 0 && mText != NULL )
		{
			vRefCText->mText = (char*) malloc( mLength );
			if( vRefCText->mText == NULL )
				throw std::bad_alloc();
			memmove( vRefCText->mText, mText, mLength );
			vRefCText->mLength = mLength;
		}
		else
		{
			vRefCText->mText = NULL;
			vRefCText->mLength = 0;
		}
		vRefCText->mRefCount = 1;
		mRefCount --;
		
		return vRefCText;
	}
}







