/*	***************************************************************************

	PROJECT:	Joker
	
	FILE:		GenTalkInstructions.h
	
	PURPOSE:	Instructions in a script. This file contains dummies for
				the platform-specific implementations of some instructions.
		
	COPYRIGHT:	(C) Copyright 2000 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		2000-11-07	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkInstructions.h"
#include	"HyperTalk.h"
#include	"TalkVarValue.h"
#include	"JokerMain.h"
#include	<iostream>
#include	<stdexcept>


#pragma mark -
#pragma mark [Instruction Procs]

/* --------------------------------------------------------------------------------
	All instruction procedures take two parameters, an "HI" parameter that contains
	the instruction's parameters and destination for the result along with other
	info like a pointer to the object currently running, and an "s" parameter
	that points to the stack with the variables for this instruction.
   ----------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------------
	HyperReplyInstruction:
		This implements the "reply" command that can be used to return information
		from an appleEvent message. This is inherently Mac-specific, but maybe
		some hosts want to support this as a compatibility layer.
		
	Here's a quick run-down of what AppleEvents are:
		
		AppleEvents are a means of messaging between applications. An
		AppleEvent is simply a data structure containing bits of data. Just like
		in HyperTalk, this data may have a specific type, but some (limited)
		conversion between data types will be performed.
		
		Each bit of data in an AppleEvent has a unique (inside this event),
		four-character "keyword" that provides a clue as to what it is, along with
		the actual data type. For example, there is "----" which is the "direct
		object", which is used to send data along with simple events. More complex
		events have additional keywords.
		
		Also, an AppleEvent itself has a "class" and an "ID", which are also
		keywords. These identify the kind of request another application is making.
		
		For example, on the Mac, whenever another program or the operating system
		calls upon your application to open a file, it sends an event of class "aevt"
		and ID "odoc" ("open document"), the direct object ("----") of which
		contains a list of file paths to open, one per line. If it's not too much
		hassle, you might want to implement your HyperAppleEventInstruction and
		HyperRequestInstruction thus that whenever a file is opened upon request,
		an appleEvent message is sent, and the request command returns the list of
		files to open. You'll also want to make your application only try to open
		the files itself if the appleEvent message wasn't intercepted.
	
	HI's FIELDS:
		0		-	TRUE if this is "reply error", FALSE if just "reply".
		1		-	Expression to return.
		2		-	TRUE if "with" follows expression, FALSE if omitted.
		3		-	The AppleEvent keyword or an empty string if none specified.
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperReplyInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vIsReplyError,
						vString,
						vIsWithKeyword,
						vAEKeyword;
		
	hi.mParam[0].GetValue( vIsReplyError, s, VALUE_TYPE_BOOL );
	hi.mParam[1].GetValue( vString, s, VALUE_TYPE_TEXT );
	hi.mParam[2].GetValue( vIsWithKeyword, s, VALUE_TYPE_BOOL );
	hi.mParam[3].GetValue( vAEKeyword, s, VALUE_TYPE_TEXT );
	
	try
	{
		// First, check whether this is "reply error":
		if( vIsReplyError.boolType )	// This simply means we use reply keyword "errs":
		{
			vIsWithKeyword.boolType = true;
			vAEKeyword.textType->CopyFromString( "errs" );
		}
		
		// Now remember the result data the user gives us:
		switch( vAEKeyword.textType->GetLength() )
		{
			case 4:		// Proper type code.
				break;
			
			case 0:		// None passed, default to direct object, identified by '----'.
				vAEKeyword.textType->CopyFromString( "----" );
				break;
			
			default:
				throw runtime_error( "An AppleEvent keyword must be exactly 4 characters long." );
		}
		
		/* FIX ME! Now you'd probably compare vAEKeyword to your list of supported keywords
			And process the data appropriately. You could collect all reply values
			in a data structure which is kept with the ValueStack. If you do this,
			you can set the parameters that are passed in when sending the appleEvent message,
			and when it returns you have the reply data ready for sending back.
			
			E.g. a CGI AppleEvent returns its data (including HTML headers etc.) in
			the direct object of the reply to an event of class "WWW" and ID "sdoc". */
		
		delete vString.textType;
		delete vAEKeyword.textType;
	}
	catch( exception& err )
	{
		delete vString.textType;
		delete vAEKeyword.textType;
		throw;
	}
}

/* --------------------------------------------------------------------------------
	HyperAppleEventInstruction:
		This command traps the "appleEvent" message that is sent on Apple Events.
	
	HI's FIELDS:
		0		-	Event class
		1		-	Event ID
		2		-	Sender
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperAppleEventInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	/* FIX ME! You can notify the application here that nobody handled the apple event
		request, and have it handle the event by itself. Or you can simply leave this
		here as a dummy that prevents the error message. */
}