/*	***************************************************************************

	PROJECT:	Joker
	
	FILE:		generic/GenTalkURLValue.cpp
	
	PURPOSE:	Generic implementation of a URL.
				Use this as a template for ports to other platforms.
		
	COPYRIGHT:	(C) Copyright 2000 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	HELPFUL SNIPPET:
		The following code snippet might be useful for a Unix port:
	
		DIR *dirp;
        direct *dp;
        dirp = opendir((String)path);
        if (dirp == NULL) return list;
        while ((dp = readdir(dirp)) != NULL)
        {
                list.append( dp->d_name );
        }
        closedir(dirp);

	
	
	REVISIONS:
		2000-10-26	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkURLValue.h"
#include	<cstdlib>
#include	<dirent.h>
#include	"missingproperty_error.h"


#pragma mark [Globals]

/* --------------------------------------------------------------------------------
	Globals/Static Variables:
   ----------------------------------------------------------------------------- */

// Put these li'l critters here.


#pragma mark -
#pragma mark [Class Methods]

/* --------------------------------------------------------------------------------
	DownloadURLData:
		Fetch the data of the file pointed to by this URL and return it in a text
		munger.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2000-20-24	UK		Created.
   ----------------------------------------------------------------------------- */

TextMunger*	TalkURLValue::DownloadURLData() const
{
	TextMunger*		vResult;
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	size_t			vReadSize = 512;
	
	mTheURL.CopyToString( vURLString );
	if( strncmp( vURLString, "file:///", 8 ) == 0 )	// String starts with file protocol?
	{
		vURLString += 8;	// Make sure we skip protocol designator.
		
		FILE*		vFile;
		vResult = new TextMunger;
		
		vFile = fopen( vURLString, "r+" );
		if( vFile == NULL )
			throw runtime_error( "Couldn't open file of local URL." );
		
		try
		{
			while( vReadSize == 512 )	// If fread returns less, it means we have reached end of file.
			{
				vReadSize = fread( vURLStringData, sizeof(char), vReadSize, vFile );
				vResult->InsertData( vURLStringData, vReadSize );
			}
		}
		catch( exception& err )
		{
			fclose( vFile );
			throw;
		}
		
		fclose( vFile );
	}
	else
	{
		/* FIX ME! Insert OS-specific code for downloading remote URLs here. Stuff the
			TextMunger with the data in vResult (the caller of this function is
			responsible for disposing it, this function *mustn't* dispose of it).
			The URL you are to download is in the TextMunger mTheURL. */
		
		vResult = NULL;
		throw logic_error( "Downloading from remote URLs is not yet implemented on this platform." );
	}
  
	return vResult;
}


/* --------------------------------------------------------------------------------
	UploadURLData:
		Replace the data of the file pointed to by this URL through the data in
		the text munger passed.
	
	TAKES:
		vData	-	The text data to upload.
	
	GIVES:
		-
	
	REVISIONS:
		2000-20-24	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::UploadURLData( TextMunger* vData ) const
{
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	size_t			vReadSize = 512;
	
	mTheURL.CopyToString( vURLString );
	if( strncmp( vURLString, "file:///", 8 ) == 0 )	// String starts with file protocol?
	{
		vURLString += 8;	// Make sure we skip protocol designator.
		
		FILE*		vFile;
		
		vFile = fopen( vURLString, "w+" );
		if( vFile == NULL )
			throw runtime_error( "Couldn't open file of local URL." );
		
		try
		{
			size_t		vStillToWrite = vData->GetLength();
			
			vData->SetOffset(0);	// Make sure we write all.
			
			// Now write in 512-byte blocks until we run out of data:
			while( vStillToWrite > 0 )
			{
				if( vStillToWrite > 512 )
					vReadSize = 512;
				else
					vReadSize = vStillToWrite;
				vStillToWrite -= vReadSize;
				vData->ReadData( vURLStringData, vReadSize );
				
				if( fwrite( vURLStringData, sizeof(char), vReadSize, vFile ) < vReadSize )
					throw runtime_error( "Couldn't write all data to file of local URL." );
			}
		}
		catch( exception& err )
		{
			fclose( vFile );
			throw;
		}
		
		fclose( vFile );
	}
	else
	{
		/* FIX ME! Insert OS-specific code for uploading to remote URLs here.
			The URL you are to upload to is in the TextMunger mTheURL. The data
			to be uploaded is in the TextMunger vData. */
		
		throw logic_error( "Uploads to remote URLs are not yet implemented on this platform." );
	}
}


/* --------------------------------------------------------------------------------
	GetPropertyValue:
		Determine the values of any properties of this URL. This implements the
		"files" property by listing the files in the specified directory using
		MoreFiles or calling upon URL access to list the files in a remote folder.
	
	TAKES:
		pName		-	Name of the property to get.
		outValue	-	A TalkValue* in which we store the property value.
	
	GIVES:
		-
	
	REVISIONS:
		2001-01-28	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::GetPropertyValue( const TextMunger& pName, TalkValue& outValue ) const
{
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	//OSErr			err;
	ValueStorage	vOutData;
	
	if( pName == "files" )
	{
		mTheURL.CopyToString( vURLString );
		if( strncmp( vURLString, "file://", 7 ) == 0 )	// String starts with file protocol?
		{
			vURLString += 7;	// Make sure we skip protocol designator.
			
            vOutData.textType = new TextMunger();
            TexMunAPtr		vKiller( vOutData.textType );
            
            DIR*				vDirectory = NULL;
            struct dirent*		vCurrFile = NULL;
            unsigned long		x = 0;
            
            // Scan folder:
            vDirectory = opendir( vURLString );
            
            while( (vCurrFile = readdir(vDirectory)) != NULL )
            {
                if( ++x == 1 || x == 2 )
                    continue;	// Skip the two first items, which are "." and "..".
                
                vOutData.textType->SetOffset( vOutData.textType->GetLength() );
                vOutData.textType->Insert<char>( '\n' );
                vOutData.textType->InsertData( vCurrFile->d_name, strlen(vCurrFile->d_name) );
            }
            
            closedir( vDirectory );
            
            // Get rid of leading return:
            if( vOutData.textType->GetLength() > 1 )
            {
                vOutData.textType->SetOffset(0);
                vOutData.textType->DeleteData(1);
            }
            outValue.SetValue( vOutData, VALUE_TYPE_TEXT );
		}
		else	// Remote URL?
		{
			throw std::runtime_error( "Remote URLs not supported on this computer." );
		}
	}
	else
		throw missingproperty_error( "URLs don't have this property.", pName );
}


/* --------------------------------------------------------------------------------
	KillURLFile:
		Delete the file/folder referenced by this URL.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2001-07-15	UK		Created.
   ----------------------------------------------------------------------------- */

void	TalkURLValue::KillURLFile()
{
	char			vURLStringData[512];	// FIX ME! Size limit on file paths.
	char*			vURLString = vURLStringData;
	//size_t			vReadSize = 512;
	short			x;
	
	mTheURL.CopyToString( vURLString );
	if( strncmp( vURLString, "file://", 7 ) == 0 )	// String starts with file protocol?
	{
		vURLString += 7;	// Make sure we skip protocol designator, but skip one less as we'll use first byte as length byte.
		
		if( remove( vURLString ) != 0 )
			throw std::runtime_error("Couldn't delete file.");
	}
	else
		throw std::runtime_error("Deleting remote files is not supported.");
}
