/* ================================================================================
	PROJECT:	JokerLib
	FILE:		JokerLib.cpp
	PURPOSE:	Source file for JokerLib, the shared library that implements an
				xTalk interpreter.
	COPYRIGHT:	JokerLib and Joker are (c) Copyright 1999-2002 by M. Uli Kusterer,
				all rights reserved.
   ============================================================================= */

#if __MWERKS__
#ifndef __CONSOLE__
#include <console.h>
#endif
#endif

#if defined(__GNUC__) && (defined(__APPLE_CPP__) || defined(__APPLE_CC__))
#include <Carbon/Carbon.h>
#else
#include <Files.h>
#include <Resources.h>
#include <Dialogs.h>
#include <CodeFragments.h>
#endif
#include <stdarg.h>
#include "HyperTalk.h"
#include "TalkTokenize.h"
#include "TextMunger.h"
#include "TalkVarValue.h"
#include "TalkExtendableEntity.h"

/* Cause JokerLib.h to declare #pragma export instead of #pragma import: */
#define		GENERATING_JOKER_LIB		1

#include	"JokerLib.h"


JokerOutputProcPtr				gOutputFunction;
JokerHandlerCallProcPtr			gHandlerCallFunction;
JokerSpendTimeProcPtr			gSpendTimeFunction;
FSSpec							gLibraryFileSpec;
short							gLibResFile;
char							gErrorMessageText[256];


/* The two macros below are used to add a try block conveniently around all
	API calls to catch any exceptions before they propagate past the bounds
	of the JokerLib fragment. JOKER_ERROR_START clears our error string
	storage, while JOKER_ERROR_END catches any exceptions and assigns the
	description string of the exception to our error global, from where it
	can be retrieved using JokerGetErrorString(). */
#define		JOKER_ERROR_START	gErrorMessageText[0] = 0;	\
								try {
#define		JOKER_ERROR_END		}	\
								catch( exception& e )	\
								{	\
									strcpy( gErrorMessageText, e.what() );	\
								}	\
								catch( ... )	\
								{	\
									strcpy( gErrorMessageText, "Unknown Exception." );	\
								}


void	HyperClientCmdInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec );
void	HyperClientObjDescInstruction( struct HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec );

#if __MWERKS__
#ifdef __cplusplus
extern "C" {
#endif
extern pascal OSErr __initialize( CFragInitBlock *theInitBlock );
#ifdef __cplusplus
}
#endif
#endif /*__MWERKS__*/


class	JokerLibEntity : public TalkExtendableEntity
{
public:
	JokerEntityProcPtr		mProc;		// ProcPtr to client-defined callback function.
	long					mRefCon;	// RefCon for use of client.
	
public:
	JokerLibEntity( JokerEntityProcPtr p ) : TalkExtendableEntity(), mProc(p) { mRefCon = 0; };
	
	virtual void		GetPropertyValue( const TextMunger& pName, TalkValue& outValue ) const
	{
		TalkVarValue	vName( pName );
		TalkVarValue	vValue( 0L );
				
		if( (*mProc)( (JokerEntityRef) this, kJokerEntityGetPropEvent, (JokerValueRef) &vName, (JokerValueRef) &vValue ) == NULL )
			TalkExtendableEntity::GetPropertyValue( pName, outValue );
		else
			vValue.CopyValueTo( outValue, false, true );	// Follow dest refs.
	};
	virtual void		SetPropertyValue( const TextMunger& pName, const TalkValue& inValue )
	{
		TalkVarValue	vName( pName );
		TalkVarValue	vValue( 0L );
		ValueStorage	vStor;
		
		vStor.valueType = const_cast<TalkValue*>(&inValue);
		vValue.SetValue( vStor, VALUE_TYPE_VALUE );
		
		if( (*mProc)( (JokerEntityRef) this, kJokerEntitySetPropEvent, (JokerValueRef) &vName, (JokerValueRef) &vValue ) == NULL )
			TalkExtendableEntity::SetPropertyValue( pName, inValue );
	};
	virtual TalkValue*	GetContents() const
	{
		return (TalkVarValue*) (*mProc)( (JokerEntityRef) this, kJokerEntityGetContentsEvent, NULL, NULL );
	};
	virtual void		Delete()
	{
		(*mProc)( (JokerEntityRef) this, kJokerEntityDeleteEvent, NULL, NULL );
	};
};

//#ifdef __cplusplus
//extern "C" {
//#endif

#if __MWERKS__
/*
 *	The following four functions provide the UI for the console package.
 *	Users wishing to replace SIOUX with their own console package need
 *	only provide the four functions below in a library.
 */

/*
 *	extern short InstallConsole(short fd);
 *
 *	Installs the Console package, this function will be called right
 *	before any read or write to one of the standard streams.
 *
 *	short fd:		The stream which we are reading/writing to/from.
 *	returns short:	0 no error occurred, anything else error.
 */

short InstallConsole(short fd)
{
#pragma unused (fd)

	return 0;
}

/*
 *	extern void RemoveConsole(void);
 *
 *	Removes the console package.  It is called after all other streams
 *	are closed and exit functions (installed by either atexit or _atexit)
 *	have been called.  Since there is no way to recover from an error,
 *	this function doesn't need to return any.
 */

void RemoveConsole(void)
{
}

/*
 *	extern long WriteCharsToConsole(char *buffer, long n);
 *
 *	Writes a stream of output to the Console window.  This function is
 *	called by write.
 *
 *	char *buffer:	Pointer to the buffer to be written.
 *	long n:			The length of the buffer to be written.
 *	returns short:	Actual number of characters written to the stream,
 *					-1 if an error occurred.
 */

long WriteCharsToConsole(char *buffer, long n)
{
	if( gOutputFunction != NULL )	// User set an output function?
		return (*gOutputFunction)( buffer, n );	// Call it.
	else
		return 0;	// Just dummy away.
}

/*
 *	extern long ReadCharsFromConsole(char *buffer, long n);
 *
 *	Reads from the Console into a buffer.  This function is called by
 *	read.
 *
 *	char *buffer:	Pointer to the buffer which will recieve the input.
 *	long n:			The maximum amount of characters to be read (size of
 *					buffer).
 *	returns short:	Actual number of characters read from the stream,
 *					-1 if an error occurred.
 */

long ReadCharsFromConsole(char *buffer, long n)
{
#pragma unused (buffer, n)

	return 0;
}

/*
 *	extern char *__ttyname(long fildes);
 *
 *	Return the name of the current terminal (only valid terminals are
 *	the standard stream (ie stdin, stdout, stderr).
 *
 *	long fildes:	The stream to query.
 *
 *	returns char*:	A pointer to static global data which contains a C string
 *					or NULL if the stream is not valid.
 */

extern char *__ttyname(long fildes)
{
#pragma unused (fildes)
	/* all streams have the same name */
	static char *__devicename = "null device";

	if (fildes >= 0 && fildes <= 2)
		return (__devicename);

	return (0L);
}


#pragma mark	-
/* --------------------------------------------------------------------------------
	Implementations of Initialization stuff:
   ----------------------------------------------------------------------------- */

pascal OSErr _Joker_Initialize_Glue( CFragInitBlock *theInitBlock )
{
	OSErr		err;
	
	err = __initialize( theInitBlock );		// Call through.
	
	// Now that all is set up, get shlib's file spec:
	BlockMove( theInitBlock->fragLocator.u.onDisk.fileSpec,
					&gLibraryFileSpec, sizeof(FSSpec) );
	
	return err;
}
#endif /*__MWERKS__*/


/* --------------------------------------------------------------------------------
	JokerInitialize:
		Init JokerLib. This includes initializing Joker as well as opening the
		resource file of this library. This resource file contains the splash
		screen and other resources that may be needed by some instructions.
		
	REVISIONS:
		2002-02-25	UK	Documented, added Bundle support.
   ----------------------------------------------------------------------------- */

pascal void	JokerInitialize()
{
	JOKER_ERROR_START
		gOutputFunction = NULL;
		gHandlerCallFunction = NULL;
		gSpendTimeFunction = NULL;
		
		short		oldRF = CurResFile();
		
	  #if defined(__GNUC__) && (defined(__APPLE_CPP__) || defined(__APPLE_CC__))
		FSRef		vFileRef;
		// Find our Bundle and the res file in it:
		CFBundleRef	fwBundle = CFBundleGetBundleWithIdentifier(CFSTR("com.ulikusterer.joker.lib") );
		CFURLRef	fwResURL = CFBundleCopyResourceURL( fwBundle, CFSTR("JokerLib.rsrc"), NULL, NULL );
		CFURLGetFSRef( fwResURL, &vFileRef );
		CFRelease( fwResURL );
		if( FSOpenResourceFile( &vFileRef, 0, NULL, fsRdPerm, &gLibResFile ) != noErr )
			throw runtime_error( "Couldn't open JokerLib's resources." );
	  #else
		gLibResFile = FSpOpenResFile( &gLibraryFileSpec, fsRdPerm );
		if( gLibResFile <= 0 )
			throw runtime_error( "Couldn't open JokerLib's resources." );
	  #endif
		
		UseResFile( oldRF );
		
		HyperTalk::ForceInitialize();
	JOKER_ERROR_END
}

/* --------------------------------------------------------------------------------
	JokerShowSplash:
		Show the splash screen from our res file. If we wanted to store the
		splash screen in a .nib, we'd also need to look that .nib file up in our
		bundle the same way JokerInitialize() looks up the res file.
		
	REVISIONS:
		2002-02-25	UK	Documented.
   ----------------------------------------------------------------------------- */

pascal void	JokerShowSplash()
{
	short			oldRF = CurResFile();
	DialogRef		theDialog;
	unsigned long	dummy;
	
	UseResFile( gLibResFile );
	
	theDialog = GetNewDialog( 32456, NULL, (WindowRef) -1 );
	if( theDialog != NULL && ResError() == noErr )
	{
		DrawDialog(theDialog);
		#if TARGET_API_MAC_CARBON
		  QDFlushPortBuffer(GetDialogPort(theDialog),NULL);
		#endif
		Delay( 120, &dummy );
		DisposeDialog( theDialog );
	}
	
	UseResFile( oldRF );
}

pascal void	JokerSetNewlineCharacter( char n )
{
	JOKER_ERROR_START
		TalkTokenizer::SetNewline(n);
	JOKER_ERROR_END
}

pascal void	JokerSetOption( JokerOptionID x, Boolean state )
{
	JOKER_ERROR_START
		HyperTalk::SetOption( x, (state == true) );
	JOKER_ERROR_END
}

pascal void	JokerSetOutputProc( JokerOutputProcPtr p )
{
	gOutputFunction = p;
}

pascal void	JokerSetHandlerCallProc( JokerHandlerCallProcPtr p )
{
	gHandlerCallFunction = p;
}

pascal void	JokerSetSpendTimeProc( JokerSpendTimeProcPtr p )
{
	gSpendTimeFunction = p;
}

pascal Boolean	JokerGetErrorString( StringPtr str )
{
	long		len = strlen( gErrorMessageText );
	
	if( len == 0 )
	{
		if( str )
			str[0] = 0;
		return false;
	}
	else
	{
		if( str )
		{
			str[0] = len;
			memmove( str+1, gErrorMessageText, len );
		}
		return true;
	}
}


#pragma mark	-
/* --------------------------------------------------------------------------------
	Implementation of stuff for Manipulating scripts:
   ----------------------------------------------------------------------------- */

pascal JokerScriptRef	JokerNewScript()
{
	JOKER_ERROR_START
		HyperTalk*		s;
	
		s = new HyperTalk();
		s->Retain();
		
		return( (JokerScriptRef) s );
	JOKER_ERROR_END
	
	return NULL;
}

pascal void			JokerRetainScript( JokerScriptRef s )
{
	JOKER_ERROR_START
		((HyperTalk*)s)->Retain();
	JOKER_ERROR_END
}

pascal void			JokerReleaseScript( JokerScriptRef s )
{
	JOKER_ERROR_START
		((HyperTalk*)s)->Release();
	JOKER_ERROR_END
}

pascal void			JokerTokenizeScript( JokerScriptRef s, const char* data, long length )
{
	JOKER_ERROR_START
		TextMunger		vText( data, length );
		HyperTalk*		vScript = (HyperTalk*) s;
		
		vScript->Tokenize( vText );
	JOKER_ERROR_END
}

pascal void			JokerParseScript( JokerScriptRef s )
{
	JOKER_ERROR_START
		((HyperTalk*)s)->Parse();
	JOKER_ERROR_END
}

/* This is for cases where you have just raw instructions, not inside a handler: */
pascal void			JokerParseScriptAsHandler( JokerScriptRef s, Str255 handlerName,
											Boolean isFcn )
{
	JOKER_ERROR_START
		TextMunger		vName( (char*) handlerName+1, (unsigned long) handlerName[0] );
	
		if( isFcn )
			;	// FIX ME!!! 
		else
			;
		
		((HyperTalk*)s)->ParseAsHandler( vName );
	JOKER_ERROR_END
}

pascal JokerCallResult	JokerRunHandler( JokerScriptRef s, Str255 handlerName, Boolean isFcn,
									JokerValueRef returnValue, long paramCount, ... )
{
	JOKER_ERROR_START
		TalkCallResult	vResult;
		TalkMemLocation	vRetVal;
		TextMunger		vName( (char*) handlerName +1, (unsigned long) handlerName[0]);
		ValueStack		vStack;
		va_list			args;
		TalkVarValue*	vValue;
		TalkVarValue*	vNewVal;
		ValueStorage	vStorage;
		
		/* Now stuff aliases to the param values on the stack:
			SendMessage will take care of removing them again,
			which is why we use aliases: the value pointed to
			is not disposed of. */
		va_start( args, paramCount );
		for( short x = 0; x < paramCount; x++ )
		{
			vValue = va_arg( args, TalkVarValue* );
			vNewVal = new TalkVarValue(0L);
			vStorage.valueType = vValue;
			vNewVal->SetValue( vStorage, VALUE_TYPE_VALUE );
			vStack.push_back( vNewVal );
		}
		va_end(args);
		
		// Push param count on stack:
		vStack.push_back( new TalkVarValue( paramCount ) );
		
		// Set up storage for result:
		vRetVal.mType = MEM_LOCATION_TYPE_IMMEDIATE;
		vRetVal.mValue = (TalkVarValue*) returnValue;
		
		// Call handler!
		vResult = ((HyperTalk*)s)->SendMessage( vName, &vStack, vRetVal, isFcn, true );
		
		if( gHandlerCallFunction != NULL && vResult == CALL_RESULT_PASSED )	// Message wasn't caught?
		{
			va_start( args, paramCount );
			vResult = (TalkCallResult) (*gHandlerCallFunction)( handlerName, returnValue, isFcn,
																paramCount, args );
		}
		
		return( (JokerCallResult) vResult );
	JOKER_ERROR_END
	
	return -1;
}


#pragma mark -
/* --------------------------------------------------------------------------------
	Manipulating values:
   ----------------------------------------------------------------------------- */

pascal JokerValueRef	JokerNewValue()
{
	JOKER_ERROR_START
		return( (JokerValueRef) new TalkVarValue(0L) );
	JOKER_ERROR_END
	
	return NULL;
}

pascal JokerValueRef	JokerCopyValue( JokerValueRef v )
{
	JOKER_ERROR_START
		TalkVarValue*		vVal = new TalkVarValue(0L);
	
		((TalkVarValue*)v)->CopyValueTo( *vVal );
		
		return( (JokerValueRef) vVal );
	JOKER_ERROR_END
	
	return NULL;
}

pascal void			JokerDisposeValue( JokerValueRef v )
{
	JOKER_ERROR_START
		delete (TalkVarValue*) v;
	JOKER_ERROR_END
}


pascal JokerVTypes	JokerGetValueAvailableTypes( JokerValueRef v )
{
	JOKER_ERROR_START
		return ((TalkVarValue*) v)->GetAvailableTypes();
	JOKER_ERROR_END
	return 0;
}


void	AppendToHandle( void* data, unsigned long vLength, Handle vStorage )
{
	Size		vHandleSize = GetHandleSize(vStorage);
	
	SetHandleSize( vStorage, vHandleSize +vLength );
	if( MemError() == noErr )
	{
		char*		vDataPtr = *(char**)vStorage;
		vDataPtr += vHandleSize;
		BlockMoveData( data, vDataPtr, vLength );
	}
}


struct JokerFlatValue
{
	JokerVTypes			mType;
	unsigned long		mSize;
	union {
		long			longType;
		double			doubleType;
		bool			boolType;
		LongRect		rectType;
		LongColor		colorType;
		LongPoint		pointType;
		char			textType[];
	};
};


pascal void			JokerFlattenValue( JokerValueRef v, Handle vStorage )
{
	TalkVarValue*		theValue = (TalkVarValue*) v;
	ValueStorage		vStor;
	Size				vHandleSize = GetHandleSize(vStorage);
	char*				vDataPtr = (char*) *vStorage;
	JokerFlatValue		vFlatValue;
	unsigned long		vRecSize = sizeof(unsigned long) +sizeof(JokerVTypes);
	vStorage += vHandleSize;
	
	HLock( vStorage );
	
	JOKER_ERROR_START
		vFlatValue.mType = theValue->GetNativeType();
		switch( vFlatValue.mType )
		{
			case VALUE_TYPE_TEXT:
			{
				theValue->GetValue( vStor, VALUE_TYPE_TEXT );
				TexMunAPtr		vKiller(vStor.textType);
				vKiller->SetOffset(0);
				vFlatValue.mSize = vKiller->GetLength();
				AppendToHandle( &vFlatValue, vRecSize, vStorage );
				AppendToHandle( vKiller->GetTextPtr(), vFlatValue.mSize, vStorage );
				break;
			}
			
			case VALUE_TYPE_LONG:
				theValue->GetValue( vStor, VALUE_TYPE_LONG );
				vFlatValue.mSize = sizeof(long);
				vFlatValue.longType = vStor.longType;
				AppendToHandle( &vFlatValue, vRecSize +vFlatValue.mSize, vStorage );
				break;
			
			case VALUE_TYPE_DOUBLE:
				theValue->GetValue( vStor, VALUE_TYPE_DOUBLE );
				vFlatValue.mSize = sizeof(double);
				vFlatValue.doubleType = vStor.doubleType;
				AppendToHandle( &vFlatValue, vRecSize +vFlatValue.mSize, vStorage );
				break;
			
			case VALUE_TYPE_RECT:
				theValue->GetValue( vStor, VALUE_TYPE_RECT );
				vFlatValue.mSize = sizeof(LongRect);
				vFlatValue.rectType = vStor.rectType;
				AppendToHandle( &vFlatValue, vRecSize +vFlatValue.mSize, vStorage );
				break;
			
			case VALUE_TYPE_POINT:
				theValue->GetValue( vStor, VALUE_TYPE_POINT );
				vFlatValue.mSize = sizeof(LongPoint);
				vFlatValue.pointType = vStor.pointType;
				AppendToHandle( &vFlatValue, vRecSize +vFlatValue.mSize, vStorage );
				break;
			
			case VALUE_TYPE_COLOR:
				theValue->GetValue( vStor, VALUE_TYPE_COLOR );
				vFlatValue.mSize = sizeof(LongColor);
				vFlatValue.colorType = vStor.colorType;
				AppendToHandle( &vFlatValue, vRecSize +vFlatValue.mSize, vStorage );
				break;
			
			case VALUE_TYPE_BOOL:
				theValue->GetValue( vStor, VALUE_TYPE_BOOL );
				vFlatValue.mSize = sizeof(bool);
				vFlatValue.boolType = vStor.boolType;
				AppendToHandle( &vFlatValue, vRecSize +vFlatValue.mSize, vStorage );
				break;
			
			case VALUE_TYPE_LIST:
				throw runtime_error("Flattening lists is not yet implemented.");
				break;
		}
	JOKER_ERROR_END
}


#pragma mark	-
/* --------------------------------------------------------------------------------
	Assigning values to values:
   ----------------------------------------------------------------------------- */

pascal void			JokerSetValueLong( JokerValueRef v, long n )
{
	JOKER_ERROR_START
		ValueStorage		vStorage;
	
		vStorage.longType = n;
		((TalkVarValue*)v)->SetValue( vStorage, VALUE_TYPE_LONG );
	JOKER_ERROR_END
}

pascal void			JokerSetValueDouble( JokerValueRef v, double n )
{
	JOKER_ERROR_START
		ValueStorage		vStorage;
	
		vStorage.doubleType = n;
		((TalkVarValue*)v)->SetValue( vStorage, VALUE_TYPE_DOUBLE );
	JOKER_ERROR_END
}

pascal void			JokerSetValueString( JokerValueRef v, Str255 str )
{
	JOKER_ERROR_START
		ValueStorage		vStorage;
	
		vStorage.textType = new TextMunger( (char*) str+1, (unsigned long) str[0] );
		TexMunAPtr		vMun( vStorage.textType );
		
		((TalkVarValue*)v)->SetValue( vStorage, VALUE_TYPE_TEXT );
	JOKER_ERROR_END
}

pascal void			JokerSetValueChars( JokerValueRef v, char* data, long length )
{
	JOKER_ERROR_START
		ValueStorage		vStorage;
		
		vStorage.textType = new TextMunger( data, length );
		TexMunAPtr		vMun( vStorage.textType );
		
		((TalkVarValue*)v)->SetValue( vStorage, VALUE_TYPE_TEXT );
	JOKER_ERROR_END
}

pascal void			JokerSetValuePoint( JokerValueRef v, const Point pos )
{
	JOKER_ERROR_START
		ValueStorage		vStorage;
		
		vStorage.pointType.x = pos.h;
		vStorage.pointType.y = pos.v;
		
		((TalkVarValue*)v)->SetValue( vStorage, VALUE_TYPE_POINT );
	JOKER_ERROR_END
}

pascal void			JokerSetValueColor( JokerValueRef v, const RGBColor* col )
{
	JOKER_ERROR_START
		ValueStorage		vStorage;
		
		vStorage.colorType.red = col->red;
		vStorage.colorType.green = col->green;
		vStorage.colorType.blue = col->blue;
		
		((TalkVarValue*)v)->SetValue( vStorage, VALUE_TYPE_COLOR );
	JOKER_ERROR_END
}

pascal void			JokerSetValueRect( JokerValueRef v, const Rect* box )
{
	JOKER_ERROR_START
		ValueStorage		vStorage;
		
		vStorage.rectType.left = box->left;
		vStorage.rectType.top = box->top;
		vStorage.rectType.right = box->right;
		vStorage.rectType.bottom = box->bottom;
		
		((TalkVarValue*)v)->SetValue( vStorage, VALUE_TYPE_RECT );
	JOKER_ERROR_END
}


#pragma mark	-
/* --------------------------------------------------------------------------------
	Retrieving values from values:
   ----------------------------------------------------------------------------- */

pascal long			JokerGetValueLong( JokerValueRef v )
{
	JOKER_ERROR_START
		ValueStorage	vStorage;
	
		((TalkVarValue*)v)->GetValue( vStorage, VALUE_TYPE_LONG );
		
		return vStorage.longType;
	JOKER_ERROR_END
	
	return 0;
}

pascal double			JokerGetValueDouble( JokerValueRef v )
{
	JOKER_ERROR_START
		ValueStorage	vStorage;
	
		((TalkVarValue*)v)->GetValue( vStorage, VALUE_TYPE_DOUBLE );
		
		return vStorage.doubleType;
	JOKER_ERROR_END
	
	return 0.0;
}

pascal void			JokerGetValueString( JokerValueRef v, Str255 str )
{
	JOKER_ERROR_START
		ValueStorage	vStorage;
	
		((TalkVarValue*)v)->GetValue( vStorage, VALUE_TYPE_TEXT );
		TexMunAPtr	vKiller( vStorage.textType );
		
		str[0] = vKiller->GetLength();
		vKiller->SetOffset(0);
		vKiller->PeekData( str+1, str[0] );
	JOKER_ERROR_END
}


pascal long			JokerGetValueChars( JokerValueRef v, char* data, long maxLength )
{
	JOKER_ERROR_START
			ValueStorage	vStorage;
		long			vLeftLength,
						vCurrAmount;
		
		((TalkVarValue*)v)->GetValue( vStorage, VALUE_TYPE_TEXT );
		TexMunAPtr	vKiller( vStorage.textType );
		
		vLeftLength = vKiller->GetLength();
		if( vLeftLength > maxLength )
		{
			vLeftLength -= maxLength;
			vCurrAmount = maxLength;
		}
		else if( vLeftLength < maxLength )
		{
			vCurrAmount = vLeftLength;
			vLeftLength -= maxLength;
		}
		else
		{
			vCurrAmount = vLeftLength;
			vLeftLength = 0;
		}
		
		if( vCurrAmount > 0 )
		{
			vKiller->SetOffset(0);
			vKiller->PeekData( data, vCurrAmount );
		}
		
		return vLeftLength;
	JOKER_ERROR_END
	
	return 0;
}

pascal void			JokerGetValuePoint( JokerValueRef v, Point* pos )
{
	JOKER_ERROR_START
		ValueStorage	vStorage;
	
		((TalkVarValue*)v)->GetValue( vStorage, VALUE_TYPE_POINT );
		
		pos->h = vStorage.pointType.x;
		pos->v = vStorage.pointType.y;
	JOKER_ERROR_END
}

pascal void			JokerGetValueColor( JokerValueRef v, RGBColor* col )
{
	JOKER_ERROR_START
		ValueStorage	vStorage;
	
		((TalkVarValue*)v)->GetValue( vStorage, VALUE_TYPE_POINT );
		
		col->red = vStorage.colorType.red;
		col->green = vStorage.colorType.green;
		col->blue = vStorage.colorType.blue;
	JOKER_ERROR_END
}

pascal void			JokerGetValueRect( JokerValueRef v, Rect* box )
{
	JOKER_ERROR_START
		ValueStorage	vStorage;
		
		((TalkVarValue*)v)->GetValue( vStorage, VALUE_TYPE_RECT );
		
		box->left = vStorage.rectType.left;
		box->top = vStorage.rectType.top;
		box->right = vStorage.rectType.right;
		box->bottom = vStorage.rectType.bottom;
	JOKER_ERROR_END
}


#pragma mark -

pascal JokerValueRef	JokerFetchValueElement( JokerValueRef v, Str255 index )
{
	JOKER_ERROR_START
		TextMunger		vIndex( (char*) index +1, (long) index[0] );
		TalkValue*		vEntry;
		
		// Get entry of that index from value:
		((TalkVarValue*)v)->GetListEntry( vEntry, vIndex );
		
		return( (JokerValueRef) vEntry );
	JOKER_ERROR_END
	
	return NULL;
}

pascal long				JokerCountValueElements( JokerValueRef v )
{
	JOKER_ERROR_START
		return ((TalkVarValue*) v)->CountEntries();
	JOKER_ERROR_END
	
	return 0;
}

#pragma mark	-

pascal JokerEntityRef		JokerNewEntity( JokerEntityProcPtr p )
{
	JOKER_ERROR_START
		return( (JokerEntityRef) new JokerLibEntity(p) );
	JOKER_ERROR_END
	
	return NULL;
}

pascal void		JokerDisposeEntity( JokerEntityRef jer )
{
	JOKER_ERROR_START
		delete (JokerLibEntity*) jer;
	JOKER_ERROR_END
}

pascal void		JokerSetEntityRefCon( JokerEntityRef jer, long value )
{
	((JokerLibEntity*) jer)->mRefCon = value;
}

pascal long		JokerGetEntityRefCon( JokerEntityRef jer )
{
	return( ((JokerLibEntity*) jer)->mRefCon );
}


pascal JokerValueRef	JokerGetEntityPropertyArray( JokerEntityRef jer )
{
	JOKER_ERROR_START
		return( (JokerValueRef) dynamic_cast<TalkVarValue*>( ((JokerLibEntity*) jer)->GetPropertyArray() ) );
	JOKER_ERROR_END
	
	return NULL;
}


pascal void		JokerSetEntityPropertyArray( JokerEntityRef jer, JokerValueRef v )
{
	JOKER_ERROR_START
		((JokerLibEntity*) jer)->SetPropertyArray( *(TalkVarValue*) v );
	JOKER_ERROR_END
}


pascal JokerValueRef	JokerGetEntityPropertyNameArray( JokerEntityRef jer )
{
	JOKER_ERROR_START
		return( (JokerValueRef) dynamic_cast<TalkVarValue*>( ((JokerLibEntity*) jer)->GetPropertyNameArray() ) );
	JOKER_ERROR_END
	
	return NULL;
}



#pragma mark	-

/* --------------------------------------------------------------------------------
	Custom Syntax:
   ----------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------------
	JokerRegisterClientCmd:
		Register a custom command.
	
	TAKES:
		paramCount		-	Number of parameters to the command, not including the
							direct argument.
		inIdentifiers	-	inIdentifiers[0] contains the name of the command.
							Any additional entries contain the labels for the
							corresponding direct argument. If any of those entries
							is NULL or an empty string, it is assumed that the
							parameter has no label.
		hasDirectArg	-	TRUE if you want to have a direct argument, FALSE if
							you want the command to start immediately with the
							first argument's label.
		p				-	The callback proc to call.
	
	REVISIONS:
		2001-07-16	UK	Changed to use a per-command callback proc instead of
						one for all commands.
		2001-07-02	UK	Created.
   ----------------------------------------------------------------------------- */

pascal void	JokerRegisterClientCmd( long paramCount, StringPtr inIdentifiers[],
										Boolean hasDirectArg, JokerClientCmdProcPtr p )
{
	JOKER_ERROR_START
		short			x;
		TokenTypeEnum	tType[TALK_INSTR_MAX_PARAMS];
		
		/* Build a list of all token IDs required for parsing this command:
			This creates new tokens for identifiers xTalk doesn't yet know. */
		for( x = 0; x <= paramCount; x++ )
		{
			if( inIdentifiers[x] != NULL )
			{
				TextMunger		vMungie( (char*) inIdentifiers[x] +1, inIdentifiers[x][0] );
				
				vMungie.RecalcHash();
				tType[x] = TalkTokenizer::GetIdentifierType( vMungie );
				if( tType[x] == TOKEN_TYPE_IDENTIFIER )	// Unknown? Make a new one!
				{
					tType[x] = TalkTokenizer::GetNewTokenID();
					TalkTokenizer::RegisterIdentifierToken( vMungie, tType[x] );
				}
			}
			else
				tType[x] = TOKEN_TYPE_INVALID;
		}
		
		/* Now loop over list of tokens and build a command entry: */
		TalkCommandEntry*		vEntry;
		
		vEntry = (TalkCommandEntry*) malloc(sizeof(TalkCommandEntry));
		vEntry->mResultParam = TALK_PARAM_IGNORE;
		vEntry->mParamCount = paramCount +1;	// +1 since this mParamCount also includes command name's entry.
		vEntry->mRefCon = (long) p;	// Keep track of callback proc.
		
		// Set up command entry's command name & direct arg:
		vEntry->mParam[0].mTokenType = tType[0];
		vEntry->mParam[0].mParamType = hasDirectArg ? TALK_PARAM_TYPE_EXPRESSION : TALK_PARAM_TYPE_INVALID;
		vEntry->mParam[0].mDestParam = hasDirectArg ? 1 : 0;
		vEntry->mParam[0].mOptional = false;
		vEntry->mParam[0].mCommandInstruction = HyperClientCmdInstruction;
		for( x = 1; x <= paramCount; x++ )
		{
			vEntry->mParam[x].mTokenType = tType[x];
			vEntry->mParam[x].mParamType = TALK_PARAM_TYPE_EXPRESSION;
			vEntry->mParam[x].mDestParam = x +(hasDirectArg ? 1 : 0);
			vEntry->mParam[x].mOptional = false;
			vEntry->mParam[x].mCommandInstruction = NULL;
		}
		
		HyperTalk::RegisterBuiltInCommand( vEntry );	// Don't dispose, command entry list is still using this.
	JOKER_ERROR_END
}


/* --------------------------------------------------------------------------------
	JokerRegisterObjectDescriptor:
		Register a custom object descriptor.
		
	REVISIONS:
		2001-07-02	UK	Created.
   ----------------------------------------------------------------------------- */

pascal void	JokerRegisterObjectDescriptor( long paramCount, StringPtr inIdentifiers[],
										Boolean hasDirectArg,
										JokerObjectDescriptorProcPtr proc )
{
	JOKER_ERROR_START
		short			x;
		TokenTypeEnum	tType[TALK_INSTR_MAX_PARAMS];
		
		/* Build a list of all token IDs required for parsing this command:
			This creates new tokens for identifiers xTalk doesn't yet know. */
		for( x = 0; x <= paramCount; x++ )
		{
			if( inIdentifiers[x] != NULL )
			{
				TextMunger		vMungie( (char*) inIdentifiers[x] +1, inIdentifiers[x][0] );
				
				vMungie.RecalcHash();
				tType[x] = TalkTokenizer::GetIdentifierType( vMungie );
				if( tType[x] == TOKEN_TYPE_IDENTIFIER )	// Unknown? Make a new one!
				{
					tType[x] = TalkTokenizer::GetNewTokenID();
					TalkTokenizer::RegisterIdentifierToken( vMungie, tType[x] );
				}
			}
			else
				tType[x] = TOKEN_TYPE_INVALID;
		}
		
		/* Now loop over list of tokens and build a object descriptor entry: */
		TalkCommandEntry*		vEntry;
		
		vEntry = (TalkCommandEntry*) malloc(sizeof(TalkCommandEntry));
		vEntry->mResultParam = TALK_PARAM_IGNORE;
		vEntry->mParamCount = paramCount +1;	// +1 since this mParamCount also includes command name's entry.
		vEntry->mRefCon = (long) proc;	// Remember proc to call for this object descriptor
		
		// Set up command entry's command name & direct arg:
		vEntry->mParam[0].mTokenType = tType[0];
		vEntry->mParam[0].mParamType = hasDirectArg ? TALK_PARAM_TYPE_EXPRESSION : TALK_PARAM_TYPE_INVALID;
		vEntry->mParam[0].mDestParam = hasDirectArg ? 1 : 0;
		vEntry->mParam[0].mOptional = false;
		vEntry->mParam[0].mCommandInstruction = HyperClientObjDescInstruction;
		for( x = 1; x <= paramCount; x++ )
		{
			vEntry->mParam[x].mTokenType = tType[x];
			vEntry->mParam[x].mParamType = TALK_PARAM_TYPE_EXPRESSION;
			vEntry->mParam[x].mDestParam = x +(hasDirectArg ? 1 : 0);
			vEntry->mParam[x].mOptional = false;
			vEntry->mParam[x].mCommandInstruction = NULL;
		}
		
		HyperTalk::RegisterObjectDescriptor( vEntry );	// Don't dispose, command entry list is still using this.
	JOKER_ERROR_END
}

//#ifdef __cplusplus
//}
//#endif


/* --------------------------------------------------------------------------------
	HyperClientCmdInstruction:
		This handles client-defined commands.
	
	HI's FIELDS:
		0		-	event class
		1		-	event ID
		2		-	sender
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperClientCmdInstruction( struct HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec )
{
	JokerClientCmdProcPtr	vMsg = (JokerClientCmdProcPtr) hi.GetRefCon();
	TalkVarValue*			vParams[TALK_INSTR_MAX_PARAMS];
	long					x;
	
	for( x = 0; x < hi.mParamCount; x++ )
	{
		vParams[x] = new TalkVarValue(0L);
		hi.mParam[x].CopyValueTo( *vParams[x], s );
	}
	
	(*vMsg)( hi.mParamCount, (JokerValueRef*) vParams );
	
	for( x = 0; x < hi.mParamCount; x++ )
		delete( vParams[x] );
}


/* --------------------------------------------------------------------------------
	HyperClientObjDescInstruction:
		This handles client-defined object descriptors.
   ----------------------------------------------------------------------------- */

void	HyperClientObjDescInstruction( struct HyperInstruction& hi, ValueStack& s,
										TalkCallRecord& vCallRec )
{
	StringPtr			vMsg = (StringPtr) hi.GetRefCon();
	TalkVarValue*		vParams[TALK_INSTR_MAX_PARAMS];
	long				x;
	ValueStorage		vResult;
	JokerObjectDescriptorProcPtr	vProc = (JokerObjectDescriptorProcPtr) hi.mRefCon;
	
	for( x = 0; x < hi.mParamCount; x++ )
	{
		vParams[x] = new TalkVarValue(0L);
		hi.mParam[x].CopyValueTo( *vParams[x], s );
	}
	
	vResult.entityType = (JokerLibEntity*) ((*vProc)( hi.mParamCount, (JokerValueRef*) vParams ));
	
	for( x = 0; x < hi.mParamCount; x++ )
		delete( vParams[x] );
	
	if( vResult.entityType == NULL )
		throw runtime_error("No such object.");
	
	hi.mResult.SetValue( vResult, s, VALUE_TYPE_ENTITY );
}


#pragma mark	-
/* --------------------------------------------------------------------------------
	Utility stuff:
   ----------------------------------------------------------------------------- */


/* --------------------------------------------------------------------------------
	JokerSpendTime:
		This is called after each line the compiler executes and at other
		regular intervals so you can process events while a script is running.
	
	TAKES:
		-
	
	GIVES:
		-
	
	REVISIONS:
		2000-10-27	UK		Created.
   ----------------------------------------------------------------------------- */

void	JokerSpendTime()
{
	if( gSpendTimeFunction )
		(*gSpendTimeFunction)();
}


#if JOKER_MACINTOSH
/* --------------------------------------------------------------------------------
	HyperAppleEventInstruction:
		This handles the "appleEvent" message. When we arrive here the message
		wasn't trapped by any handlers. So we check whether it is an event Joker
		knows how to handle internally or otherwise we mark the event as unused
		so our AppleEvent handler can generate an error message.
	
	HI's FIELDS:
		0		-	event class
		1		-	event ID
		2		-	sender
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperAppleEventInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	s.SetCurrMacAEHandled(false);	// ... make sure we know it wasn't handled.
}
#endif /*JOKER_MACINTOSH*/







