/* ================================================================================
	PROJECT:	JokerLib
	FILE:		JokerLibTest.c
	PURPOSE:	A little testing application that demonstrates the general use of
				JokerLib.
	COPYRIGHT:	(c) Copyright 2001 by M. Uli Kusterer, all rights reserved.
   ============================================================================= */

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */


#if defined(__GNUC__) && (defined(__APPLE_CPP__) || defined(__APPLE_CC__))
#include	<Carbon/Carbon.h>
#else
#include <Types.h>
#include <Memory.h>
#include <Quickdraw.h>
#include <Fonts.h>
#include <Events.h>
#include <Menus.h>
#include <Windows.h>
#include <TextEdit.h>
#include <Dialogs.h>
#include <OSUtils.h>
#include <ToolUtils.h>
#include <SegLoad.h>
#include <Sound.h>
#endif
#include <string.h>
#include <stdio.h>
#if defined(__MWERKS__)
#include <sioux.h>
#endif

// This is the main header file for JokerLib:
#include "JokerLib.h"


/* --------------------------------------------------------------------------------
	Data Structures:
   ----------------------------------------------------------------------------- */

/* Data structure associated with each of our objects. This is entirely
	client-defined. */
struct UObjData
{
	JokerEntityRef	mJokerObject;	// Reference to the Entity that represents this object to Joker.
	Str255			mName;			// Storage for our object's built-in "name" propoerty.
	JokerValueRef	mContents;		// Storage for our object's contents that we can pass to Joker.
};


/* --------------------------------------------------------------------------------
	Globals:
   ----------------------------------------------------------------------------- */

struct UObjData		gUlisObjectData;	// This example only supports one global object.



/* --------------------------------------------------------------------------------
	Prototypes:
   ----------------------------------------------------------------------------- */

void Initialize(void);
#if defined(__MWERKS__)
extern long	WriteCharsToConsole( char* data, long amount );
#endif


/* --------------------------------------------------------------------------------
	UlisOutputProc:
		Callback procedure that handles console output by scripts. In this case
		we just call through to SIOUX using an MSL-defined routine. If you're not
		using CodeWarrior, you'll have to do this differently, or in a UI app you
		would probably just display this in the message box.
	
	TAKES:
		data	-	Pointer to the data to output.
		amount	-	Number of bytes to output from the pointer.
	
	GIVES:
		long	-	Length of data actually written.
	
	REVISIONS:
		2001-07-02	UK	Documented
   ----------------------------------------------------------------------------- */

pascal long	UlisOutputProc( char* data, long amount )
{
	#if defined(__MWERKS__)
	return WriteCharsToConsole( data, amount );	// MSL-specific function that directly writes to the console.
	#else
	long	x;
	
	for( x = 0; x < amount; x++ )
		printf("%c",data[x]);
	return x;
	#endif
}


/* --------------------------------------------------------------------------------
	UlisHandlerCallProc:
		Callback procedure that is called whenever a message isn't trapped that
		you sent using JokerRunHandler().
	
	TAKES:
		handlerName	-	Name of the handler that wasn't handled.
		result		-	Here you can return a result from the handler call.
		isFcn		-	TRUE if it's a function FALSE if a command.
		paramCount	-	Number of entries in the va_list.
		args		-	va_list with the parameters, va_start() has already been
						called on this.
	
	GIVES:
		JokerCallResult	-	Return whether you intercepted the message here or
							whether this handler exited or it was passed.
	
	REVISIONS:
		2001-07-02	UK	Documented
   ----------------------------------------------------------------------------- */

pascal JokerCallResult	UlisHandlerCallProc( Str255 handlerName, JokerValueRef result, Boolean isFcn,
									long paramCount, va_list args )
{
	char		vStr[256];
	
	memmove( vStr, handlerName +1, handlerName[0] );
	vStr[handlerName[0]] = 0;
	
	printf( "The Handler call \"%s\" wasn't intercepted.\n", vStr );
	
	return kJokerCallResultIntercepted;
}


/* --------------------------------------------------------------------------------
	UlisCmdMessageProc:
		Callback procedure that is called whenever our client-defined
		command is called.
	
	TAKES:
		paramCount	-	Number of parameters passed to the command.
		params		-	Array containing the parameters.
	
	REVISIONS:
		2001-07-16	UK	Adapted to new per-command callback.
		2001-07-02	UK	Documented
   ----------------------------------------------------------------------------- */

pascal void	UlisCmdMessageProc( long paramCount, JokerValueRef params[] )
{
	char		vStr[256];
	Str255		myStr;
	
	// Fetch 1st param, convert to C string & display:
	JokerGetValueString( params[0], myStr );
	memmove( vStr, myStr +1, myStr[0] );
	vStr[myStr[0]] = 0;
	printf( "User invoked our custom command\n\twith parameters \"%s\"", vStr );
	
	// ... same for param 2:
	JokerGetValueString( params[1], myStr );
	memmove( vStr, myStr +1, myStr[0] );
	vStr[myStr[0]] = 0;
	printf( "\n\tand \"%s\".\n", vStr );
}


/* --------------------------------------------------------------------------------
	UlisObjDescProc:
		Callback procedure that is called whenever one of your client-defined
		object descriptors is used.
		
		Since this is only a demo, this function simply checks whether its
		parameter is "1". If the user requested object number one, we simply
		return our pre-fabricated object. Otherwise we return NULL to indicate
		an error. In a real-life program you would probably use the number to
		look up an object in an array and then return that.
	
	TAKES:
		paramCount	-	Number of parameters passed to the command.
		params		-	Array containing the parameters.
	
	REVISIONS:
		2001-07-15	UK	Created.
   ----------------------------------------------------------------------------- */

pascal JokerEntityRef	UlisObjDescProc( long paramCount, JokerValueRef params[] )
{
	long		objNum;
	
	// First param (direct argument) is object number:
	objNum = JokerGetValueLong( params[0] );
	
	// We only have one object, but we could look it up in a list here:
	if( objNum == 1 )
		return gUlisObjectData.mJokerObject;
	else
		return NULL;	// Indicate we couldn't find it.
}


/* --------------------------------------------------------------------------------
	UlisEntityProc:
		Callback procedure that is called whenever one of your client-defined
		objects is used.
	
	TAKES:
		target	-	The object we're messing with.
		event	-	An action selector telling us what to do.
		param	-	Additional data.
		param2	-	Additional data.
		JokerValueRef	-	Return value if event == kJokerEntityGetContentsEvent.
							Else return NULL on error and 1 on success.
	
	REVISIONS:
		2001-07-15	UK	Created.
   ----------------------------------------------------------------------------- */

pascal JokerValueRef	UlisEntityProc( JokerEntityRef target, JokerEntityEvent event,
										JokerValueRef param, JokerValueRef param2 )
{
	Str255				vPropName;
	struct UObjData*	vCurrObj = (struct UObjData*) JokerGetEntityRefCon(target);
	
	switch( event )
	{
		case kJokerEntityGetPropEvent:
			JokerGetValueString( param, vPropName );
			if( EqualString( "\pname", vPropName, false, true ) )
				JokerSetValueString(param2,vCurrObj->mName);
			else
				return NULL;	// No such property.
			break;
		
		case kJokerEntitySetPropEvent:
			JokerGetValueString( param, vPropName );
			if( EqualString( "\pname", vPropName, false, true ) )
				JokerGetValueString(param2,vCurrObj->mName);
			else
				return NULL;	// No such property.
			break;
		
		case kJokerEntityGetContentsEvent:
			return vCurrObj->mContents;
			break;
		
		case kJokerEntityDeleteEvent:
			// Should delete object here.
			return NULL;	// Indicate we didn't.
			break;
	}
	
	return( (JokerValueRef) 1 );	// Return this to indicate found properties.
}


/* --------------------------------------------------------------------------------
	CheckError:
		A little wrapper that calls JokerGetErrorString() to check for errors.
		If there was an error, this displays the error message on the console
		and returns TRUE, otherwise it returns FALSE.
	
	TAKES:
		handlerName	-	Name of the command called.
		paramCount	-	Number of parameters passed to the command.
		params		-	Array containing the parameters.
	
	REVISIONS:
		2001-07-02	UK	Documented
   ----------------------------------------------------------------------------- */

Boolean	CheckError()
{
	Str255		errStr;
	
	if( JokerGetErrorString(errStr) )
	{
		errStr[errStr[0]+1] = 0;	// So we can use it as a C string.
		printf("\nAn Error occurred: %s",(char*) errStr+1);
		return true;
	}
	else
		return false;
}


/* --------------------------------------------------------------------------------
	main:
		Main entry point. Registers our callbacks, sets up options to behave like
		HyperCard except for addes support for escape sequences in strings, and
		registers a custom command with syntax "combine <1> with <2>".
		It also registers a custom object descriptor "object <number>".
		Then it loads a script from TEXT resource ID 128 and calls its startUp
		handler with one parameter and displays its return value.
	
	REVISIONS:
		2001-07-16	UK	Adapted to new per-command callback.
		2001-07-15	UK	Added support for client-defined objects.
		2001-07-02	UK	Documented
   ----------------------------------------------------------------------------- */

int	main(void)
{
	JokerScriptRef		vScript = NULL;
	Handle				vText;
	JokerValueRef		vParam;
	JokerValueRef		vReturnValue;
	Str255				str;
	char*				vStr;
	StringPtr			vIdentifiers[2] = { "\pcombine", "\pwith" };
	StringPtr			vObjDesc[1]		= { "\pobject" };
	
	Initialize();
	
	// JokerLib not present?
	if( ((long)JokerInitialize) == kUnresolvedCFragSymbolAddress )
	{
		SysBeep(10);
		return 0;
	}
	
	// Init Joker:
	JokerInitialize();
	if( CheckError() )	return -1;
	
	// We need this to init the console, since our output proc calls WriteCharsToConsole() directly.
	printf( "JokerLibTest 1.1\nA simple test application that uses JokerLib.\n\n" );
	
	// Set up some options & callback procs:
	JokerSetNewlineCharacter( '\r' );	// Make sure line ends are proper.
	JokerSetOption( kJokerOptionAllowUnquotedLiterals, true );
	JokerSetOption( kJokerOptionDistinguishFcnCmd, true );
	JokerSetOption( kJokerOptionUnqoutedLitsAsVars, true );
	JokerSetOption( kJokerOptionEscapeCharsInStrings, true );
	
	JokerSetOutputProc( UlisOutputProc );
	JokerSetHandlerCallProc( UlisHandlerCallProc );
	
	// Register a custom command:
	JokerRegisterClientCmd( 1, vIdentifiers, true, UlisCmdMessageProc );
	if( CheckError() )	return -1;
	
	// Register a custom object descriptor:
	JokerRegisterObjectDescriptor( 0, vObjDesc, true, UlisObjDescProc );
	if( CheckError() )	return -1;
	
	// Create a demo object so we can actually use it:
	gUlisObjectData.mJokerObject = JokerNewEntity( UlisEntityProc );
	gUlisObjectData.mName[0] = 0;
	gUlisObjectData.mContents = JokerNewValue();
	JokerSetEntityRefCon( gUlisObjectData.mJokerObject, (long) &gUlisObjectData );
	
	// Now get script text from somewhere:
	vText = GetResource( 'TEXT', 128 );
	if( vText == NULL ) return -1;
	DetachResource( vText );
	HLock( vText );
	
	// Make script, tokenize and parse it:
	vScript = JokerNewScript();
	if( CheckError() )	return -1;
	JokerTokenizeScript( vScript, (char*) *vText, GetHandleSize(vText) );
	if( CheckError() )	return -1;
	JokerParseScript( vScript );
	if( CheckError() )	return -1;
	
	// Set up params and return value storage:
	vParam = JokerNewValue();
	if( CheckError() )	return -1;
	vReturnValue = JokerNewValue();
	if( CheckError() )	return -1;
	JokerSetValueString( vParam, "\pInput data" );
	if( CheckError() )	return -1;
	
	// Run handler, and output return value:
	JokerRunHandler( vScript, "\pstartUp", false, vReturnValue, 1, vParam );
	if( CheckError() )	return -1;
	JokerGetValueString( vReturnValue, str );
	if( CheckError() )	return -1;
	str[str[0]+1] = 0;	// Terminate so we can use it as C string.
	
	printf( "\nReturn Value: %s", ((char*) str +1) );
	
	// Clean up:
	JokerDisposeValue( vParam );
	JokerDisposeValue( vReturnValue );
	JokerDisposeEntity( gUlisObjectData.mJokerObject );
	JokerReleaseScript( vScript );
	DisposeHandle( vText );
	
	/* You _must_ call the following at startup or shutdown if you haven't
		paid for JokerLib: */
	JokerShowSplash();

	return 0;
}

/* --------------------------------------------------------------------------------
	Initialize:
		Initialize the Macintosh Toolbox. and set up SIOUX not to do it again.
	
	REVISIONS:
		2001-07-02	UK	Documented
   ----------------------------------------------------------------------------- */

void Initialize(void)
{
	#if TARGET_API_MAC_OS8
	InitGraf(&qd.thePort);
	InitFonts();
	InitWindows();
	InitMenus();
	TEInit();
	InitDialogs(nil);
	#endif
	InitCursor();
	
	#if defined(__MWERKS__)
	SIOUXSetTitle("\pJokerLibTest");
	SIOUXSettings.initializeTB = false;	// We already did that.
	#endif
}


