/* =============================================================================
	
	PROJECT:	Joker
	
	FILE:		GetPartialPath.c
	
	PURPOSE:	Utility functions to get full (=absolute) or partial
				(=relative) pathnames from an FSSpec. This code is roughly
				based on Jim Luther's MoreFiles code, 
	
	COPYRIGHT:	(c) copyright 2002 by M. Uli Kusterer, all rights reserved.
	
	AUTHORS:	UK	-	M. Uli Kusterer, <witness@weblayout.com>
	
	REVISIONS:
		2002-03-01	UK	Documented.
	
   ========================================================================== */

/* -----------------------------------------------------------------------------
	Headers:
   -------------------------------------------------------------------------- */

#include	"GetPartialPath.h"
#include	<string.h>


/* -----------------------------------------------------------------------------
	FSpGetPartialPath:
		Return a partial (=relative) pathname for the specified file relative
		to a directory. Note that the directory is specified by dirID only, and
		is thus assumed to be on the same volume as the file itself.
	
	TAKES:
		spec	-	The file for which you need a partial pathname.
		dirID	-	The dirID of a directory (=folder) on the same volume as
					the file in spec.
	
	GIVES:
		outPath	-	The Str255 passed here is set to the partial pathname.
		OSErr	-	Error code if something went wrong, otherwise noErr.
	
	REVISIONS:
		2002-03-01	UK	Documented.
   -------------------------------------------------------------------------- */

OSErr	FSpGetPartialPath( const FSSpec *spec,	// target item
							long dirID,			// build partial pathname down to this directory
							Str255 outPath )	// Path name is returned here.
{
	OSErr		result;
	OSErr		realResult;
	FSSpec		tempSpec;
	CInfoPBRec	pb;
	Str255		fullPath;
	short		depth = 0;
	
	fullPath[0] = 0;
	
	/* Make a copy of the input FSSpec that can be modified */
	BlockMoveData( spec, &tempSpec, sizeof(FSSpec) );
	
	if( tempSpec.parID == fsRtParID )
	{
		/* The object is a volume */
		
		/* Add a colon to make it a full pathname */
		++tempSpec.name[0];
		tempSpec.name[tempSpec.name[0]] = ':';
		
		/* We're done */
		BlockMoveData( tempSpec.name, fullPath, tempSpec.name[0] +1 );
		result = MemError();
	}
	else
	{
		/* The object isn't a volume */
		
		/* Is the object a file or a directory? */
		pb.dirInfo.ioNamePtr = tempSpec.name;
		pb.dirInfo.ioVRefNum = tempSpec.vRefNum;
		pb.dirInfo.ioDrDirID = tempSpec.parID;
		pb.dirInfo.ioFDirIndex = 0;
		result = PBGetCatInfoSync(&pb);
		
		// Allow file/directory name at end of path to not exist.
		realResult = result;
		if( (result == noErr) || (result == fnfErr) )
		{
			/* if the object is a directory, append a colon so full pathname ends with colon */
			if ( (result == noErr) && (pb.hFileInfo.ioFlAttrib & ioDirMask) != 0 )
			{
				++tempSpec.name[0];
				tempSpec.name[tempSpec.name[0]] = ':';
			}
		  
			/* Put the object name in first */
			BlockMoveData( tempSpec.name, fullPath, tempSpec.name[0] +1 );
			result = MemError();
			if( result == noErr )
			{
			    /* Get the ancestor directory names */
				pb.dirInfo.ioNamePtr = tempSpec.name;
				pb.dirInfo.ioVRefNum = tempSpec.vRefNum;
				pb.dirInfo.ioDrParID = tempSpec.parID;
				while( (result == noErr) && (pb.dirInfo.ioDrParID != dirID) )  /* loop until we have an error or find the root directory */
				{
					pb.dirInfo.ioFDirIndex = -1;
					pb.dirInfo.ioDrDirID = pb.dirInfo.ioDrParID;
					result = PBGetCatInfoSync(&pb);
					if( result == noErr /*&& (pb.dirInfo.ioDrParID != dirID)*/ )	// UK 2002-03-07 Commented since this would lose last item of partial folder pathnames.
					{
						depth++;
						
						/* Append colon to directory name */
						++tempSpec.name[0];
						tempSpec.name[tempSpec.name[0]] = ':';
			        
						/* Add directory name to beginning of fullPath */
						PrependPString( tempSpec.name, fullPath );
						result = MemError();
					}
				}
			}
		}
	}
	
	if ( (result == noErr) && depth > 0 )
	{
		// add a colon to the beginning of the partial pathname if it consists of more than a file name.
		tempSpec.name[0] = 1;
		tempSpec.name[1] = ':';
		PrependPString( tempSpec.name, fullPath );
	}
	
	if( result == noErr )
	{
		BlockMoveData( fullPath, outPath, fullPath[0] +1 );
		
		result = realResult;  // return realResult in case it was fnfErr
	}
	
	return( result );
}


/* -----------------------------------------------------------------------------
	FSpGetFullPath:
		Return a full (=absolute) pathname for the specified file.
	
	TAKES:
		spec	-	The file for which you need a partial pathname.
	
	GIVES:
		outPath	-	The Str255 passed here is set to the full pathname.
		OSErr	-	Error code if something went wrong, otherwise noErr.
	
	REVISIONS:
		2002-03-01	UK	Documented.
   -------------------------------------------------------------------------- */

OSErr	FSpGetFullPath( const FSSpec *spec,	// target item
							Str255 outPath )	// Path name is returned here.
{
	OSErr		result;
	OSErr		realResult;
	FSSpec		tempSpec;
	CInfoPBRec	pb;
	Str255		fullPath;
	
	fullPath[0] = 0;
	
	/* Make a copy of the input FSSpec that can be modified */
	BlockMoveData( spec, &tempSpec, sizeof(FSSpec) );
	
	if( tempSpec.parID == fsRtParID )
	{
		/* The object is a volume */
		
		/* Add a colon to make it a full pathname */
		++tempSpec.name[0];
		tempSpec.name[tempSpec.name[0]] = ':';
		
		/* We're done */
		BlockMoveData( tempSpec.name, fullPath, tempSpec.name[0] +1 );
		result = MemError();
	}
	else
	{
		/* The object isn't a volume */
		
		/* Is the object a file or a directory? */
		pb.dirInfo.ioNamePtr = tempSpec.name;
		pb.dirInfo.ioVRefNum = tempSpec.vRefNum;
		pb.dirInfo.ioDrDirID = tempSpec.parID;
		pb.dirInfo.ioFDirIndex = 0;
		result = PBGetCatInfoSync(&pb);
		
		// Allow file/directory name at end of path to not exist.
		realResult = result;
		if( (result == noErr) || (result == fnfErr) )
		{
			/* if the object is a directory, append a colon so full pathname ends with colon */
			if ( (result == noErr) && (pb.hFileInfo.ioFlAttrib & ioDirMask) != 0 )
			{
				++tempSpec.name[0];
				tempSpec.name[tempSpec.name[0]] = ':';
			}
		  
			/* Put the object name in first */
			BlockMoveData( tempSpec.name, fullPath, tempSpec.name[0] +1 );
			result = MemError();
			if( result == noErr )
			{
			    /* Get the ancestor directory names */
				pb.dirInfo.ioNamePtr = tempSpec.name;
				pb.dirInfo.ioVRefNum = tempSpec.vRefNum;
				pb.dirInfo.ioDrParID = tempSpec.parID;
				do
				{
					pb.dirInfo.ioFDirIndex = -1;
					pb.dirInfo.ioDrDirID = pb.dirInfo.ioDrParID;
					result = PBGetCatInfoSync(&pb);
					if( result == noErr )
					{
						/* Append colon to directory name */
						++tempSpec.name[0];
						tempSpec.name[tempSpec.name[0]] = ':';
			        
						/* Add directory name to beginning of fullPath */
						PrependPString( tempSpec.name, fullPath );
						result = MemError();
					}
				}
				while( (result == noErr) && (pb.dirInfo.ioDrDirID != fsRtDirID) );  /* loop until we have an error or find the root directory */
			}
		}
	}
		
	if( result == noErr )
	{
		BlockMoveData( fullPath, outPath, fullPath[0] +1 );
		
		result = realResult;  // return realResult in case it was fnfErr
	}
	
	return( result );
}


/* --------------------------------------------------------------------------------
	FSpIsAFolder:
		Determine whether an FSSpec is a folder.
	
	TAKES:
		tempSpec	-	The file system object to check.
		isFolder	-	A boolean variable that will be filled on output.
	
	GIVES:
		isFolder	-	The variable pointed to by this parameter will be set to
						TRUe if tempSpec is a folder, FALSE if it isn't.
	
	REVISIONS:
		2001-12-13	UK		Changed to use memmove().
   ----------------------------------------------------------------------------- */

OSErr	FSpIsAFolder( const FSSpec* tempSpec, Boolean *isFolder )
{
	CInfoPBRec	pb;
	OSErr		result;
	Str255		vName;
	
	memmove( vName, tempSpec->name, tempSpec->name[0] +1 );
	
	pb.dirInfo.ioNamePtr = vName;
	pb.dirInfo.ioVRefNum = tempSpec->vRefNum;
	pb.dirInfo.ioDrDirID = tempSpec->parID;
	pb.dirInfo.ioFDirIndex = 0;
	result = PBGetCatInfoSync(&pb);
	
	// Now check folder bit:
	*isFolder = (pb.hFileInfo.ioFlAttrib & ioDirMask) == ioDirMask;
	
	return result;
}


/* --------------------------------------------------------------------------------
	PrependPString:
		Insert one string before the other, concatenating the two.
		
		E.g. PrependString( "\pfoo", "\pbar" );
		
		sets the second string to be "\pfoobar" (its previous value was "\pbar").
	
	TAKES:
		prependString	-	The string to become the start of the product.
		str				-	The string to become the end of the product.
	
	GIVES:
		str				-	The Str255 pointed to by this is set to the
							concatenation of prependString and str.
	
	REVISIONS:
		2002-03-01	UK		Documented.
   ----------------------------------------------------------------------------- */

void	PrependPString( Str255 prependString, Str255 str )
{
	Str255		tempString;
	short		x,
				y = 1;
	
	BlockMoveData( prependString, tempString, prependString[0] +1 );
	
	x = prependString[0] +1;
	
	while( x < 256 && y <= str[0] )
		tempString[ x++ ] = str[y++];
	
	tempString[0] = x -1;
	
	BlockMoveData( tempString, str, x );
}