/*	***************************************************************************

	PROJECT:	Joker
	
	FILE:		TalkInstructions.h
	
	PURPOSE:	Instructions in a script.
		
	COPYRIGHT:	(C) Copyright 1999 by M. Uli Kusterer, all rights reserved.
				
	REACH ME AT:
				E-MAIL:		witness@weblayout.com
				URL:		http://www.weblayout.com/witness
	
	
	REVISIONS:
		1999-02-14	UK		Created.
				
	************************************************************************ */

#pragma mark [Headers]

/* --------------------------------------------------------------------------------
	Headers:
   ----------------------------------------------------------------------------- */

#include	"TalkInstructions.h"
#include	"HyperTalk.h"
#include	"TalkVarValue.h"
#include	"JokerMain.h"
#include	<AERegistry.h>
#include	<iostream>
#include	<Files.h>
#include	<TextUtils.h>
#include	<Sound.h>
#include	"GetPartialPath.h"

#if COMPILING_SAL
	#include	"XBaseApplication.h"
#endif


// Compatibility glue for AE opacity:
#if !ACCESSOR_CALLS_ARE_FUNCTIONS

OSErr	AEGetDescData( const AEDesc *theAEDesc,
						void *dataPtr,
						Size maximumSize )
{
	Size		vActualSize;
	
	vActualSize = GetHandleSize( theAEDesc->dataHandle );
	if( vActualSize > maximumSize )
		vActualSize = maximumSize;
	
	BlockMoveData( *theAEDesc->dataHandle, dataPtr, vActualSize );
	
	return MemError();
}

Size	AEGetDescDataSize( const AEDesc *theAEDesc )
{
	return GetHandleSize( theAEDesc->dataHandle );
}

#endif

void			MacRunOneFile( char* vFileName );



#pragma mark -
#pragma mark [Instruction Procs]

/* --------------------------------------------------------------------------------
	All instruction procedures take two parameters, an "HI" parameter that contains
	the instruction's parameters and destination for the result along with other
	info like a pointer to the object currently running, and an "s" parameter
	that points to the stack with the variables for this instruction.
   ----------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------------
	HyperReplyInstruction:
		This implements the "reply" command that can be used to return information
		from an appleEvent message. This is inherently Mac-specific, but maybe
		some hosts want to support this as a compatibility layer.
	
	HI's FIELDS:
		0		-	TRUE if this is "reply error", FALSE if just "reply".
		1		-	Expression to return.
		2		-	TRUE if "with" follows expression, FALSE if omitted.
		3		-	The AppleEvent keyword or an empty string if none specified.
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperReplyInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vIsReplyError,
						vString,
						vIsWithKeyword,
						vAEKeyword;
		
	hi.mParam[0].GetValue( vIsReplyError, s, VALUE_TYPE_BOOL );
	hi.mParam[1].GetValue( vString, s, VALUE_TYPE_TEXT );
	hi.mParam[2].GetValue( vIsWithKeyword, s, VALUE_TYPE_BOOL );
	hi.mParam[3].GetValue( vAEKeyword, s, VALUE_TYPE_TEXT );
	
	try
	{
		// First, check whether this is "reply error":
		if( vIsReplyError.boolType )	// This simply means we use reply keyword "errs":
		{
			vIsWithKeyword.boolType = true;
			vAEKeyword.textType->CopyFromString( "errs" );
		}
		
		// Now actually stash the data in the AppleEvent reply:
		AppleEventPtr		vReply;
		
		vReply = s.GetCurrMacAEReply();
		if( vReply != NULL )
		{
			AEKeyword		vKeyword;
			
			switch( vAEKeyword.textType->GetLength() )
			{
				case 4:		// Proper type code.
					vAEKeyword.textType->CopyToText( (char*) &vKeyword );
					break;
				
				case 0:		// None passed, use direct object.
					vKeyword = keyDirectObject;
					break;
				
				default:
					throw std::runtime_error( "An AppleEvent keyword must be exactly 4 characters long." );
			}
			(OSErr) AEPutParamPtr( vReply, vKeyword, typeChar,
									vString.textType->GetTextPtr(), 
									vString.textType->GetLength() );
		}
		
		delete vString.textType;
		delete vAEKeyword.textType;
	}
	catch( std::exception& err )
	{
		delete vString.textType;
		delete vAEKeyword.textType;
		throw;
	}
}


/* --------------------------------------------------------------------------------
	HyperRequestInstruction:
		This implements the "request" command that can be used to obtain
		information from an appleEvent message. This is inherently Mac-specific,
		but maybe some ports want to support this partially as a compatibility
		layer for CGIs etc.
	
	HI's FIELDS:
		0		-	What to request, i.e. "data", "sender", "id", "class" etc.
		1		-	TRUE if "with" follows expression, FALSE if omitted.
		2		-	The AppleEvent keyword or an empty string if none specified.
		Result	-	The data requested.
   ----------------------------------------------------------------------------- */

void	HyperRequestInstruction( struct HyperInstruction& hi, ValueStack& s,
									TalkCallRecord& vCallRec )
{
	ValueStorage		vString,
						vIsWithKeyword,
						vAEKeyword;
	TalkValue*			vValue;
		
	hi.mParam[0].GetValue( vString, s, VALUE_TYPE_TEXT );
	hi.mParam[1].GetValue( vIsWithKeyword, s, VALUE_TYPE_BOOL );
	hi.mParam[2].GetValue( vAEKeyword, s, VALUE_TYPE_TEXT );
	
	try
	{
		// Now actually stash the data in the AppleEvent reply:
		AppleEventPtr		vEvent;
		OSErr				vError;
		
		vEvent = s.GetCurrMacAEvt();
		if( vEvent != NULL )
		{
			AEKeyword		vKeyword;
			AEDesc			vTheParamDesc;
			Size			vDataSize;
			
			if( (*vString.textType) == "data" )
			{
				switch( vAEKeyword.textType->GetLength() )
				{
					case 4:		// Proper type code.
						vAEKeyword.textType->CopyToText( (char*) &vKeyword );
						break;
					
					case 0:		// None passed, use direct object.
						vKeyword = keyDirectObject;
						break;
					
					default:
						throw std::runtime_error( "An AppleEvent keyword must be exactly 4 characters long." );
				}
				vError = AEGetParamDesc( vEvent, vKeyword, typeChar, &vTheParamDesc );
				if( vError == errAECoercionFail )	// Couldn't get as text?
					vError = AEGetParamDesc( vEvent, vKeyword, typeWildCard, &vTheParamDesc );	// Get native type and hope our conversion code below handles it.
				if( vError != noErr )
					throw std::runtime_error( "Couldn't get descriptor for Apple Event parameter." );
				switch( vTheParamDesc.descriptorType )
				{
					case typeAEList:
					{
						long		vCount;
						TextMunger	vTempText;
						AEDesc		vCurrEntry;
						
						// Clear munger used for result value:
						vString.textType->SetLength(0);
						vString.textType->SetOffset(0);
						
						// Loop over items in list:
						vError = AECountItems( &vTheParamDesc, &vCount );
						for( long x = 1; x <= vCount; x++ )
						{
							// Get list item:
							vError = AEGetNthDesc( &vTheParamDesc, x, typeChar,
													&vKeyword, &vCurrEntry );
							if( vError != noErr )
								throw std::runtime_error( "Couldn't convert apple event parameter's list item to text format." );
							vDataSize = AEGetDescDataSize( &vCurrEntry );
							if( vDataSize > 0 )
							{
								// Copy list item into a text munger:
								vTempText.SetLength( vDataSize );
								vError = AEGetDescData( &vCurrEntry, vTempText.GetTextPtr(),
														vDataSize );
								if( vError != noErr )
									throw std::runtime_error( "Couldn't get data from Apple Event parameter's list entry." );
							}
							else
								vTempText.SetLength( 0 );
							AEDisposeDesc( &vCurrEntry );
							
							// Append text munger to result value:
							vString.textType->InsertMunger( vTempText );
							vString.textType->Insert<char>( '\n' );
						}
						
						// If list wasn't empty:
						if( vCount >= 1 )
							vString.textType->SetLength(vString.textType->GetLength() -1);	// Get rid of trailing return.
						break;
					}
					
					case typeChar:
						vDataSize = AEGetDescDataSize( &vTheParamDesc );
						vString.textType->SetLength( vDataSize );
						vError = AEGetDescData( &vTheParamDesc, vString.textType->GetTextPtr(),
												vDataSize );
						if( vError != noErr )
							throw std::runtime_error( "Couldn't get data from Apple Event parameter." );
						break;
					
					default:
						AEDisposeDesc( &vTheParamDesc );
						throw std::runtime_error( "Apple Event parameter can't be converted to text form." );
				}
				
				AEDisposeDesc( &vTheParamDesc );
			}
			else if( (*vString.textType) == "id" )
			{
				DescType		returnedType;
				long			returnedSize;
				
				vString.textType->SetLength( sizeof(AEEventID) );
				vError = AEGetAttributePtr( vEvent, keyEventIDAttr,
							 				typeType, &returnedType,
							 				vString.textType->GetTextPtr(),
							 				sizeof(AEEventID), &returnedSize );
			}
			else if( (*vString.textType) == "class" )
			{
				DescType		returnedType;
				long			returnedSize;
				
				vString.textType->SetLength( sizeof(AEEventClass) );
				vError = AEGetAttributePtr( vEvent, keyEventClassAttr,
							 				typeType, &returnedType,
							 				vString.textType->GetTextPtr(),
							 				sizeof(AEEventClass), &returnedSize );
			}
			else
				throw std::runtime_error( "I don't know how to request that from an AppleEvent." );
			
			hi.mResult.GetValue( vValue, s );
			vValue->SetValue( vString, VALUE_TYPE_TEXT, true );
		}
		
		delete vString.textType;
		delete vAEKeyword.textType;
	}
	catch( std::exception& err )
	{
		delete vString.textType;
		delete vAEKeyword.textType;
		throw;
	}
}


/* --------------------------------------------------------------------------------
	HyperBeepInstruction:
		Output an error sound.
	
	HI's FIELDS:
		0		-	Number of times to beep.
		Result	-	*unused*
   ----------------------------------------------------------------------------- */

void	HyperBeepInstruction( struct HyperInstruction& hi, ValueStack& s,
							TalkCallRecord& vCallRec )
{
	ValueStorage	vCount;
	
	hi.mParam[0].GetValue( vCount, s, VALUE_TYPE_LONG );
	
	while( vCount.longType > 0 )
	{
		SysBeep(10);	// 10 ticks of error sound.
		vCount.longType--;
	}
}